/*
 * Single cell boost LED driver program for bike light with side and front visibility.
 * Steps up supply voltage of 2.7V to 4.2V, to about 10V.
 * The code has battery sleep mode, patterns,
 * 
 * Pressing the button for short duration increments the mode.
 * Pressing the button for medium duration turns on or off the bike light.
 * Pressing the button for long duration while the light is on toggles the mode type.
 * Pressing the button for long duration while the light is off toggles the battery chemistry.
 * 
 * Short flashes when the light turns on indicates that LiFePO4 is selected.
 * Long flashes when the light turns on indicates that NiMH is selected. It also works for 3 cell NiMH.
 * 
 * Default modes
 * Fade (48h)
 * Triple pulse followed by a pause (35h)
 * Day flashes (17h)
 * 
 * Other modes
 * Low battery
 * Solid low (10 hour)
 * Solid high (2 hour)
 * 
 * Solid, fade, and short pulses were used to avoid being confused with turn signals.
 * Estimated battery life for LiFePO4 with a capacity of 1800mAh
 * 
 * Min supply voltage is 2.4V for 8 MHz clock frequency.
 * 
 */
#include <EEPROM.h>
#include <avr/sleep.h>
 
// Utility macros
#define adc_disable() (ADCSRA &= ~(1<<ADEN)) // disable ADC
#define adc_enable()  (ADCSRA |=  (1<<ADEN)) // re-enable ADC

// Pins
const byte batteryMonitorPin = A2;
const byte driverFbPin = A3;
const byte driverPwmPin = 1;
const byte buttonPin = 2;

boolean buttonState;
const int prescalerCompensate = 64;
const unsigned int debounceDelay = 250 * prescalerCompensate; // Used to compensate for the change in the prescaler.
byte delayCounts = 0;

// Boost LED driver.
int onTime = 0;
const int period = 255; // Determines the frequency and resolution.
const int maxOnTime = 180; // You can set it to limit the maximum voltage and supply current.
int feedbackReading;
int desiredFeedback; // Depends on Rsense and drive current.
int brightnessHigh, brightnessLow;
/*
   Rsense of 2 ohm
   desiredFeedback=Idrive*2*1023/1.1
   651 for a drive current of 350mA
   Estimated 1 hour batt life
*/

// Bike light patterns.
byte lowBatteryTypeNumber = 1;
byte lowBatteryModeNumber = 3;
byte mode = lowBatteryModeNumber; // Start with low battery mode.
byte modeType = lowBatteryTypeNumber;
unsigned long elapsedMillis;
unsigned long prevMillis;
const int stepTime = 10; // Actual time in milliseconds is stepTime/68. Fade frequency is approximately 2.5 Hz.
byte stepLevel;
const int stepTop = 39;
const byte fadeLevels[40] = {0, 0, 1, 1, 3, 4, 6, 8, 10, 13, 16, 20, 23, 27, 32, 37, 42, 47, 53, 59, 65, 59, 53, 47, 42, 37, 32, 27, 23, 20, 16, 13, 10, 8, 6, 4, 3, 1, 1, 0};
/*
  ADC values for different battery chemistries.
  AREF=1.1V, resistors of 1M and 200k.
   1 cell LiFePO4: Low=2.8V  Dead:2.7; Readings:0.466V,0.451V; ADC:434,419
   1 cell Li-ion: Low=3.4V; Dead:3.3V; Readings:0.570V,0.555V; ADC:530,516; Can be used for 3 cell NiMH as well.
*/
// resistance values have options for two cell
const int lifepo4LowADC = 434;
const int lifepo4deadADC = 419;
const int liionLowADC = 530;
const int liionDeadADC = 516;
int lowBattADC;
int deadBattADC;
boolean liionSelected = EEPROM.read(0); // true for 3.7V Li-ion batteries, false for 3.2V LiFePO4 batteries
int batteryReading;

void setup() {

  pinMode(batteryMonitorPin, INPUT);
  pinMode(driverFbPin, INPUT);
  pinMode(driverPwmPin, OUTPUT);
  pinMode(buttonPin, INPUT_PULLUP);

  analogReference(INTERNAL);

  enableBoostConverter();
  sleepNow();
}

void loop() {
  readButton();
  LEDDriver();

  switch (mode) {
    // Default modes.
    case 0: // Fade
      fade(); // 48 hour
      batteryReading = analogRead(batteryMonitorPin);
      break;
    case 1: // Triple pulses followed by a pause.
      triplePulse(); // 35 hour
      break;
    case 2: // Day pulses
      dayPulse(); // 17 hour
      break;
      
    // Other modes
    case 3: // Low battery
      desiredFeedback = 1;
      batteryReading = analogRead(batteryMonitorPin);
      break;
    case 4: // Solid low
      desiredFeedback = 100; // 10 hour
      batteryReading = analogRead(batteryMonitorPin);
      break;
    case 5: // Solid high
      //desiredFeedback = 651; // 1 hour. If there's no problem with battery reading, you can use this.
      desiredFeedback = 325; // 2 hour
      batteryReading = analogRead(batteryMonitorPin);
      break;
  }

  if (batteryReading < lowBattADC) { // If battery is low, go to low battery mode.
    mode = lowBatteryModeNumber;
    modeType = lowBatteryTypeNumber;
  }
  if (batteryReading < deadBattADC) { // If battery is considered dead, go to sleep.
    sleepNow();
  }
}

void enableBoostConverter() {
  TCCR0A = _BV(WGM01) | _BV(WGM00) | _BV(COM0B1);
  TCCR0B = _BV(WGM02) | _BV(CS00);
  OCR0A = period;
  OCR0B = 5;
}

