/*
gps2.ino

30/12/2101

D.R.Patterson

gps logger

For MTK3339 chipset on the ADAFRUIT ultimate GPS logger
LCD DFROBOT works

LCD and SD support for logging (Logging can be toggled)

GPS board:
  Lines to 7,8 and 10 cut before MEGA pins
  TX connexted to Mega RX1
  RX connected to Mega TX1
  ccs connected to Mega pin 32 (Could be any other not used by the lcd)
Lcd display for
  Latitude and Longitude position
  UK National Gid reference
  Altitude  
  Speed
  Bearing
  Distance travelled from origin (origin can be reset)
  Range from origin
  Time
  Number of satellites
  Horizontal dilution of position (HDOP) 
LCD values in metric or imperial

For the lcd display a speed of <= cutoff (0.65) mph is shown as 0
The number of satellites is shown as a vertical bar in lower left corner.
The largest number that can be displayed is  8, after that the bar is full.

HDOP is indicated in the lower centre moving bar.
HDOP indicates the spread of observed satellites-
Increased spread increases accuracy and the HDOP number decreases.
HDOP values of 1 are very good! 
A HDOP precision of < 0.5 is shown by a full bar.
A HDOP of 0.5 to 1.5 (1) is shown with 7 bars.
A HDOP of 6 is shown with 1 bar.
A HDOP of >6 displays no bars.

Buttons:
  a) Select changes lcd 1st line between
  altitude/position (Longitude and Latitude)
  position (Long & Lat)
  position in ordnance survey x,y co-ordinates
  distance covered from origin
  range to origin
  altitude
  time
  b) Left toggles metric/ imperial
  c) Right toggles logging- an L is shown in the lower right when logging
     An ! mark in this position indicates no sd card 
  d) Up toggles back display (Useful power saver)
  e) Down resets distance to zero and sets the current position
     as the origin for range (The crow flies distance  to origin)
To activate a button- hold down until led stops flashing.
Then release.

LED Indicator:
1 second interval Flashing red -  No Fix
1 second interval Flashing green- Fix
Four short green flashes - A button has been pressed
1 red flash repeated 5 times - Error 1: SD Card init failed
2 red flashes repeated 5 times - Error 2: SD Card full speed init failed
3 red flashes repeated 5 times - Error 3: Can not create the log file
4 red flashes repeated 5 times - Error 4: 100 files in log folder and do not delete

The cumulative distance estimate will have cumulative rounding errors
from multiple one second interval positions.
The range will be more accurate as it is determined by only two locations.

Note that the precision of the Ordnance survey value is limited-
The arduino float values hold 6-7 significant decimal places.
The conversion formula involves many calculations and the rounding errors
have cumulative effects.
Ultimately precision is restricted by the returned sin, cos and tan values
The easting value appears to be accurate to +- 0.1
The northing to +- 0.02

By comparison the Longitude/latitude values appear accurate to +- 0.00001

Connecting a ufl to SMA lead does not affect the performance.
Only an active arial (1575.42MHz) would be recognised by the gPS chip.

Modified from:
  Ladyada's logger
  with Bill Greiman's SdFat library
using MTK3339 chipset http://www.adafruit.com/products/1272

NB Adafruit sd.h library modified to use calls to SDADA.h
Thus avoiding need to remove original SD library because of
name conflicts.

*/

#include <SPI.h>
#include <Adafruit_GPS.h>
#include <SoftwareSerial.h>
// If using the default adafruit sd library use #include <SD.h>
#include <SDADA.h>
#include <avr/sleep.h>

#include <LiquidCrystal.h>
LiquidCrystal lcd(8, 9, 4, 5, 6, 7);
const byte lcdport = 0;
const byte backlight = 10;
boolean lcdDisplay = true;
boolean dswitch = false;

// ************************** config ************************************************

// set to true to only log to SD when GPS has a fix, for debugging, keep it false
#define LOG_FIXONLY true
// Set GPSECHO to 'false' to turn off echoing the GPS data to the Serial console
#define GPSECHO  false
const boolean debug = false; // stop/start all serial output
#define chipSelect 32 // The pin used to select the gps
boolean sain = true;  // Set to true if using the sainsmart lcd shield
                      // false for another eg DFROBOT
boolean pwm = true; // control backlight with pwm
byte pwmval = 127; // pwm lcd on setting

// ************************** end config ********************************************

#define btnRIGHT  0      //values to use with the button switch function
#define btnUP     1
#define btnDOWN   2
#define btnLEFT   3
#define btnSELECT 4
#define btnNONE   5
int lcdbut[6];

const byte degreeSymbol = B11011111;
const byte leftsymbol = 127;
const byte rightsymbol = 126;
// setup patterns to represent number of satellites / HDOP
byte sats [8][8] = {
0,0,0,0,0,0,0,14,
0,0,0,0,0,0,14,14,
0,0,0,0,0,14,14,14,
0,0,0,0,14,14,14,14,
0,0,0,14,14,14,14,14,
0,0,14,14,14,14,14,14,
0,14,14,14,14,14,14,14,
14,14,14,14,14,14,14,14,
};

byte firstline =1;
boolean metric = false;
// Defines for clearing register bits
#ifndef mycbi
#define mycbi(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))
#endif
// Defines for setting register bits
#ifndef mysbi
#define mysbi(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))
#endif

HardwareSerial mySerial = Serial1;
Adafruit_GPS GPS(&mySerial);

// Set the pins used
#define ledRed 2
#define ledGreen 3

Sd2Card card;
File logfile;

char buf[20]; // used for type conversions during display
String myfile;
boolean hascard = false;
boolean logging = false;
// this keeps track of whether we're using the interrupt
boolean usingInterrupt;

String oldtime;
unsigned long ctime;
float startlatitude, startlongitude, oldlatitude , oldlongitude, travelled;

const float cutoff = 0.65;  // This is the speed in mph above which distance is logged
                            // by travelled and bearing is displayed
const float kmcutoff = cutoff * 1.609344;
byte distcount = 0;
const byte deflimit = 4;
byte dlimit = deflimit; // used to control number of readings before new distance is calculated
boolean moved = false;
const float fconvert = 3.2808399; // metres to feet
const float mconvert = 0.0006213712; // metres to miles

void setup() {
lcd.begin(16, 2);
  if (sain){
  // button levels for sain
  lcdbut[0] = 1000;// No button
  lcdbut[1] = 100; // Right
  lcdbut[2] = 200; // Up
  lcdbut[3] = 400; // Down
  lcdbut[4] = 600; // Left
  lcdbut[5] = 900; // Select
  }else{
  // button levels for DFROBOT
  lcdbut[0]=1000;
  lcdbut[1] = 60;
  lcdbut[2] = 150;
  lcdbut[3] = 300;
  lcdbut[4] = 500;
  lcdbut[5] = 800;
  }
  
  if(debug) {
  Serial.begin(115200);
  Serial.println("\r\nUltimate GPSlogger Shield");
  }
pinMode(ledRed, OUTPUT);
pinMode(ledGreen, OUTPUT);
  if (pwm){
  analogWrite(backlight, pwmval);
  }else{
  pinMode(backlight,OUTPUT);
  digitalWrite(backlight,HIGH);
  }
for(byte i=0; i<8; i++)lcd.createChar(i,sats[i]);

lcd.clear();
lcd.print("Sel:Pos/Alt/Time");
lcd.setCursor(0,1);lcd.write(leftsymbol);
lcd.print("Units  ");
lcd.write(rightsymbol);lcd.print("LogFile");
  if(!sain){
  // just reading lcd port
  // this does not work with the sain
  cli();
  ADCSRA = 0;
  ADCSRB = 0;
  ADMUX |= lcdport;
  mysbi(ADCSRA,ADEN); // //enable only the lcdport ADC
  sei();
  int temp = analogRead(lcdport);
  }
// make sure that the default chip select pin is set to
// output, even if you don't use it:
pinMode(SS,OUTPUT); // default mega select pin
pinMode(chipSelect,OUTPUT);
digitalWrite(chipSelect,LOW);
// see if the card is present and can be initialized:
  if (!SD.begin(chipSelect, 11, 12, 13)) {
    if(debug) Serial.println(F("Card init. failed!"));
  error(1);
  } else{
    if (!card.init(SPI_FULL_SPEED, chipSelect, 11, 12, 13)) {
    error(2);
      if(debug) {
      Serial.println(F("initialization failed. Things to check:"));
      Serial.println(F("* is a card is inserted?"));
      Serial.println(F("* Is your wiring correct?"));
      Serial.println(F("* did you change the chipSelect pin to match your shield or module?"));
      }
    }else{
    hascard = true;
      if(debug) Serial.println(F("Card init. succeeded"));
    }
  }
  if(hascard){
  // Create gps sub-directory
    if (SD.exists("/gps") == false) SD.mkdir("/gps");
  }

// connect to the GPS at the desired rate
GPS.begin(9600);
// Set the update rate
// 1Hz works well and leaves time to perform logging
GPS.sendCommand(PMTK_SET_NMEA_UPDATE_1HZ);   // alternatively 0.1Hz, 5Hz and 10Hz update rate

// RMC (recommended minimum) : GGA (fix data) including altitude
GPS.sendCommand(PMTK_SET_NMEA_OUTPUT_RMCGGA);

// uncomment this line to turn on only the "minimum recommended" data
//GPS.sendCommand(PMTK_SET_NMEA_OUTPUT_RMCONLY);

// Turn off updates on antenna status, if the firmware permits it
GPS.sendCommand(PGCMD_NOANTENNA);

// 1ms Interrupt on timer0 to facilitate gps data echo- if GPSECHO = true
useInterrupt(true); // read input by interrupt if true
    if(debug) Serial.println("Ready!");
ctime = millis(); // display options with eye catching green led and and moving bar 
digitalWrite(ledGreen,HIGH);
  do{
    for(byte i=0; i<8;i++){
    lcd.setCursor(6,1);
    lcd.write(i);lcd.write(i);
    delay(250);
    }
  } while ( (millis() - ctime) < 5000 );
digitalWrite(ledGreen,LOW);
lcd.clear();
oldlatitude = startlatitude = -1000;
travelled = 0;
ctime = millis();
}

// **************************** START MAIN LOOP *****************************************
void loop() {
String mytime, myyear, lcdyear, theyear, logtime;
float latitude , longitude, altitudem, altitudef, speedknots, speedmph, speedkph, heading;
float myseconds, mymillis, geoid;
int hdop;
boolean fixed = false; // status of fix
unsigned long timenow;
byte mylen;
int satellites;

int thisbutton = read_LCD_buttons();
  if (thisbutton != btnNONE){
    for (byte i=1; i < 5; i++){
    digitalWrite(ledGreen, HIGH);
    delay(50);
    digitalWrite(ledGreen, LOW);
    delay(50);
    }
  }
  if (thisbutton == btnSELECT){
  firstline++;
    if (firstline == 8) firstline=1;
    if(firstline == 1){
    ctime = millis();
    dswitch=false;
    }
  lcd.setCursor(0,0);lcd.print("                ");
  }
  if(thisbutton== btnLEFT) metric = !metric;
  if(thisbutton== btnRIGHT){
    if (logging) closefile(); else openfile();
  }
  if(thisbutton== btnUP){
  lcdDisplay= !lcdDisplay;
  if (pwm){
    if(lcdDisplay) analogWrite(backlight,pwmval); else analogWrite(backlight,0);
  } else digitalWrite(backlight,lcdDisplay);
  }
  if(thisbutton== btnDOWN){
  travelled = 0;
  distcount = 0;
  startlatitude = -1000; // reset start position for range - the crow flies distance
  moved = false;
  dlimit = deflimit; 
  lcd.setCursor(0,0);
  lcd.print(" Distance now 0 ");
  delay(1000);
  }
  if (! usingInterrupt) {
    // read data from the GPS in the 'main loop'
    char c = GPS.read();
    // if you want to debug, this is a good time to do it!
    if (GPSECHO) if (c && debug) Serial.print(c);
  }
  
 // ************* CHECK FOR NEW GPS DATA ************* 
  // if a sentence is received, we can check the checksum, parse it...
  if (GPS.newNMEAreceived()) {
    // a tricky thing here is if we print the NMEA sentence, or data
    // we end up not listening and catching other sentences! 
    // so be very wary if using OUTPUT_ALLDATA and trying to print out data
    // Don't call lastNMEA more than once between parse calls!  Calling lastNMEA 
    // will clear the received flag and can cause very subtle race conditions if
    // new data comes in before parse is called again.
    
    char *stringptr = GPS.lastNMEA();
    if (!GPS.parse(stringptr)) return; // go around if false
    // this also sets the newNMEAreceived() flag to false

  // log and lcd
    if(debug) Serial.println("Parsed");

    if(GPS.fix){
    digitalWrite(ledGreen, HIGH);
    fixed = true; // remember state in case it changes mid process
    mymillis = GPS.milliseconds;
    myseconds = GPS.seconds;
    mytime = String(GPS.hour) + ":" + String(GPS.minute)+":";
    logtime = mytime;
    logtime += String(int(myseconds + mymillis / 1000 + 0.1 ) ); // round up if within 0.1 S
    mytime += String(myseconds + mymillis / 1000); // full recalled time
    myyear = String(GPS.day) + "/" + String(GPS.month) + "/";
    lcdyear = myyear;
    theyear = String(GPS.year);
    myyear += theyear;
    mylen = theyear.length();
    lcdyear += theyear.substring(mylen-1,mylen); // last digit only to fit in
    latitude = GPS.latitudeDegrees;
    longitude = GPS.longitudeDegrees;
    altitudem = GPS.altitude;
    altitudef = altitudem * fconvert;
    speedknots = GPS.speed;
    speedmph = speedknots * 1.15077945;
    speedkph = speedknots * 1.85200;
    heading = GPS.angle;
    satellites = GPS.satellites;
    geoid = GPS.geoidheight;
    hdop = 7-int(GPS.HDOP + 0.5);
      if (startlatitude == -1000){
      startlatitude = latitude;
      startlongitude = longitude;
      }
      if (oldlatitude == -1000){
      oldlatitude = latitude;
      oldlongitude= longitude;
      }else{
        if (logtime != oldtime){ // change of 1 second required
          if (speedmph > cutoff ) moved = true;
          if (speedmph > 10) dlimit = 3; // 10mph = 4.47 m/s So reduce interval as speed increases 
          if (speedmph > 20) dlimit = 2; // at 20 mph travels 8.9408 m per second
          if (speedmph > 30) dlimit = 1; // // 30mph is 13.41m/s
        distcount ++;
          if (distcount >= dlimit){ // update distance travelled if it moved
          distcount = 0;        
            if (moved) travelled += distance_between(oldlatitude, oldlongitude, latitude, longitude);
          moved = false;
          dlimit = deflimit;// default counter for distance measurement:
                            // 3mph = 1.34 m/s. @4 seconds cover 5.36 m
                            // no point in measuring every second because the gps position resolution
                            // needs at least 5m to discern a distance!
          oldlatitude = latitude;
          oldlongitude= longitude;
          }
        }
      }

      if (hdop < 0) hdop = 0;
    lcd.setCursor(0,1);
    int satshow = satellites;
      if (satshow > 8) satshow = 8;
      if (satshow == 0) lcd.print(" "); else lcd.write(satshow - 1);
    float displayspeed = 0;
    byte L =5;
      if (metric) {
        if (speedkph > kmcutoff) displayspeed = speedkph;
        if (displayspeed < 100 ) L=4;
      lcd.print(dtostrf(displayspeed,L,1,buf));
      lcd.print(" kph ");
      }else{
      L = 5;
        if (speedmph > cutoff) displayspeed = speedmph;
        if (displayspeed < 100 ) L=4;
      lcd.print(dtostrf(displayspeed,L,1,buf));
      lcd.print(" mph ");
      }
    lcd.setCursor(10,1);  
      if (hdop == 0) lcd.print(" "); else lcd.write(hdop); // indicate horizontal depletion of precision 
    lcd.setCursor(11,1);
      if (displayspeed == 0){
      lcd.print("N/A");
      }else{     
      lcd.print(int(heading + 0.5));
      lcd.print("  ");
      }
    lcd.setCursor(14,1);lcd.write(degreeSymbol);
    // process first lcd line
    lcd.setCursor(0,0);
    lcd.print("                ");
    lcd.setCursor(0,0);
 
      if (firstline == 1){ // alternate altitude and position
        if(dswitch) showalt(altitudem,altitudef); else showlonglat(longitude,latitude);
        timenow = millis();
        if(timenow - ctime > 2999) {// switch after 3s
        dswitch = !dswitch;
        ctime = timenow;
        }
      }

      if (firstline == 2) showlonglat(longitude, latitude); // just long/lat position
        
      if (firstline == 3) ordnance(latitude, longitude); // position in Ordnance survey format
      
      if (firstline == 4) showalt(altitudem, altitudef); // altitude

      if (firstline == 5) {
      byte dp =3; // adjust dp with magnitude of distance
      lcd.print("Dist:");
      String units = " m";
        if(metric) {
        float ltravelled = travelled;
          if (ltravelled > 1000){
          ltravelled = travelled / 1000;
          units = "Km";
          } else dp = 1; // one dp if in metres
           if (ltravelled >= 10000) dp=2;
           if (ltravelled >= 100000) dp=1;
        lcd.print(ltravelled,dp);lcd.print(units);
        }else{
        float miles = travelled * mconvert;
          if (miles >= 100) dp=1;
        lcd.print(miles, dp);lcd.print(" M");
        }
      }

      if (firstline == 6) {
      byte dp =3; // adjust dp with magnitude of distance
      lcd.print("Rnge:");
      String units = " m";
      float range = distance_between(startlatitude, startlongitude, latitude, longitude);

        if (metric){
          if (range > 1000){
          range = range / 1000;
          units = "Km";
          }else dp = 1;
          if (range >= 10000) dp = 2;
          if (range >= 100000) dp = 1;
        lcd.print(range,dp);lcd.print(units);
        }else{
        range = range *mconvert;
          if (range >= 100) dp = 1;
        lcd.print(range, dp);lcd.print(" M");
        }
      }
      
      if (firstline == 7){ // show the time and date
      lcd.print(logtime);lcd.setCursor(9,0);
        if (myyear.length() < 8) lcd.print(myyear); else lcd.print(lcdyear);
      }
      
    }else{
    digitalWrite(ledRed, HIGH);
    lcd.clear();
    lcd.setCursor(6,0);
    lcd.print("No Fix");
      if(debug) Serial.println("No Fix");
    }
  // indicate if logging  
  lcd.setCursor(15,1);
    if (!hascard){
    lcd.print("!");
    } else{
      if(logging) lcd.print("L"); else lcd.print(" ");
    }
  // logging
    if (logtime == oldtime) goto missme; // Only log if the recorded time has changed
  oldtime = logtime;
    if (! logging ) goto missme;  
    if (LOG_FIXONLY && (fixed == false)) goto missme;

    if(debug) Serial.println("Log");
  //logfile.print(pad(mytime,9));
  logfile.print(pad(mytime,13)); // includes milli seconds
  logfile.print(pad(myyear,9));
  logfile.print(dtostrf(longitude,10,5,buf));
  logfile.print(dtostrf(latitude,10,5,buf));
  logfile.print(dtostrf(altitudem,9,1,buf)); logfile.print(" m");
  logfile.print(dtostrf(geoid,9,1,buf)); logfile.print(" m");
  logfile.print(dtostrf(speedmph,7,2,buf)); logfile.print(" mph");
  logfile.print(dtostrf(heading,6,1,buf)); logfile.println(" deg");
  logfile.flush();

  missme:
    if(debug){Serial.println();Serial.flush();}
  delay(30);
    if(fixed) digitalWrite(ledGreen, LOW); else digitalWrite(ledRed, LOW);
  }
}
// ****************************** END MAIN LOOP *****************************************

void showalt(float altitudem, float altitudef){
lcd.print("Alt ");
  if (metric){
  lcd.print(altitudem,1);lcd.print(" m");
  }else{
  lcd.print(altitudef,1);lcd.print(" ft");
  }
}

void showlonglat(float longitude, float latitude){
  if (longitude <=0) {
  lcd.print(-longitude,4);
  lcd.print("W");
  }else{
  lcd.print(longitude,4);
  lcd.print("E");
  }
lcd.setCursor(8,0);
  if (latitude <=0){
  lcd.print(-latitude,4);lcd.print("S");
  }else{
  lcd.print(latitude,4); lcd.print("N");
  }
}

// blink out an error code
void error(uint8_t errno) {
delay(1000);
byte i;
  for (byte j=0; j<5; j++){
    for (i=0; i<errno; i++) {
    digitalWrite(ledRed, HIGH);
    delay(100);
    digitalWrite(ledRed, LOW);
    delay(100);
    }
  delay(500);
  }
delay(500);
}

void ordnance(float phi, float lamda){
/*
cf Ordnance survey: A guide to coordinate systems in Great Britain Page 40
http://www.ordnancesurvey.co.uk/docs/support/guide-coordinate-systems-great-britain.pdf
*/
const float piconvert = 174532925.199433; // 1.74532925199433E-02 multiplied by 1E10 
phi =  phi * piconvert;
lamda = lamda * piconvert; 
const float lamda0 = -349065850.398866; // -2 West in radians -3.49065850398866E-02 multiplied by 1E10
const float phi0 = 8552113334.77221; // 49 N in radians 0.855211333477221 multiplied by 1E10
float ldif = lamda - lamda0;
ldif = ldif / 1.0E10;
float phiminus = phi - phi0;
phiminus = phiminus / 1.0E10;
float phiplus = phi + phi0;
phiplus = phiplus / 1.0E10;
phi = phi / 1.0E10;
lamda = lamda / 1.0E10;
const float E0 = 400000.0;
const float N0 = -100000.0;
const float F0 = 0.9996013; // 0.9996012717;

const float e2 = 0.006670540; // excentricity constant 6.67054007414942E-03
const float aF0 = 6375020.4809890; // 6375020.48098897
const float bF0 = 6353722.4904879; // 6353722.49048791
float base = 1.0 - (e2 * sin(phi) * sin(phi) );
float vp = sqrt(base);
float v = aF0 / vp;
base = 1.0 - (e2 * sin(phi)*sin(phi));
float p = (v * (1.0 - e2)) / base;
float n2 = (v / p) - 1.0;

// Compute meridional arc M
// const float n = 0.001673220; // 1.67322025032509E-03
// Use externally calculated constants for greater precission
const float f1 = 1.001676726; // 1.0 + n + 1.25 * np2 + 1.25 * np3
const float f2 = 0.005028072; // 3.0 * n + 3.0 * np2 + 21.0 * np3 / 8.0
const float f3 = 0.000002336959; // np2 * 15.0 / 8.0 + np3 * 15.0 / 8.0
const float f4 = 0.000000006831501; // np3 * 35.0 / 24.0

float mp1 = f1 * phiminus;
float mp2 = f2 * sin(phiminus) * cos(phiplus);
float mp3 = f3 * sin(2.0 * phiminus) * cos(2.0 * phiplus);
float mp4 = f4 * sin(3.0 * phiminus) * cos(3.0 * phiplus);
float M = bF0 * (mp1 - mp2 + mp3 - mp4);
float I = M + N0;
float sinphi = sin(phi);
float cosphi = cos(phi);
float tanphi = tan(phi);
float II = v * sinphi * cosphi / 2.0;
float III = (v * sinphi * pow(cosphi,3) / 24.0) * (5.0 - tanphi * tanphi + 9.0 * n2);
float IIIA = (v * sinphi * pow(cosphi, 5) / 720.0) * (61.0- 58.0 * tanphi * tanphi + pow(tanphi, 4) );
float IV = v * cosphi;
float V = (v * pow(cosphi, 3) / 6.0) * ( (v / p) - tanphi * tanphi );
float VI = (v * pow(cosphi, 5) / 120.0) * ( 5.0 - 18.0 * tanphi * tanphi + pow(tanphi, 4) + 14.0 * n2  - 58.0 * tanphi * tanphi * n2  );
float North = I + II * ldif * ldif + III * pow(ldif ,4) + IIIA * pow(ldif ,6);
float East = E0 + IV * ldif + V * pow(ldif ,3) + VI * pow(ldif ,5);
String region = NE2NGR(East , North);
lcd.print(region);
lcd.print(" ");
lcd.print(East,2);
lcd.print(" ");
lcd.setCursor(10,0);
lcd.print(North,2);
lcd.print(" ");
  if (debug){
  Serial.println(region);
  Serial.print("East ");Serial.println(East);
  Serial.print("North ");Serial.println(North);
  }
}

String NE2NGR(float &east, float &north){
/*
Extract the map section from the first digits of the longitude and latitude
Alex
http://www.codeproject.com/Articles/13577/GPS-Deriving-British-Ordnance-Survey-Grid-Referenc
*/
east = long(east + 0.5);
north = long(north + 0.5);
int intex;
int intnx;
float eX = east / 500000;
float nX = north / 500000;
float tmp = floor(eX) - 5.0 * floor(nX) + 17.0;
nX = 5 * (nX - floor(nX));
eX = 20.0 - 5.0 * floor(nX) + floor(5.0 * (eX - floor(eX)));
  if (eX > 7.5) eX = eX + 1;    // I is not used
  if (tmp > 7.5) tmp = tmp + 1; // I is not used
String eing = String(long(east));
String ning = String(long(north));
tmp = tmp + 65;
eX  = eX  + 65;
char temp [3]= {char(tmp), char(eX) , char(0)};
String region = String(temp);

// Adjust the accuraccy so that we get more recognisable grid reference numbers
int lnth = eing.length();
eing = eing.substring(lnth - 5);
//east = east/1000.0; // this way introduces more rounding erros
eing = eing.substring(0,2) + "." + eing.substring(2,5);
lnth = ning.length();
ning = ning.substring(lnth - 5);
//north = north/1000.0;
ning = ning.substring(0,2) + "." + ning.substring(2,5);
east = eing.toFloat();  // pass back through function call
north = ning.toFloat(); // pass back through function call
return region;
}

String lpad(String temp , byte L){
byte mylen = temp.length();
  if(mylen > (L - 1))return temp.substring(0,L-1);
  for (byte i=0; i< (L-mylen); i++) temp = " " + temp;
return temp;
}

String pad(String temp , byte L){
byte mylen = temp.length();
  if(mylen > (L - 1))return temp.substring(0,L-1);
  for (byte i=0; i< (L-mylen); i++) temp = temp + " ";
return temp;
}

void openfile(){
  if (!hascard){
  return;
  }
byte i;
char filename[18];
strcpy(filename, "/gps/GPSLOG00.TXT");
filename[17] =char(0);
boolean foundit =false;
  for (i = 0; i < 100; i++) {
  filename[11] = '0' + i/10;
  filename[12] = '0' + i%10;
    if (! SD.exists(filename)){
    foundit=true;
    break;
    }
  }
  if (!foundit){
  int thisbutton = -1;
  lcd.clear();
  lcd.print("Dir full-Erase?");
  lcd.setCursor(0,1);
  lcd.print("Sel:Ok Right:No");
    do{
    thisbutton = read_LCD_buttons();
    } while(thisbutton == btnNONE);
  lcd.clear();
    if (thisbutton == btnSELECT){
    lcd.print("Deleting");
      for(i=0; i<50; i++){
      filename[11] = '0' + i/10;
      filename[12] = '0' + i%10;
        if (SD.exists(filename)) SD.remove(filename); 
      }
    i=0;
    filename[11] = '0' + i/10;
    filename[12] = '0' + i%10;
    }else{
    error(4);
    hascard = false;
    return;  
    }
  }  
logfile = SD.open(filename, FILE_WRITE);
  if( ! logfile ) {
    if(debug) {Serial.print("Couldn't create ");Serial.println(filename);}
  error(3);
  hascard = false;
  }else{
  logging = true;
  myfile = filename;
  logfile.println(pad("Time   ",13) + lpad("Date   ",9) + lpad("Longitude",10) 
  + lpad("Latitude",10) + lpad("Altitude",9) + " m" + lpad("Geoid",9) + " m"+ lpad("Speed",7) + " mph Bearing");
  logfile.flush();
    if(debug) {Serial.print("Writing to ");Serial.println(filename);}
  }
lcd.clear();
}

void closefile(){
  if (!hascard) return;
logfile.flush();
logfile.close();
logging = false;
  if(debug) Serial.println("File " + myfile + " closed");
}

float distance_between (float lat1, float long1, float lat2, float long2){
/*
returns distance in meters between two positions, both specified 
as signed decimal-degrees latitude and longitude. Uses great-circle 
distance computation for hypothetical sphere of radius 6372795 meters.
Because Earth is no exact sphere, rounding errors may be up to 0.5%.
Author Maarten Lamers
*/
float delta = radians(long1-long2);
float sdlong = sin(delta);
float cdlong = cos(delta);
lat1 = radians(lat1);
lat2 = radians(lat2);
float slat1 = sin(lat1);
float clat1 = cos(lat1);
float slat2 = sin(lat2);
float clat2 = cos(lat2);
delta = (clat1 * slat2) - (slat1 * clat2 * cdlong); 
delta = sq(delta); 
delta += sq(clat2 * sdlong); 
delta = sqrt(delta); 
float denom = (slat1 * slat2) + (clat1 * clat2 * cdlong); 
delta = atan2(delta, denom); 
return delta * 6372795.0; 
}

void useInterrupt(boolean v) {
  if (v) {
    // Timer0 is already used for millis() - we'll just interrupt somewhere
    // in the middle and call the "Compare A" function above
    OCR0A = 0xAF;
    TIMSK0 |= _BV(OCIE0A);
    usingInterrupt = true;
  } 
  else {
    // do not call the interrupt function COMPA anymore
    TIMSK0 &= ~_BV(OCIE0A);
    usingInterrupt = false;
  }
}

// Interrupt is called once a millisecond, looks for any new GPS data, and stores it
SIGNAL(TIMER0_COMPA_vect) {
  char c = GPS.read();
  // if you want to debug, this is a good time to do it!
  #ifdef UDR0
      if (GPSECHO) if (c && debug) UDR0 = c; 
      // writing direct to UDR0 is much much faster than Serial.print 
      // but only one character can be written at a time. 
  #endif
}


int read_LCD_buttons(){
int adc_key_in = analogRead(lcdport);
  if (adc_key_in > lcdbut[0]) return btnNONE;
  while (analogRead(0) <= lcdbut[0]){delay(10);}
  
  if (adc_key_in < lcdbut[1])   return btnRIGHT;
  if (adc_key_in < lcdbut[2])  return btnUP;
  if (adc_key_in < lcdbut[3])  return btnDOWN;
  if (adc_key_in < lcdbut[4])  return btnLEFT;
  if (adc_key_in < lcdbut[5])  return btnSELECT; 
  return btnNONE;
}
