/*
 * udp.c
 *
 *  Created on: May 30, 2011
 *      Author: Justin
 */

#include <stddef.h>
#include "stm32f10x.h"

#include "udp.h"

#include "wiznet5100.h"

// offsets for UDP header
#define UDP_HEADER_DESTADDR      0
#define UDP_HEADER_PORT          4
#define UDP_HEADER_PAYLOADSIZE   6
#define UDP_HEADER_SIZE          8

extern bool Udp_OpenAvailableSocket(uint16_t port, uint8_t* socket)
{
	if (WIZnet5100_FindAvailableSocket(socket))
	{
	    return Udp_Open(*socket, port);
	}

	return FALSE;
}

extern bool Udp_Open(uint8_t socket, uint16_t port)
{
	bool retval = FALSE;

	if (!WIZnet5100_ClaimSocket(socket))
	{
		return FALSE;
	}

	// Make sure we close the socket first
	if (WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_SR) != SOCK_CLOSED)
	{
	    Udp_Close(socket);
	}

	// Assigned Socket 0 Mode Register
	WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_MR, MR_UDP);

	// Now open the Socket 0
	WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_PORT0, (port & 0xFF00) >> 8);
	WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_PORT1, (port & 0x00FF));
	WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR, CR_OPEN);                   // Open Socket

    // Wait for Opening Process
    while(WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR));

    // Check for Init Status
    uint8_t status = WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_SR);
    while (status != SOCK_UDP && status != SOCK_CLOSED)
    {
    	status = WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_SR);
    }
    if (status == SOCK_UDP)
    {
        retval = TRUE;
    }
    else
    {
    	Udp_Close(socket);
    }

    return retval;
}

extern bool Udp_Send(uint8_t socket, ipaddress_t ip, uint16_t port, const uint8_t *buf, uint16_t buflen)
{
    uint16_t ptr,offaddr,realaddr,txsize,timeout;

    if (buflen == 0) return FALSE;

    // Make sure the TX Free Size Register is available
    txsize=WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_FSR0);
	txsize=(((txsize & 0x00FF) << 8 ) + WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_FSR1));

    timeout=0;
    while (txsize < buflen)
    {
       txsize=WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_FSR0);
       txsize=(((txsize & 0x00FF) << 8 ) + WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_FSR1));
       // Timeout for approx 1000 ms
       if (timeout++ > 1000)
       {
         // Disconnect the connection
         Udp_Close(socket);
         return FALSE;
       }
   }

   // write teh value of remote ip
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DIPR0, ip[0]);
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DIPR1, ip[1]);
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DIPR2, ip[2]);
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DIPR3, ip[3]);

    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DPORT0, (port & 0xFF00) >> 8);
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DPORT1, (port & 0x00FF));

    // add a delay here
    //uint16_t index, index2;
    //for (index = 0; index <= 7200; index++)
    //	for (index2 = 0; index2 < 10000; index2++);

   // Read the Tx Write Pointer
   ptr = WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_WR0);
   offaddr = (((ptr & 0x00FF) << 8 ) + WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_WR1));

    while(buflen)
    {
        buflen--;
        // Calculate the real W5100 physical Tx Buffer Address
        realaddr = WIZnet5100_SocketTransmitBufferAddress(socket) + (offaddr & TX_BUF_MASK);
        // Copy the application data to the W5100 Tx Buffer
        WIZnet5100_WriteByte(realaddr,*buf);
        offaddr++;
        buf++;
    }

    // Increase the S0_TX_WR value, so it point to the next transmit
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_WR0,(offaddr & 0xFF00) >> 8 );
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_WR1,(offaddr & 0x00FF));

    // Now Send the SEND command
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR,CR_SEND);

    // Wait for Sending Process
    while(WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR));

    while ((WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_IR) & 0x10) == 0)
    {
    	if ((WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_IR) & 0x08) != 0)
    	{
    		WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_IR, 0x18);
    		return FALSE;
    	}
    }

    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_IR, 0x10);

    return TRUE;
}

extern bool Udp_Receive(uint8_t socket, ipaddress_t* fromip, uint16_t* fromport, uint8_t *buf, uint16_t buflen)
{
    uint16_t ptr,offaddr,realaddr;

    if (buflen == 0) return FALSE;

    // If the request size > MAX_BUF,just truncate it
    if (buflen > MAX_BUF)
      buflen=MAX_BUF - 2;
    // Read the Rx Read Pointer
    ptr = WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RD0);
    offaddr = (((ptr & 0x00FF) << 8 ) + WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RD1));

    uint16_t index;
    for(index = 0; index < 4; index++)
    {
    	realaddr = WIZnet5100_SocketReceiveBufferAddress(socket) + (offaddr & RX_BUF_MASK);
    	(*fromip)[index] = WIZnet5100_ReadByte(realaddr);
    	offaddr++;
    }

    *fromport = 0;
    for(index = 0; index < 2; index++)
    {
    	realaddr = WIZnet5100_SocketReceiveBufferAddress(socket) + (offaddr & RX_BUF_MASK);
    	*fromport = ((*fromport) << 8) + WIZnet5100_ReadByte(realaddr);
    	offaddr++;
    }

    uint16_t datasize = 0;
    for(index = 0; index < 2; index++)
    {
       	realaddr = WIZnet5100_SocketReceiveBufferAddress(socket) + (offaddr & RX_BUF_MASK);
       	datasize = (datasize << 8) + WIZnet5100_ReadByte(realaddr);
       	offaddr++;
    }

    while(buflen)
    {
      buflen--;
      realaddr = WIZnet5100_SocketReceiveBufferAddress(socket) + (offaddr & RX_BUF_MASK);
      *buf = WIZnet5100_ReadByte(realaddr);
      offaddr++;
      buf++;
    }
    *buf='\0';        // String terminated character

    // Increase the S0_RX_RD value, so it point to the next receive
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RD0,(offaddr & 0xFF00) >> 8 );
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RD1,(offaddr & 0x00FF));

    // Now Send the RECV command
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR,CR_RECV);
    //_delay_us(5);    // Wait for Receive Process
    while(WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR));

    return TRUE;
}

extern void Udp_Close(uint8_t socket)
{
	/* send the close command */
	WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR, CR_CLOSE);

	/* wait for the disconnecting process */
	while (WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR));

	WIZnet5100_ReleaseSocket(socket);
}

extern uint16_t Udp_GetReceivedSize(uint8_t socket)
{
    uint16_t size = ((WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RSR0) & 0x00FF) << 8) +
    		         (WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RSR1));

    if (size < UDP_HEADER_SIZE) return 0;

    return size - UDP_HEADER_SIZE;
}
