/*
 * tcp.c
 *
 *  Created on: May 30, 2011
 *      Author: Justin
 */

#include <stddef.h>
#include "stm32f10x.h"

#include "tcp.h"

#include "wiznet5100.h"

extern bool Tcp_OpenAvailableSocket(uint16_t port, uint8_t* socket)
{
	if (WIZnet5100_FindAvailableSocket(socket))
	{
	    return Tcp_Open(*socket, port);
	}

	return FALSE;
}

extern bool Tcp_Open(uint8_t socket, uint16_t tcp_port)
{
    bool retval = FALSE;

    if (!WIZnet5100_ClaimSocket(socket))
    {
    	return FALSE;
    }

    // Make sure we close the socket first
    if (WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_SR) != SOCK_CLOSED)
    {
        Tcp_Close(socket);
    }

    // Assigned Socket 0 Mode Register
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_MR, MR_TCP);

    // Now open the Socket 0
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_PORT0, (tcp_port & 0xFF00) >> 8);
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_PORT1, tcp_port & 0x00FF);
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR, CR_OPEN);                   // Open Socket

    // Wait for Opening Process
    while(WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR));

    // Check for Init Status
    if (WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_SR) == SOCK_INIT)
    {
        retval = TRUE;
    }
    else
    {
    	Tcp_Close(socket);
    }

    return retval;
}

extern bool Tcp_Connect(uint8_t socket, const ipaddress_t ip, uint16_t port)
{
	bool retval = FALSE;

	if (WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_SR) == SOCK_INIT)
	{
		// Set the IP address
		WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DIPR0, ip[0]);
		WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DIPR1, ip[1]);
		WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DIPR2, ip[2]);
		WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DIPR3, ip[3]);

		// Set the destination port
		WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DPORT0, (uint8_t)((port & 0xFF00) >> 8));
		WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_DPORT1, (uint8_t)(port & 0x00FF));

	    // Send the CONNECT command
	    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR, CR_CONNECT);

	    // Wait for connecting process
	    while(WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR));

	    // Check for connecting status
	    uint8_t status = WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_SR);
	    while (status != SOCK_ESTABLISHED && status != SOCK_CLOSED)
	    {
	    	status = WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_SR);
	    }
	    if (status == SOCK_ESTABLISHED)
	    {
	        retval = TRUE;
	    }
	    else
	    {
	        Tcp_Close(socket);
	    }
	 }
	 return retval;
}

extern bool Tcp_Listen(uint8_t socket)
{
   bool retval = FALSE;

   if (WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_SR) == SOCK_INIT)
   {
       // Send the LISTEN Command
       WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR, CR_LISTEN);

       // Wait for Listening Process
       while(WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR));

       // Check for Listen Status
       if (WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_SR) == SOCK_LISTEN)
       {
           retval = TRUE;
       }
       else
       {
           Tcp_Close(socket);
       }
    }
    return retval;
}

extern void Tcp_Close(uint8_t socket)
{
	/* send the close command */
	WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR, CR_CLOSE);

	/* wait for the disconnecting process */
	while (WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR));

	WIZnet5100_ReleaseSocket(socket);
}

extern void Tcp_Disconnect(uint8_t socket)
{
   /* send the disconnect command */
   WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR, CR_DISCON);

   /* wait for the disconnecting process */
   while(WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR));

   WIZnet5100_ReleaseSocket(socket);
}

extern bool Tcp_Send(uint8_t socket, const uint8_t *buf, uint16_t buflen)
{
    uint16_t ptr,offaddr,realaddr,txsize,timeout;

    if (buflen <= 0) return FALSE;

    // Make sure the TX Free Size Register is available
    txsize=WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_FSR0);
    txsize=(((txsize & 0x00FF) << 8 ) + WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_FSR1));

    timeout=0;
    while (txsize < buflen)
    {
       txsize=WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_FSR0);
       txsize=(((txsize & 0x00FF) << 8 ) + WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_FSR1));
       // Timeout for approx 1000 ms
       if (timeout++ > 1000)
       {
         // Disconnect the connection
         Tcp_Disconnect(socket);
         return FALSE;
       }
   }

   // Read the Tx Write Pointer
   ptr = WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_WR0);
   offaddr = (((ptr & 0x00FF) << 8 ) + WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_WR1));

    while(buflen)
    {
        buflen--;
        // Calculate the real W5100 physical Tx Buffer Address
        realaddr = WIZnet5100_SocketTransmitBufferAddress(socket) + (offaddr & TX_BUF_MASK);
        // Copy the application data to the W5100 Tx Buffer
        WIZnet5100_WriteByte(realaddr,*buf);
        offaddr++;
        buf++;
    }

    // Increase the S0_TX_WR value, so it point to the next transmit
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_WR0,(offaddr & 0xFF00) >> 8 );
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_TX_WR1,(offaddr & 0x00FF));

    // Now Send the SEND command
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR,CR_SEND);

    // Wait for Sending Process
    while(WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR));

    return TRUE;
}

extern bool Tcp_Receive(uint8_t socket,uint8_t *buf,uint16_t buflen)
{
    uint16_t ptr,offaddr,realaddr;

    if (buflen <= 0) return FALSE;

    // If the request size > MAX_BUF,just truncate it
    if (buflen > MAX_BUF)
      buflen=MAX_BUF - 2;
    // Read the Rx Read Pointer
    ptr = WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RD0);
    offaddr = (((ptr & 0x00FF) << 8 ) + WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RD1));

    while(buflen)
    {
      buflen--;
      realaddr = WIZnet5100_SocketReceiveBufferAddress(socket) + (offaddr & RX_BUF_MASK);
      *buf = WIZnet5100_ReadByte(realaddr);
      offaddr++;
      buf++;
    }
    *buf='\0';        // String terminated character

    // Increase the S0_RX_RD value, so it point to the next receive
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RD0,(offaddr & 0xFF00) >> 8 );
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RD1,(offaddr & 0x00FF));

    // Now Send the RECV command
    WIZnet5100_WriteByte(WIZnet5100_SocketRegisterOffset(socket) + S0_CR, CR_RECV);
    //_delay_us(5);    // Wait for Receive Process

    return TRUE;
}

extern uint16_t Tcp_GetReceivedBytes(uint8_t socket)
{
	uint16_t result = ((WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RSR0) & 0x00FF) << 8 ) + WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RSR1);
	while (result != ((WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RSR0) & 0x00FF) << 8 ) + WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RSR1))
	{
		result = ((WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RSR0) & 0x00FF) << 8 ) + WIZnet5100_ReadByte(WIZnet5100_SocketRegisterOffset(socket) + S0_RX_RSR1);
	}

	return result;
}
