/*
 * oauth.c
 *
 *  Created on: Jun 5, 2011
 *      Author: Justin
 */

#include "stm32f10x.h"

#include "oauth.h"
#include "dictionary.h"
#include "hmacsha1.h"
#include "base64.h"
#include "http.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#define OAUTH_VERSION  "1.0"
#define OAUTH_PARAMETERPREFIX  "oauth_"

#define OAUTH_CONSUMERKEY_KEY     "oauth_consumer_key"
#define OAUTH_CALLBACK_KEY        "oauth_callback"
#define OAUTH_VERSION_KEY         "oauth_version"
#define OAUTH_SIGNATUREMETHOD_KEY "oauth_signature_method"
#define OAUTH_SIGNATURE_KEY       "oauth_signature"
#define OAUTH_TIMESTAMP_KEY       "oauth_timestamp"
#define OAUTH_NONCE_KEY           "oauth_nonce"
#define OAUTH_TOKEN_KEY           "oauth_token"
#define OAUTH_TOKENSECRET_KEY     "oauth_token_secret"

#define OAUTH_HMACSHA1Signature   "HMAC-SHA1"
#define OAUTH_PlainTextSignature  "PLAINTEXT"
#define OAUTH_RSASHA1Signature    "RSA-SHA1"

static bool  _oauth_isinitialized = FALSE;
static char* _oauth_consumerkey;
static char* _oauth_consumersecret;
static char* _oauth_token;
static char* _oauth_tokensecret;

static char Oauth_UnreservedCharacters[] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-_.~";
static char Oauth_HexDigits[] = "0123456789ABCDEF";

extern void Oauth_Init(const char* consumerkey, const char* consumersecret, const char* token, const char* tokensecret)
{
	if (_oauth_isinitialized)
	{
		free(_oauth_consumerkey);
		free(_oauth_consumersecret);
		free(_oauth_token);
		free(_oauth_tokensecret);
	}

	_oauth_consumerkey = (char*)malloc(strlen(consumerkey) + 1);
	_oauth_consumersecret = (char*)malloc(strlen(consumersecret) + 1);
	_oauth_token = (char*)malloc(strlen(token) + 1);
	_oauth_tokensecret = (char*)malloc(strlen(tokensecret) + 1);

	strcpy(_oauth_consumerkey, consumerkey);
	strcpy(_oauth_consumersecret, consumersecret);
	strcpy(_oauth_token, token);
	strcpy(_oauth_tokensecret, tokensecret);

	_oauth_isinitialized = TRUE;
}

extern bool Oauth_IsInitialized()
{
	return _oauth_isinitialized;
}

static bool Oauth_IsReservedCharacter(char character)
{
    return strchr(Oauth_UnreservedCharacters, character) == 0;
}

static uint16_t Oauth_UrlEncode(const char* input, char* output, uint16_t size)
{
	if (output != 0)
	{
	    memset(output, 0, size + 1);
	}

	uint16_t character;
	uint16_t index = 0;
	for (character = 0; character < strlen(input); character++)
	{
		if (Oauth_IsReservedCharacter(input[character]))
		{
			if (output != 0)
			{
				output[index+0] = '%';
				output[index+1] = Oauth_HexDigits[(input[character] >> 4) & 0x0F];
				output[index+2] = Oauth_HexDigits[(input[character] >> 0) & 0x0F];
			}
			index += 3;
		}
		else
		{
			if (output != 0)
			{
			    output[index] = input[character];
			}
			index++;
		}
	}

    return index;
}

static uint16_t Oauth_GenerateNonce(char* nonce, uint16_t size)
{
	char* NONCE_CHARACTERS = "0123456789abcdef";

	if (nonce != 0)
	{
		memset(nonce, 0, size + 1);
	}

	// todo
	if (nonce != 0)
	{
		srand(RTC_GetCounter());

		uint16_t index;
		for (index = 0; index < 12; index++)
		{
			uint16_t random = rand() % strlen(NONCE_CHARACTERS);
	        nonce[index] = NONCE_CHARACTERS[random];
		}
	}
	return 12;
}

static uint16_t Oauth_GenerateTimestamp(char* timestamp, uint16_t size)
{
	if (timestamp != 0)
	{
		uint32_t time = RTC_GetCounter();
		sprintf(timestamp, "%lu", time);
	}
	return 10;
}

static uint16_t Oauth_NormalizeUrl(const char* url, char* result, uint16_t size)
{
    char* urlptr = (char*)url;
    uint16_t index = 0;

    // extract the protocol
    while (*urlptr != ':')
    {
        if (result != 0)
        {
            result[index] = *urlptr;
        }
        urlptr++;
        index++;
    }

    while (*urlptr == ':' || *urlptr == '/')
    {
        urlptr++;
    }
    if (result != 0)
    {
        strcpy(result + index, "://");
    }
    index += 3;

    // extract the domain
    while (*urlptr != ':' && *urlptr != '/' && *urlptr != '?' && *urlptr != 0)
    {
        if (result != 0)
        {
            result[index] = *urlptr;
        }
        urlptr++;
        index++;
    }

    if (result != 0)
    {
        result[index] = ':';
    }
    index++;

    // extract the port (if provided)
    if (*urlptr == ':')
    {
        urlptr++;
        while (*urlptr != '/' && *urlptr != '?' && *urlptr != 0)
        {
            if (result != 0)
            {
                result[index] = *urlptr;
            }
            urlptr++;
            index++;
        }
    }
    else
    {
        // todo should check protocol
    	if (result != 0)
    	{
            strcpy(result + index, "80");
    	}
        index += 2;
    }

    // extract the path
    if (*urlptr == '/')
    {
        while (*urlptr != '?' && *urlptr != 0)
        {
            if (result != 0)
            {
                result[index] = *urlptr;
            }
            urlptr++;
            index++;
        }
    }

    if (result != 0)
    {
    	result[index] = 0;
    }
    return index;
}

static uint16_t Oauth_NormalizeParameters(const dictionary_t* parameters, char* normalizedparameters, uint16_t size)
{
	uint16_t index = 0;

	dictionaryentry_t* item = Dictionary_GetFirst(parameters);
	while (item != 0)
	{
		if (index != 0)
		{
			if (normalizedparameters != 0)
			{
				normalizedparameters[index] = '&';
			}
			index++;
		}

		if (normalizedparameters != 0)
		{
			strcpy(normalizedparameters + index, item->Name);
		}
		index += strlen(item->Name);

		if (normalizedparameters != 0)
		{
			normalizedparameters[index] = '=';
		}
		index++;

		uint16_t encodedsize = Oauth_UrlEncode(item->Value, 0, 0);
		char* encoded = (char*)malloc(encodedsize + 1);
		Oauth_UrlEncode(item->Value, encoded, encodedsize);
		if (normalizedparameters != 0)
		{
			strcpy(normalizedparameters + index, encoded);
		}
		index += encodedsize;
		free(encoded);

		item = Dictionary_GetNext(item);
	}

	return index;
}

static uint16_t Oauth_GenerateSignatureBase(const char* url, dictionary_t* parameters, char* signaturebase, uint16_t size)
{
	if (signaturebase != 0)
	{
		memset(signaturebase, 0, size + 1);
	    strcpy(signaturebase, "POST");
	}
	uint16_t index = 4;

	if (signaturebase != 0)
    {
	    signaturebase[index] = '&';
    }
	index++;

	//uint16_t normalizedurlsize = Oauth_NormalizeUrl(url, 0, 0);
	//char* normalizedurl = (char*)malloc(normalizedurlsize + 1);
	//Oauth_NormalizeUrl(url, normalizedurl, normalizedurlsize);
	char* normalizedurl = (char*)malloc(strlen(url) + 1);
	strcpy(normalizedurl, url);

	uint16_t encodedurlsize = Oauth_UrlEncode(normalizedurl, 0, 0);
	if (signaturebase != 0)
	{
	    Oauth_UrlEncode(normalizedurl, signaturebase + index, encodedurlsize);
	}
	index += encodedurlsize;
	free(normalizedurl);

	if (signaturebase != 0)
	{
	    signaturebase[index] = '&';
	}
	index++;

	uint16_t normalizedparameterssize = Oauth_NormalizeParameters(parameters, 0, 0);
	char* normalizedparameters = (char*)malloc(normalizedparameterssize + 1); memset(normalizedparameters, 0, normalizedparameterssize + 1);
	Oauth_NormalizeParameters(parameters, normalizedparameters, normalizedparameterssize);

	uint16_t encodedparameterssize = Oauth_UrlEncode(normalizedparameters, 0, 0);
	if (signaturebase != 0)
	{
	    Oauth_UrlEncode(normalizedparameters, signaturebase + index, encodedparameterssize);
	}
	index += encodedparameterssize;
	free(normalizedparameters);

    return index;
}

static uint16_t Oauth_ComputeHash(const char* data, const char* key, char* hash, uint16_t size)
{
	uint32_t result[5];
	hmac_sha1(result, key, strlen(key) * 8, data, strlen(data) * 8);
	return Base64_Encode((uint8_t*)result, 20, hash, size); // result size is 5 * sizeof(uint32_t)
}

static uint16_t Oauth_GenerateSignature(const char* url, dictionary_t* postdata, char* signature, uint16_t size)
{
	// generate the signature base
	uint16_t signaturebasesize = Oauth_GenerateSignatureBase(url, postdata, 0, 0);
	char* signaturebase = (char*)malloc(signaturebasesize + 1);
	Oauth_GenerateSignatureBase(url, postdata, signaturebase, signaturebasesize);

	// determine the key size
	uint16_t keysize = Oauth_UrlEncode(_oauth_consumersecret, 0, 0) +
			           Oauth_UrlEncode(_oauth_tokensecret, 0, 0) +
			           1;

	// create the HMAC-SHA1 key
	char* key = (char*)malloc(keysize + 1); memset(key, 0, keysize + 1);
	char* keyptr = key;
	keyptr += Oauth_UrlEncode(_oauth_consumersecret, keyptr, keysize);
	strcat(keyptr++, "&");
	Oauth_UrlEncode(_oauth_tokensecret, keyptr, keysize - (keyptr - key));

	// generate the signature
	uint16_t result = Oauth_ComputeHash(signaturebase, key, signature, size);
	free(key);
	free(signaturebase);
	return result;
}

extern bool Oauth_Get(const char* url, uint8_t* socket)
{
	return FALSE;
}
extern bool Oauth_Post(const char* url, dictionary_t* parameters, uint8_t* socket)
{
	// consumer key
	Dictionary_Add(parameters, OAUTH_CONSUMERKEY_KEY, _oauth_consumerkey);

	// nonce
	uint16_t noncesize = Oauth_GenerateNonce(0, 0);
	char* nonce = (char*)malloc(noncesize + 1);
    Oauth_GenerateNonce(nonce, noncesize);
    Dictionary_Add(parameters, OAUTH_NONCE_KEY, nonce);
    free(nonce);

	// signature method
	Dictionary_Add(parameters, OAUTH_SIGNATUREMETHOD_KEY, OAUTH_HMACSHA1Signature);

	// timestamp
	uint16_t timestampsize = Oauth_GenerateTimestamp(0, 0);
    char* timestamp = (char*)malloc(timestampsize + 1);
	Oauth_GenerateTimestamp(timestamp, timestampsize);
	Dictionary_Add(parameters, OAUTH_TIMESTAMP_KEY, timestamp);
	free(timestamp);

	// token
	Dictionary_Add(parameters, OAUTH_TOKEN_KEY, _oauth_token);

	// version
	Dictionary_Add(parameters, OAUTH_VERSION_KEY, OAUTH_VERSION);

	uint16_t signaturesize = Oauth_GenerateSignature(url, parameters, 0, 0);
	char* signature = (char*)malloc(signaturesize + 1);
	Oauth_GenerateSignature(url, parameters, signature, signaturesize);
	Dictionary_Add(parameters, OAUTH_SIGNATURE_KEY, signature);
	free(signature);

	uint16_t postdatasize = Oauth_NormalizeParameters(parameters, 0, 0);
	char* postdata = (char*)malloc(postdatasize + 1);
	Oauth_NormalizeParameters(parameters, postdata, postdatasize);
	bool result = Http_Post(url, "application/x-www-form-urlencoded", (uint8_t*)postdata, postdatasize, socket);
	free(postdata);

	return result;
}

extern uint16_t Oauth_GetReceivedBytes(uint8_t socket)
{
	return Http_GetReceivedBytes(socket);
}

extern bool Oauth_GetResponse(uint8_t socket, char* buffer, uint16_t buffersize)
{
	return Http_GetResponse(socket, buffer, buffersize);
}

extern void Oauth_Close(uint8_t socket)
{
	Http_Close(socket);
}
