#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "stm32f10x.h"
#include "stm32f10x_rtc.h"
#include "interrupts.h"
#include "wiznet5100.h"
#include "dhcp.h"
#include "dns.h"
#include "tcp.h"
#include "sntp.h"
#include "webserver.h"
#include "oauth.h"

#include "twitter.h"

#include "nokia5110.h"
#include "easybutton.h"

/* Private definitions -------------------------------------------------------*/
#define __RTC_CLKSRC_VAL          0x00000100
#define __RTC_PERIOD              0x000003E8
#define __RTCCLK                  (32768)

int __errno = 0;

/* Private variables ---------------------------------------------------------*/
volatile uint32_t __delay;

/* Private function prototypes -----------------------------------------------*/
static void _Init();
static void _Delay(uint32_t nTime);
static void _OnSysTick(void);
static void _TweetCallback(uint16_t statuscode, const char* reason);

static const char* _difficultthings[] =
{
	"Anti-gravity",
	"Faster than light travel",
	"World peace",
	"Time travel",
	"Human cloning",
	"Perpetual motion",
	"Training a cat",
	"Understanding women",
	"Licking my elbow",
	"Quantum mechanics"
};

static const uint16_t _difficultthings_size = 10;

int main(void)
{
	_Init();

	Nokia5110_Init();
	Nokia5110_ClearDisplay();
	Nokia5110_WriteString("Easy Button", 0);
	Nokia5110_WriteString("Aquiring IP", 2);

	WIZnet5100_Init();
	while(!Dhcp_Start());

	uint32_t time;
	while (!Sntp_GetTime(&time));

	// set the Real Time Clock
	RTC_WaitForLastTask();
	RTC_SetCounter(time);
	RTC_WaitForLastTask();
	RTC_WaitForSynchro();

	char buffer[15];
	ipaddress_t ip;
	WIZnet5100_GetIpAddress(&ip);
	sprintf(buffer, "%d.%d.%d.%d", ip[0], ip[1], ip[2], ip[3]);

	Nokia5110_WriteString("Browse to", 2);
	Nokia5110_WriteString(buffer, 3);

	Twitter_RegisterCallback(_TweetCallback);

	// Loop forever
	uint8_t socket;
    WebServer_Init(&socket);
	while (TRUE)
	{
		WebServer_Check(socket);

		if (Oauth_IsInitialized())
		{
			Nokia5110_WriteString("Ready", 2);
			Nokia5110_WriteString("", 3);

			if (EasyButton_IsPressed())
			{
				Nokia5110_WriteString("", 5);

				// debounce
				_Delay(100);

				srand(RTC_GetCounter());
				uint16_t index = rand() % _difficultthings_size;

				char buffer[128]; memset(buffer, 0, 128);
				sprintf(buffer, "%s - that was easy!", _difficultthings[index]);

				Twitter_Tweet(buffer);

				// await release
				while (EasyButton_IsPressed())
				{
					WebServer_Check(socket);
				}
			}
		}
	}

	return 0;
}

/* Private functions ---------------------------------------------------------*/

static void _Init()
{
	SystemInit();

	/* Enable PWR and BKP clocks ---------------------------------------------*/
	RCC_APB1PeriphClockCmd(RCC_APB1Periph_PWR | RCC_APB1Periph_BKP, ENABLE);

	/* Allow access to BKP Domain --------------------------------------------*/
	PWR_BackupAccessCmd(ENABLE);

	/* Reset Backup Domain ---------------------------------------------------*/
	BKP_DeInit();

	/* Enable LSE ------------------------------------------------------------*/
	RCC_LSEConfig(RCC_LSE_ON);

	/* Wait till LSE is ready ------------------------------------------------*/
	while(RCC_GetFlagStatus(RCC_FLAG_LSERDY) == RESET);

	/* Select LSE as RTC Clock Source ----------------------------------------*/
	RCC_RTCCLKConfig(RCC_RTCCLKSource_LSE);

	/* Enable RTC Clock ------------------------------------------------------*/
	RCC_RTCCLKCmd(ENABLE);
	RTC_WaitForLastTask();

	/* Wait for RTC registers synchronization --------------------------------*/
	RTC_WaitForSynchro();

	/* Wait until last write operation on RTC registers has finished ---------*/
	RTC_WaitForLastTask();

	/* Set RTC prescaler: set RTC period to 1sec -----------------------------*/
	RTC_SetPrescaler(__RTC_PERIOD*__RTCCLK/1000-1); // RTC period = RTCCLK/RTC_PR = (32.768 KHz)/(32767+1)

	/* Wait until last write operation on RTC registers has finished ---------*/
	RTC_WaitForLastTask();

	/* Wait for RTC registers synchronization --------------------------------*/
	RTC_WaitForSynchro();

	/* Setup SysTick Timer for 1 msec interrupts -----------------------------*/
	SysTick_Config(SystemCoreClock / 1000);
	Interrupts_RegisterCallback(INTERRUPTS_SysTick, _OnSysTick);

	_Delay(1000); // found to be necessary during testing - reason unknown
}

// _OnSysTick()
// ----------
// Decrement the delay counter.
static void _OnSysTick(void)
{
    if (__delay != 0x00)
    {
        __delay--;
    }
}

// _Delay()
// --------
// Delay for a given number of milliseconds.
static void _Delay(uint32_t time)
{
    __delay = time;

    while(__delay != 0);
}

static void _TweetCallback(uint16_t statuscode, const char* reason)
{
	Nokia5110_WriteString(reason, 5);
}
