/*
 * http.c
 *
 *  Created on: May 30, 2011
 *      Author: Justin
 */
#include "stm32f10x.h"

#include "http.h"
#include "wiznet5100.h"
#include "tcp.h"
#include "dns.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

static Http_GetFunc _getfunc = 0;
static Http_PostFunc _postfunc = 0;

static uint16_t Http_ExtractDomain(const char* url, char* result, uint16_t size);
static uint16_t Http_ReadLine(const char* buffer, char* result, uint16_t size);
static uint16_t Http_GetHeaderValue(const char* line, char* result, uint16_t size);
static uint16_t Http_GetHeader(const char* buffer, const char* headername, char* result, uint16_t size);

extern void Http_Init(uint8_t socket, Http_GetFunc getfunc, Http_PostFunc postfunc)
{
	_getfunc = getfunc;
	_postfunc = postfunc;
}

extern void Http_Check(uint8_t socket)
{
	uint16_t size;
	uint8_t status = WIZnet5100_Status(socket);

    switch(status)
    {
	case SOCK_CLOSED:
		// Open the socket
	    if (Tcp_Open(socket, 80))
		{
		    // Listen to socket
		    Tcp_Listen(socket);
	    }
	    break;

    case SOCK_ESTABLISHED:
	    // Get the client request size
        size = Tcp_GetReceivedBytes(socket);

        if (size > 0)
		{
        	char* buffer = (char*)malloc(size + 1);
        	memset(buffer, 0, size + 1);

		    // Now read the client Request
			if (!Tcp_Receive(socket, (uint8_t*)buffer, size))
			{
				free(buffer);
				break;
			}

			// Check the Request Header
			if (_getfunc != 0 && strncmp(buffer, "GET /", strlen("GET /")) == 0)
			{
				_getfunc(buffer);
			}
			else if (_postfunc != 0 && strncmp(buffer, "POST /", strlen("POST /")) == 0)
			{
				_postfunc(buffer);
			}

			free(buffer);
		}
		break;

	case SOCK_FIN_WAIT:
	case SOCK_CLOSING:
	case SOCK_TIME_WAIT:
	case SOCK_CLOSE_WAIT:
	case SOCK_LAST_ACK:
	    // Force to close the socket
		Tcp_Close(socket);
	    break;
    }
}

extern bool Http_Get(const char* url, uint8_t* socket);
extern bool Http_Post(const char* url, const char* contenttype, uint8_t* content, uint16_t contentlength, uint8_t* socket)
{
	char post[512]; memset(post, 0, 512);

	/*
	strcpy(post, "POST ");
	strcat(post, url);
	strcat(post, " HTTP/1.0\r\n");

	strcat(post, "Content-Length: ");
	strcat(post, itoa(contentlength));
	strcat(post, "\r\n");

	strcat(post, "Content-Type: ");
	strcat(post, contenttype);
	strcat(post, "\r\n");

	strcat("\r\n");

	strcat("content");
	*/
	sprintf(post, "POST %s HTTP/1.0\r\nContent-Length: %u\r\nContent-Type: %s\r\n\r\n%s", url, contentlength, contenttype, content);

	uint16_t domainsize = Http_ExtractDomain(url, 0, 0);
	char* domain = (char*)malloc(domainsize + 1);
	Http_ExtractDomain(url, domain, domainsize);

	bool result = FALSE;
	uint16_t retrys = 5;
	while (!result && retrys--)
	{
		ipaddress_t ip;
		if (!Dns_Lookup(domain, &ip))
		{
			continue;
		}

		if (!Tcp_OpenAvailableSocket(0xC000, socket))
		{
			continue;
		}

		//if (!Tcp_Connect(*socket, ip, 8080))
		if (!Tcp_Connect(*socket, ip, 80))
		{
			Tcp_Close(*socket);
			continue;
		}

		if (!Tcp_Send(*socket, (uint8_t*)post, strlen(post)))
		{
			Tcp_Close(*socket);
			continue;
		}

		result = TRUE;
	}

	return result;
}

extern uint16_t Http_GetReceivedBytes(uint8_t socket)
{
	return Tcp_GetReceivedBytes(socket);
}

extern uint16_t Http_GetResponse(uint8_t socket, char* buffer, uint16_t size)
{
	uint16_t result = Http_GetReceivedBytes(socket);
	if (buffer != 0)
	{
	    Tcp_Receive(socket, (uint8_t*)buffer, size);
	}
	return result;
}

extern void Http_Close(uint8_t socket)
{
	Tcp_Close(socket);
}

static uint16_t Http_ExtractDomain(const char* url, char* result, uint16_t size)
{
    char* urlptr = (char*)url;
    uint16_t index = 0;

    if (result != 0)
    {
    	memset(result, 0, size + 1);
    }

    // skip the protocol
    while (*urlptr != ':')
    {
        urlptr++;
    }

    while (*urlptr == ':' || *urlptr == '/')
    {
        urlptr++;
    }

    // extract the domain
    while (*urlptr != ':' && *urlptr != '/' && *urlptr != '?' && *urlptr != 0)
    {
        if (result != 0)
        {
            result[index] = *urlptr;
        }
        urlptr++;
        index++;
    }

    return index;
}

extern httpmethod_t Http_GetRequestMethod(const char* request)
{
	if (strncmp(request, "GET ", 4) == 0)
	{
		return HTTP_GET;
	}

	if (strncmp(request, "POST ", 5) == 0)
	{
		return HTTP_POST;
	}

	return HTTP_UnknownMethod;
}

extern uint16_t Http_GetRequestURL(const char* request, char* result, uint16_t size)
{
	if (result != 0)
	{
		memset(result, 0, size + 1);
	}

	char* ptr = (char*)request;

	// skip the method
	while (*ptr != 0 && *ptr != ' ' && *ptr != '\t') ptr++;

	// skip the whitespace
	while (*ptr == ' ' || *ptr == '\t') ptr++;

	uint16_t index = 0;
	while (*ptr != 0 && *ptr != ' ' && *ptr != '\t')
	{
		if (result != 0 && index < size)
		{
			result[index] = *ptr;
		}

		index++;
		ptr++;
	}

	return index;
}
extern httpversion_t Http_GetRequestVersion(const char* request)
{
	char* ptr = (char*)request;

	// skip the method
	while (*ptr != 0 && *ptr != ' ' && *ptr != '\t') ptr++;

	// skip the whitespace
	while (*ptr == ' ' || *ptr == '\t') ptr++;

	// skip the url
	while (*ptr != 0 && *ptr != ' ' && *ptr != '\t') ptr++;

	// skip the whitespace
	while (*ptr == ' ' || *ptr == '\t') ptr++;

	if (strncmp(ptr, "HTTP/1.0", 8) == 0)
	{
		return HTTP_1_0;
	}

	if (strncmp(ptr, "HTTP/1.1", 8) == 0)
	{
		return HTTP_1_1;
	}

	return HTTP_UnknownVersion;
}

extern httpversion_t Http_GetResponseVersion(const char* response)
{
	if (strncmp(response, "HTTP/1.0", 8) == 0)
	{
		return HTTP_1_0;
	}

	if (strncmp(response, "HTTP/1.1", 8) == 0)
	{
		return HTTP_1_1;
	}

	return HTTP_UnknownVersion;
}

extern uint8_t Http_GetResponseStatusCode(const char* response)
{
	char* ptr = (char*)response;

	// skip the version
	while (*ptr != 0 && *ptr != ' ' && *ptr != '\t') ptr++;

	// skip the whitespace
	while (*ptr == ' ' || *ptr == '\t') ptr++;

	return atoi(ptr);
}

extern uint16_t Http_GetResponseReason(const char* response, char* result, uint16_t size)
{
	if (result != 0)
	{
		memset(result, 0, size + 1);
	}

    char* ptr = (char*)response;

	// skip the version
	while (*ptr != 0 && *ptr != ' ' && *ptr != '\t') ptr++;

	// skip the whitespace
	while (*ptr == ' ' || *ptr == '\t') ptr++;

	// skip the status code
	while (*ptr != 0 && *ptr != ' ' && *ptr != '\t') ptr++;

	// skip the whitespace
	while (*ptr == ' ' || *ptr == '\t') ptr++;

	// read the reason string
	uint16_t index = 0;
	while (*ptr != 0 && *ptr != '\r' && *ptr != '\n')
	{
		if (result != 0 && index < size)
		{
			result[index] = *ptr;
		}
		index++;
		ptr++;
	}

	return index;
}

extern uint16_t Http_GetContentType(const char* request, char* result, uint16_t size)
{
    return Http_GetHeader(request, "Content-Length", result, size);
}

extern uint16_t Http_GetContentLength(const char* request)
{
    uint16_t contentlengthsize = Http_GetHeader(request, "Content-Length", 0, 0);
    char* contentlength = (char*)malloc(contentlengthsize + 1);
    Http_GetHeader(request, "Content-Length", contentlength, contentlengthsize);

    uint16_t result = atoi(contentlength);

    free(contentlength);
    return result;
}

extern uint16_t Http_GetContent(const char* buffer, char* result, uint16_t size)
{
    if (result != 0)
    {
        memset(result, 0, size + 1);
    }

    char* ptr = (char*)buffer;

    uint16_t linesize = Http_ReadLine(ptr, 0, 0);
    while (linesize > 0)
    {
        ptr += linesize;
        if (*ptr == '\r') ptr++;
        if (*ptr == '\n') ptr++;

        linesize = Http_ReadLine(ptr, 0, 0);
    }

    if (*ptr == '\r') ptr++;
    if (*ptr == '\n') ptr++;

    uint16_t index = 0;
    while (*ptr != 0)
    {
        if (result != 0 && index < size)
        {
            result[index] = *ptr;
        }

        index++;
        ptr++;
    }

    return index;
}

static uint16_t Http_ReadLine(const char* buffer, char* result, uint16_t size)
{
    if (result != 0)
    {
        memset(result, 0, size + 1);
    }

    char* ptr = (char*)buffer;

    uint16_t index = 0;
    while (*ptr != 0 && *ptr != '\r' && *ptr != '\n')
    {
        if (result != 0 && index < size)
        {
            result[index] = *ptr;
        }

        index++;
        ptr++;
    }

    return index;
}

// split the first line of a header; caller will need to append any folded lines
static uint16_t Http_GetHeaderValue(const char* line, char* result, uint16_t size)
{
    if (result != 0)
    {
        memset(result, 0, size + 1);
    }

    char* ptr = (char*)line;

    // skip the header name
    while (*ptr != 0 && *ptr != ':') ptr++;

    // skip the separator
    if (*ptr != 0) ptr++;

    // skip any whitespace
    while (*ptr == ' ' || *ptr == '\t') ptr++;

    // read in the header value
    uint16_t index = 0;
    while(*ptr != 0)
    {
        if (result != 0 && index < size)
        {
            result[index] = *ptr;
        }

        index++;
        ptr++;
    }

    return index;
}

static uint16_t Http_GetHeader(const char* buffer, const char* headername, char* result, uint16_t size)
{
    char* ptr = (char*)buffer;

    // skip the first line
    *ptr += Http_ReadLine(ptr, 0, 0);
    if (*ptr == '\r') ptr++;
    if (*ptr == '\n') ptr++;

    uint16_t linesize = Http_ReadLine(ptr, 0, 0);
    while (linesize > 0)
    {
        bool foundheader = FALSE;
        uint16_t headersize;

        char* line = (char*)malloc(linesize + 1);
        Http_ReadLine(ptr, line, linesize);

        foundheader = (strncmp(line, headername, strlen(headername)) == 0);
        if (foundheader)
        {
            headersize = Http_GetHeaderValue(line, 0, 0);
            if (result != 0)
            {
                Http_GetHeaderValue(line, result, size);
            }
        }

        free(line);

        ptr += linesize;
        if (*ptr == '\r') ptr++;
        if (*ptr == '\n') ptr++;

        while (*ptr == ' ' || *ptr == '\t')
        {
            uint16_t additionallinesize = Http_ReadLine(ptr, 0, 0);
            char* additionalline = (char*)malloc(additionallinesize);
            Http_ReadLine(ptr, additionalline, additionallinesize);

            headersize += additionallinesize;

            if (foundheader)
            {
                if (result != 0)
                {
                    strcat(result, additionalline);
                }
            }

            free(additionalline);

            ptr += linesize;
            if (*ptr == '\r') ptr++;
            if (*ptr == '\n') ptr++;
        }

        if (foundheader)
        {
            return headersize;
        }

        linesize = Http_ReadLine(ptr, 0, 0);
    }

    return 0;
}
