#include "formdata.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dictionary.h"

static uint16_t Formdata_ExtractField(const char* field, uint16_t fieldindex, char* result, uint16_t size);
static uint16_t Formdata_ExtractKey(const char* field, char* result, uint16_t size);
static uint16_t Formdata_ExtractValue(const char* field, char* result, uint16_t size);

/*
*    Formdata_Decode()
*    -----------------
*
*    const char*   formdata: Buffer to extract the fields from.
*    dictionary_t* result:   Dictionary of resulting key/value fields.
*
*    bool          RETURN:   TRUE on success.
*/
// todo - handle URL decoding of formdata values (return FALSE on error)
extern bool Formdata_Decode(const char* formdata, dictionary_t* result)
{
    uint16_t fieldindex = 0;
    uint16_t fieldsize = Formdata_ExtractField(formdata, fieldindex, 0, 0);
    while (fieldsize > 0)
    {
        char* field = (char*)malloc(fieldsize + 1);
        Formdata_ExtractField(formdata, fieldindex, field, fieldsize);

        uint16_t keysize = Formdata_ExtractKey(field, 0, 0);
        char* key = (char*)malloc(keysize + 1);
        Formdata_ExtractKey(field, key, keysize);

        uint16_t valuesize = Formdata_ExtractValue(field, 0, 0);
        char* value = (char*)malloc(valuesize + 1);
        Formdata_ExtractValue(field, value, valuesize);

        Dictionary_Add(result, key, value);

        free(key);
        free(value);
        free(field);
 
        fieldindex++;
        fieldsize = Formdata_ExtractField(formdata, fieldindex, 0, 0);
    }

    return TRUE;
}

/*
*    Formdata_ExtractField()
*    -----------------------
*
*    const char* formdata: Buffer to extract the field from.
*    uint16_t    field:    The index of the field to extract.
*    char*       result:   Allocated buffer for result (or NULL if calling to determine size).
*    uint16_t    size:     The size of the allocated buffer (excludes null terminator).
*
*    uint16_t    RETURN:   The size of the requested field.
*/
static uint16_t Formdata_ExtractField(const char* formdata, uint16_t field, char* result, uint16_t size)
{
    if (result != 0)
    {
        memset(result, 0, size + 1);
    }

    char* ptr = (char*)formdata;

    uint16_t fieldindex = 0;
    for (fieldindex = 0; fieldindex < field; fieldindex++)
    {
        while (*ptr != 0 && *ptr != '&')
        {
            ptr++;
        }

        if (*ptr == 0)
        {
            return 0;
        }

        ptr++;
    }

    uint16_t index = 0;
    while(*ptr != 0 && *ptr != '&')
    {
        if (result != 0 && index < size)
        {
            result[index] = *ptr;
        }

        index++;
        ptr++;
    }

    return index;
}

/*
*    Formdata_ExtractKey()
*    ---------------------
*
*    const char* field:  Buffer to extract the key from.
*    char*       result: Allocated buffer for result (or NULL if calling to determine size).
*    uint16_t    size:   The size of the allocated buffer (excludes null terminator).
*
*    uint16_t    RETURN: The size of the requested key.
*/
static uint16_t Formdata_ExtractKey(const char* field, char* result, uint16_t size)
{
    if (result != 0)
    {
        memset(result, 0, size + 1);
    }

    char* ptr = (char*)field;

    uint16_t index = 0;
    while(*ptr != 0 && *ptr != '=')
    {
        if (result != 0 && index < size)
        {
            result[index] = *ptr;
        }

        index++;
        ptr++;
    }

    return index;
}

/*
*    Formdata_ExtractValue()
*    -----------------------
*
*    const char* field:  Buffer to extract the value from.
*    char*       result: Allocated buffer for result (or NULL if calling to determine size).
*    uint16_t    size:   The size of the allocated buffer (excludes null terminator).
*
*    uint16_t    RETURN: The size of the requested value.
*/
static uint16_t Formdata_ExtractValue(const char* field, char* result, uint16_t size)
{
    if (result != 0)
    {
        memset(result, 0, size + 1);
    }

    char* ptr = (char*)field;

    while(*ptr != 0 && *ptr != '=')
    {
        ptr++;
    }

    if (*ptr == 0)
    {
        return 0;
    }

    ptr++; // skip the '='

    uint16_t index = 0;
    while(*ptr != 0)
    {
        if (result != 0 && index < size)
        {
            result[index] = *ptr;
        }

        index++;
        ptr++;
    }

    return index;
}
