/*
 * dns.c
 *
 *  Created on: May 31, 2011
 *      Author: Justin
 */
#include "stm32f10x.h"

#include "dns.h"
#include "udp.h"

#include <stdlib.h>
#include <string.h>

#define DNS_PORT   53

#define DNS_HEADERLENGTH 12

// flags and codes (high byte)
#define QR_QUERY          (0 << 7)
#define QR_RESPONSE       (1 << 7)
#define OPCODE_QUERY      (0 << 3)
#define OPCODE_IQUERY     (1 << 3) // obsolete
#define OPCODE_STATUS     (2 << 3)
// reserved
#define OPCODE_NOTIFY     (4 << 3)
#define OPCODE_UPDATE     (5 << 3)
#define AUTHORITIVE       (1 << 3)
#define TRUNCATION        (1 << 1)
#define RECURSION_DESIRED (1 << 0)

// flags and codes (low byte)
#define RECURSION_AVAILABLE (1 << 7)
#define RCODE_NOERROR        0
#define RCODE_FORMATERROR    1
#define RCODE_SERVERFAILURE  2
#define RCODE_NAMEERROR      3
#define RCODE_NOTIMPLEMENTED 4
#define RCODE_REFUSED        5
#define RCODE_YXDOMAIN       6
#define RCODE_YXRRSET        7
#define RCODE_NXRRSET        8
#define RCODE_NOTAUTH        9
#define RCODE_NOTZONE       10

static ipaddress_t _primarydns   = { 0, 0, 0, 0 };
static ipaddress_t _secondarydns = { 0, 0, 0, 0 };

// forward declarations
static bool Dns_SendQuery(uint8_t socket, const char* host);
static bool Dns_GetResponse(uint8_t socket, ipaddress_t* ip);

extern void Dns_SetPrimary(ipaddress_t ip)
{
	_primarydns[0] = ip[0];
	_primarydns[1] = ip[1];
	_primarydns[2] = ip[2];
	_primarydns[3] = ip[3];
}

extern void Dns_SetSecondary(ipaddress_t ip)
{
	_secondarydns[0] = ip[0];
	_secondarydns[1] = ip[1];
	_secondarydns[2] = ip[2];
	_secondarydns[3] = ip[3];
}

extern bool Dns_Lookup(const char* host, ipaddress_t* ip)
{
    bool result = FALSE;

    uint8_t socket;
    if (!Udp_OpenAvailableSocket(DNS_PORT, &socket))
    {
       	return FALSE;
    }

    if (!Dns_SendQuery(socket, host))
    {
    	Udp_Close(socket);
    	return FALSE;
    }

    result = Dns_GetResponse(socket, ip);

    Udp_Close(socket);
    return result;
}

static bool Dns_SendQuery(uint8_t socket, const char* host)
{
	bool result = FALSE;

	uint16_t hostlength = strlen(host);
	uint8_t* buffer = (uint8_t*)malloc(DNS_HEADERLENGTH + hostlength);

	// identifier
	buffer[0]  = 42;
	buffer[1]  = 42;

	// flags and codes
	buffer[2]  = QR_QUERY & OPCODE_QUERY & RECURSION_DESIRED;
	buffer[3]  = 0;

	buffer[4]  = 0; // question resource records (WORD)
	buffer[5]  = 1;
	buffer[6]  = 0; // answer resource records (WORD)
	buffer[7]  = 0;
	buffer[8]  = 0; // authority resource records (WORD)
	buffer[9]  = 0;
	buffer[10] = 0; // additional resource records (WORD)
	buffer[11] = 0;

	uint16_t index = DNS_HEADERLENGTH;
	const char* startptr = host;
	const char* endptr = host + hostlength;
	while (startptr < endptr)
	{
		const char* nextptr = strchr(startptr, '.');
		if (!nextptr)
		{
			nextptr = endptr;
		}

		uint8_t size = (uint8_t)(nextptr - startptr);
		buffer[index++] = size;
		strncpy((char*)buffer + index, startptr, size);
		index += size;

		if (*nextptr == '.')
		{
			startptr = nextptr + 1;
		}
		else
		{
			startptr = endptr;
		}
	}
	buffer[index++] = 0; // end of record

	buffer[index++] = 0; // question type
	buffer[index++] = 1;
	buffer[index++] = 0; // question class
	buffer[index++] = 1;

	result = Udp_Send(socket, _primarydns, DNS_PORT, buffer, index);
	free(buffer);

	return result;
}

static bool Dns_GetResponse(uint8_t socket, ipaddress_t* ip)
{
	bool result = FALSE;

	uint16_t failures = 1000;
    while (failures--) // timeout
    {
    	uint16_t size = Udp_GetReceivedSize(socket);
    	if (size > 0)
    	{
    		uint8_t* response = (uint8_t*)malloc(size);

    		ipaddress_t fromip;
   		   	uint16_t fromport;
    		Udp_Receive(socket, &fromip, &fromport, response, size);

    		// decode


    		uint16_t questions = ((uint16_t)(response[4]) << 8) +
    				             ((uint16_t)(response[5]));

    		uint16_t answers = ((uint16_t)(response[6]) << 8) +
    		    			   ((uint16_t)(response[7]));

    		// 12 bytes of header
    		uint16_t index = 12;

    		// read the questions
    		uint16_t question;
    		for (question = 0; question < questions; question++)
    		{
    			// skip the name
    			uint8_t size = response[index++];
    			while (size > 0)
    			{
    			    if ((size & 0xC0) == 0x0C)
    			    {
    			    	index++;
    			    	break;
    			    }

    			    index += size;
    			    size = response[index++];
    			}

    			// skip the type
				//uint16_t type = ((uint16_t)(response[index + 0]) << 8) +
				//				((uint16_t)(response[index + 1]));
				index += 2;

				// skip the class
				//uint16_t class = ((uint16_t)(response[index + 0]) << 8) +
				//				 ((uint16_t)(response[index + 1]));
				index += 2;
    		}

    		uint16_t answer;
    		for (answer = 0; answer < answers; answer++)
    		{
    			// skip the name
				uint8_t size = response[index++];
				while (size > 0)
				{
					if ((size & 0xC0) == 0xC0)
					{
						index++;
						break;
					}

					index += size;
					size = response[index++];
				}

    			// read the type
    			uint16_t type = ((uint16_t)(response[index + 0]) << 8) +
    					        ((uint16_t)(response[index + 1]));
    			index += 2;

    			/* read the class
    			uint16_t class = ((uint16_t)(response[index + 0]) << 8) +
    					         ((uint16_t)(response[index + 1]));
    			*/
    			index += 2;

    			/* read the time to live
    			uint32_t ttl = ((uint32_t)(response[index + 0]) << 24) +
    					       ((uint32_t)(response[index + 1]) << 16) +
    					       ((uint32_t)(response[index + 2]) << 8) +
    					       ((uint32_t)(response[index + 3]));
    			*/
    			index += 4;

    			// read the data length
    			uint16_t rdlength = ((uint16_t)(response[index + 0]) << 8) +
    			    			    ((uint16_t)(response[index + 1]));
    			index += 2;

    			if (type == 1)
    			{
    				(*ip)[0] = response[index + 0];
    				(*ip)[1] = response[index + 1];
    				(*ip)[2] = response[index + 2];
    				(*ip)[3] = response[index + 3];
    				result = TRUE;
    				break;
    			}

    			index += rdlength;
    		}

    		free(response);
    		break;
    	}
    }

    return result;
}
