/*
 * dhcp.c
 *
 *  Created on: May 27, 2011
 *      Author: Justin
 */

#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include "stm32f10x.h"

#include "wiznet5100.h"
#include "udp.h"
#include "dns.h"

// UDP port numbers for DHCP
#define DHCP_SERVER_PORT  67        // from server to client
#define DHCP_CLIENT_PORT  68        // from client to server

// DHCP message OP code
#define DHCP_BOOTREQUEST   1
#define DHCP_BOOTREPLY     2

// DHCP message type
#define DHCP_DISCOVER      1
#define DHCP_OFFER         2
#define DHCP_REQUEST       3
#define DHCP_DECLINE       4
#define DHCP_ACK           5
#define DHCP_NAK           6
#define DHCP_RELEASE       7
#define DHCP_INFORM        8

// MISC DHCP stuff
#define DHCP_TIMEOUT       (80000000 * 4) // 4 seconds, assumes 80Mhz clkfreq
#define DHCP_FLAGSBROADCAST 0x8000
#define DHCP_HTYPE10MB      1
#define DHCP_HLENETHERNET   6
#define MAGIC_COOKIE_0      0x63
#define MAGIC_COOKIE_1      0x82
#define MAGIC_COOKIE_2      0x53
#define MAGIC_COOKIE_3      0x63

// DHCP Options
#define OPT_padOption                0
#define OPT_subnetMask               1
#define OPT_timerOffset              2
#define OPT_routersOnSubnet          3
#define OPT_timeServer               4
#define OPT_nameServer               5
#define OPT_dns                      6
#define OPT_logServer                7
#define OPT_cookieServer             8
#define OPT_lprServer                9
#define OPT_impressServer            10
#define OPT_resourceLocationServer   11
#define OPT_hostName                 12
#define OPT_bootFileSize             13
#define OPT_meritDumpFile            14
#define OPT_domainName               15
#define OPT_swapServer               16
#define OPT_rootPath                 17
#define OPT_extentionsPath           18
#define OPT_IPforwarding             19
#define OPT_nonLocalSourceRouting    20
#define OPT_policyFilter             21
#define OPT_maxDgramReasmSize        22
#define OPT_defaultIPTTL             23
#define OPT_pathMTUagingTimeout      24
#define OPT_pathMTUplateauTable      25
#define OPT_ifMTU                    26
#define OPT_allSubnetsLocal          27
#define OPT_broadcastAddr            28
#define OPT_performMaskDiscovery     29
#define OPT_maskSupplier             30
#define OPT_performRouterDiscovery   31
#define OPT_routerSolicitationAddr   32
#define OPT_staticRoute              33
#define OPT_trailerEncapsulation     34
#define OPT_arpCacheTimeout          35
#define OPT_ethernetEncapsulation    36
#define OPT_tcpDefaultTTL            37
#define OPT_tcpKeepaliveInterval     38
#define OPT_tcpKeepaliveGarbage      39
#define OPT_nisDomainName            40
#define OPT_nisServers               41
#define OPT_ntpServers               42
#define OPT_vendorSpecificInfo       43
#define OPT_netBIOSnameServer        44
#define OPT_netBIOSdgramDistServer   45
#define OPT_netBIOSnodeType          46
#define OPT_netBIOSscope             47
#define OPT_xFontServer              48
#define OPT_xDisplayManager          49
#define OPT_dhcpRequestedIPaddr      50
#define OPT_dhcpIPaddrLeaseTime      51
#define OPT_dhcpOptionOverload       52
#define OPT_dhcpMessageType          53
#define OPT_dhcpServerIdentifier     54
#define OPT_dhcpParamRequest         55
#define OPT_dhcpMsg                  56
#define OPT_dhcpMaxMsgSize           57
#define OPT_dhcpT1value              58
#define OPT_dhcpT2value              59
#define OPT_dhcpClassIdentifier      60
#define OPT_dhcpClientIdentifier     61
#define OPT_endOption                255

// offsets for DHCP packet (UDP header is in front of this)
#define DHCP_OP      0 //8
#define DHCP_HTYPE   1 //9
#define DHCP_HLEN    2 //10
#define DHCP_HOPS    3 //11
#define DHCP_XID     4 //12
#define DHCP_SECS    8 //16
#define DHCP_FLAGS   10 //18
#define DHCP_CIADDR  12 //20
#define DHCP_YIADDR  16 //24
#define DHCP_SIADDR  20 //28
#define DHCP_GIADDR  24 //32
#define DHCP_CHADDR  28 //36
#define DHCP_SNAME   44 //52
#define DHCP_FILE    108 //116
#define DHCP_COOKIE  236 //244
#define DHCP_OPT     240 //248
#define DHCP_END     548 //556

// return codes
#define SUCCESS                0
#define SOCKET_FAILED_OPENING  1
#define SERVER_TIMEOUT         2
#define SERVER_NAK             3

#define BUFFER_SIZE  2048

#define XID 0x42424242

static ipaddress_t DHCP_BROADCAST_IP = { 255,255,255,255};

static ipaddress_t ipaddress;
static subnetmask_t subnetmask;
static ipaddress_t gatewayaddress;
static ipaddress_t dhcpserverip;
static uint32_t leasetime;
static ipaddress_t dnsserver;
static ipaddress_t timeserverip;
static macaddress_t macaddress;
static char* hostname = "DragonEgg";

static uint16_t DHCP_FillDiscoverPacket(uint32_t xid, uint8_t* buffer);
static uint16_t DHCP_FillRequestPacket(uint32_t xid, uint8_t* buffer);
static uint8_t DHCP_ParseReply(uint32_t xid, uint8_t expectedType, uint8_t* packetBuffer, uint16_t packetSize);

extern bool Dhcp_Start()
{
	WIZnet5100_GetMacAddress(&macaddress);

	uint8_t socket;
	if (!Udp_OpenAvailableSocket(DHCP_CLIENT_PORT, &socket))
	{
		return FALSE;
	}

	uint8_t *packet;
	ipaddress_t fromip;
	uint16_t fromport;

	uint8_t buffer[DHCP_END];
	DHCP_FillDiscoverPacket(XID, buffer);

	Udp_Send(socket, DHCP_BROADCAST_IP, DHCP_SERVER_PORT, buffer, DHCP_END);

	uint16_t failures = 1000;
	while(failures--) // not timeout
	{
		uint16_t size = Udp_GetReceivedSize(socket);
		if (size > 0)
		{
			packet = (uint8_t*)malloc(size);
			Udp_Receive(socket, &fromip, &fromport, packet, size);

			switch (DHCP_ParseReply(XID, DHCP_OFFER, packet, size))
			{
			case 1:
				free(packet);

				// claim the offered IP
				DHCP_FillRequestPacket(XID, buffer);

				Udp_Send(socket, DHCP_BROADCAST_IP, DHCP_SERVER_PORT, buffer, DHCP_END);

				uint16_t replyfailures = 1000;
				while(replyfailures--) // not timeout
				{
					uint16_t size = Udp_GetReceivedSize(socket);
					if (size > 0)
					{
						packet = (uint8_t*)malloc(size);
						Udp_Receive(socket, &fromip, &fromport, packet, size);

						switch (DHCP_ParseReply(XID, DHCP_ACK, packet, size))
						{
						case 1:
							free(packet);

							// got reply
							WIZnet5100_SetGatewayAddress(gatewayaddress);
							WIZnet5100_SetSubnetMask(subnetmask);
							WIZnet5100_SetIpAddress(ipaddress);
							Dns_SetPrimary(dnsserver);
							Udp_Close(socket);
							return TRUE;

						case 2:
							free(packet);

							// NAK
							Udp_Close(socket);
							return FALSE;
						}
					}
				}

				Udp_Close(socket);
				return FALSE;

			case 2:
				free(packet);

				// NAK
				Udp_Close(socket);
				return FALSE;
			}
		}
	}

	Udp_Close(socket);
	return FALSE;
}

static uint16_t DHCP_FillBasePacket(uint32_t xid, uint8_t messageType, uint8_t* buffer) //| index, optionOffset, hostNameLength
{
  uint16_t index;
  //BYTEFILL(@byte[packetBuffer][DHCP_OP], 0, DHCP_END-UDP_HEADER_SIZE);
  memset((void*)buffer, 0, DHCP_END);
  buffer[DHCP_OP] = DHCP_BOOTREQUEST;
  buffer[DHCP_HTYPE] = DHCP_HTYPE10MB;
  buffer[DHCP_HLEN] = DHCP_HLENETHERNET;
  buffer[DHCP_XID+0] = (uint8_t)((xid & 0x000000FF) >> 0);
  buffer[DHCP_XID+1] = (uint8_t)((xid & 0x0000FF00) >> 8);
  buffer[DHCP_XID+2] = (uint8_t)((xid & 0x00FF0000) >> 16);
  buffer[DHCP_XID+3] = (uint8_t)((xid & 0xFF000000) >> 24);
  buffer[DHCP_FLAGS+0] = DHCP_FLAGSBROADCAST>>8; //                   'high-order byte8
  buffer[DHCP_FLAGS+1] = (DHCP_FLAGSBROADCAST & 0x00FF); //             'low-order byte

  //repeat index from 0 to 5
  for (index = 0; index <= 5; index++)
  {
      buffer[DHCP_CHADDR+index] = macaddress[index];
  }

  buffer[DHCP_COOKIE+0] = MAGIC_COOKIE_0;
  buffer[DHCP_COOKIE+1] = MAGIC_COOKIE_1;
  buffer[DHCP_COOKIE+2] = MAGIC_COOKIE_2;
  buffer[DHCP_COOKIE+3] = MAGIC_COOKIE_3;

  uint16_t optionOffset = 0;
  buffer[DHCP_OPT+optionOffset++] = OPT_dhcpMessageType;
  buffer[DHCP_OPT+optionOffset++] = 1;
  buffer[DHCP_OPT+optionOffset++] = messageType;

  buffer[DHCP_OPT+optionOffset++] = OPT_dhcpClientIdentifier;
  buffer[DHCP_OPT+optionOffset++] = 7;
  buffer[DHCP_OPT+optionOffset++] = 1;
  //repeat index from 0 to 5
  for (index = 0; index <= 5; index++)
  {
      buffer[DHCP_OPT+optionOffset++] = macaddress[index];
  }

  buffer[DHCP_OPT+optionOffset++] = OPT_hostName;
  uint16_t hostNameLength = strlen(hostname);
  //' force to be even, so next option after this will be word aligned (16bit)
  if ((hostNameLength & 1) == 1)
  {
    hostNameLength += 1;
  }
  buffer[DHCP_OPT+optionOffset++] = hostNameLength;
  //repeat index from 0 to hostNameLength-1
  for (index = 0; index < hostNameLength; index++)
  {
    buffer[DHCP_OPT+optionOffset++] = hostname[index];
  }

  return optionOffset;
}

// called to request ip address
static uint16_t DHCP_FillDiscoverPacket(uint32_t xid, uint8_t* buffer)
{
  uint16_t optionOffset = DHCP_FillBasePacket(xid, DHCP_DISCOVER, buffer);

  buffer[DHCP_OPT+optionOffset++] = OPT_dhcpParamRequest;
  buffer[DHCP_OPT+optionOffset++] = 6;
  buffer[DHCP_OPT+optionOffset++] = OPT_subnetMask;
  buffer[DHCP_OPT+optionOffset++] = OPT_domainName;
  buffer[DHCP_OPT+optionOffset++] = OPT_dns;
  buffer[DHCP_OPT+optionOffset++] = OPT_routersOnSubnet;
  buffer[DHCP_OPT+optionOffset++] = OPT_dhcpT1value;
  buffer[DHCP_OPT+optionOffset++] = OPT_dhcpT2value;
  //buffer[DHCP_OPT+optionOffset++] = OPT_timeServer;

  buffer[DHCP_OPT+optionOffset++] = OPT_endOption;

  return optionOffset;
}

// called to accept offered ip address
static uint16_t DHCP_FillRequestPacket(uint32_t xid, uint8_t* buffer)
{
  uint16_t optionOffset = DHCP_FillBasePacket(xid, DHCP_REQUEST, buffer);

  buffer[DHCP_OPT+optionOffset++] = OPT_dhcpRequestedIPaddr;
  buffer[DHCP_OPT+optionOffset++] = 4;
  buffer[DHCP_OPT+optionOffset++] = ipaddress[0];
  buffer[DHCP_OPT+optionOffset++] = ipaddress[1];
  buffer[DHCP_OPT+optionOffset++] = ipaddress[2];
  buffer[DHCP_OPT+optionOffset++] = ipaddress[3];

  buffer[DHCP_OPT+optionOffset++] = OPT_dhcpServerIdentifier;
  buffer[DHCP_OPT+optionOffset++] = 4;
  buffer[DHCP_OPT+optionOffset++] = dhcpserverip[0];
  buffer[DHCP_OPT+optionOffset++] = dhcpserverip[1];
  buffer[DHCP_OPT+optionOffset++] = dhcpserverip[2];
  buffer[DHCP_OPT+optionOffset++] = dhcpserverip[3];

  buffer[DHCP_OPT+optionOffset++] = OPT_dhcpParamRequest;
  buffer[DHCP_OPT+optionOffset++] = 8;
  buffer[DHCP_OPT+optionOffset++] = OPT_subnetMask;
  buffer[DHCP_OPT+optionOffset++] = OPT_routersOnSubnet;
  buffer[DHCP_OPT+optionOffset++] = OPT_dns;
  buffer[DHCP_OPT+optionOffset++] = OPT_domainName;
  buffer[DHCP_OPT+optionOffset++] = OPT_dhcpT1value;
  buffer[DHCP_OPT+optionOffset++] = OPT_dhcpT2value;
  buffer[DHCP_OPT+optionOffset++] = OPT_performRouterDiscovery;
  buffer[DHCP_OPT+optionOffset++] = OPT_staticRoute;

  buffer[DHCP_OPT+optionOffset++] = OPT_endOption;

  return optionOffset;
}

static uint8_t DHCP_ParseReply(uint32_t xid, uint8_t expectedType, uint8_t* packetBuffer, uint16_t payloadSize) //| index, optionOffset, option, optionLength, port, payloadSize
{
	uint16_t index;
  //uint16_t port = packetBuffer[UDP_HEADER_PORT] * 256 + packetBuffer[UDP_HEADER_PORT+1];
  //uint16_t payloadsize = packetBuffer[UDP_HEADER_PAYLOADSIZE] * 256 +  packetBuffer[UDP_HEADER_PAYLOADSIZE+1];

  // verify the packet is from the DHCP server and is at least of minimum size
  //if (port != DHCP_SERVER_PORT || payloadsize < DHCP_OPT)
  //{
  //  return 0; // ' this is not a DHCP packet
  //}
	if (payloadSize < DHCP_OPT)
	{
		return 0;
	}

  // verify that the packet is a DHCP Reply
  if (packetBuffer[DHCP_OP] != DHCP_BOOTREPLY)
  {
    return 0; // this is not a DHCP Reply
  }

  // verify XID match
  if (xid != (packetBuffer[DHCP_XID+3] << 24) +
		     (packetBuffer[DHCP_XID+2] << 16) +
		     (packetBuffer[DHCP_XID+1] << 8) +
		     (packetBuffer[DHCP_XID+0]))
  {
	return 0;
  }

  // verify MAC Address match
  //repeat index from 0 to 5
  for (index = 0; index <= 5; index++)
  {
    if (macaddress[index] != packetBuffer[DHCP_CHADDR+index])
    {
      return 0; // does not match, so not for us
    }
  }

  // save the IP that is offered/claimed
  ipaddress[0] = packetBuffer[DHCP_YIADDR+0];
  ipaddress[1] = packetBuffer[DHCP_YIADDR+1];
  ipaddress[2] = packetBuffer[DHCP_YIADDR+2];
  ipaddress[3] = packetBuffer[DHCP_YIADDR+3];

  leasetime = 0;

  // scan through options pulling out stuff we need
  uint16_t optionOffset = 0;
  //repeat

  bool done = FALSE;
  while (!done)
  {
    uint8_t option = packetBuffer[DHCP_OPT+optionOffset++];
    uint16_t optionLength = 0;
    if (option != 0)
    {
      optionLength = packetBuffer[DHCP_OPT+optionOffset++];
    }

    switch (option)
    {
    	case OPT_endOption:
    		done = TRUE;
            break;

    	case OPT_dhcpMessageType:
        if (packetBuffer[DHCP_OPT+optionOffset] != expectedType)
        {
          if (packetBuffer[DHCP_OPT+optionOffset] == DHCP_NAK)
          {
            return 2;
          }
          else
          {
            return 0;
          }
        }
        break;

    	case OPT_subnetMask:
            subnetmask[0] = packetBuffer[DHCP_OPT+optionOffset+0];
            subnetmask[1] = packetBuffer[DHCP_OPT+optionOffset+1];
            subnetmask[2] = packetBuffer[DHCP_OPT+optionOffset+2];
            subnetmask[3] = packetBuffer[DHCP_OPT+optionOffset+3];
            break;

    	case OPT_routersOnSubnet:
            gatewayaddress[0] = packetBuffer[DHCP_OPT+optionOffset+0];
            gatewayaddress[1] = packetBuffer[DHCP_OPT+optionOffset+1];
            gatewayaddress[2] = packetBuffer[DHCP_OPT+optionOffset+2];
            gatewayaddress[3] = packetBuffer[DHCP_OPT+optionOffset+3];
            break;

    	case OPT_dhcpIPaddrLeaseTime:
            leasetime = (packetBuffer[DHCP_OPT+optionOffset+3] << 24) +
                        (packetBuffer[DHCP_OPT+optionOffset+2] << 16) +
                        (packetBuffer[DHCP_OPT+optionOffset+1] << 8) +
                        (packetBuffer[DHCP_OPT+optionOffset+0]);
            break;
    	case OPT_dns:
            dnsserver[0] = packetBuffer[DHCP_OPT+optionOffset+0];
            dnsserver[1] = packetBuffer[DHCP_OPT+optionOffset+1];
            dnsserver[2] = packetBuffer[DHCP_OPT+optionOffset+2];
            dnsserver[3] = packetBuffer[DHCP_OPT+optionOffset+3];
            break;

    	case OPT_dhcpServerIdentifier:
            dhcpserverip[0] = packetBuffer[DHCP_OPT+optionOffset+0];
            dhcpserverip[1] = packetBuffer[DHCP_OPT+optionOffset+1];
            dhcpserverip[2] = packetBuffer[DHCP_OPT+optionOffset+2];
            dhcpserverip[3] = packetBuffer[DHCP_OPT+optionOffset+3];
            break;

    	case OPT_timeServer:
    		timeserverip[0] = packetBuffer[DHCP_OPT+optionOffset+0];
    		timeserverip[1] = packetBuffer[DHCP_OPT+optionOffset+1];
    		timeserverip[2] = packetBuffer[DHCP_OPT+optionOffset+2];
    		timeserverip[3] = packetBuffer[DHCP_OPT+optionOffset+3];
    		break;

    	case OPT_domainName:
        //BYTEMOVE(@DomainName[0], @byte[packetBuffer][DHCP_OPT+optionOffset], optionLength)
        break;
    }

        optionOffset += optionLength;
    }

  return 1;
}
