#include <EEPROM.h>
#include <avr/sleep.h>
// Front yellow light powered by Li-ion
// Modes: 2 Hz, 5 Hz, 10 Hz, fade, single flash, triple flash (day), triple flash (night), solid
// Buttons: power/mode, pattern/brightness

byte modeNumber;
const byte modeTop = 2;
const byte patternTop = 7;
const byte brightnessTop = 5;

// EEPROM memory addresses
int modeAddr = 0;
int yellowPatternAddrs[] = {
  1, 2, 3};
int yellowIntensityAddrs[] = {
  4, 5, 6};

// Stores the patterns and brightness levels for each mode.
byte yellowPatterns[6];
byte yellowIntensities[6];

// LED driver parameters
int yellowHighV;
int yellowLowV;
int yellowDimV;
int yellowVdesired;
int yellowReading;
int yellowOnTime;
const int maxOnTime = 255;

// Strobe patterns
int yellowHiTime;
int yellowLoTime;
int yellowBlinkCt;
int yellowPulseReps;
int yellowDelayTime;
int yellowStepTime;
int yellowStep;
const int yellowStepTop = 19;
int yellowFadeLevelsMax[20] = {
  0, 37, 73, 110, 147, 183, 220, 257, 293, 330, 330, 293, 257, 220, 183, 147, 110, 73, 37, 0};
int yellowFadeLevels[20];
unsigned long lastYellowToggleTime;
boolean yellowState;

int debounceDelay = 250;

//ATTiny85 pins
const byte battPin = A3;
const byte yellowIPin = A2;
const byte powerButtonPin = 2;
const byte yellowPWMPin = 1;
const byte yellowButtonPin = 0;

// Overdischarge protection parameters.
int batteryReading;
const int batteryCutoffADC = 592; // Selected for minimum supply voltage of 12V with R1 = 1M, and R2 = 56k.

void setup() {

  pinMode(powerButtonPin, INPUT_PULLUP);
  pinMode(yellowButtonPin, INPUT_PULLUP);
  pinMode(yellowIPin, INPUT);
  pinMode(battPin, INPUT);

  // Loads the data stored in EEPROM.
  modeNumber = EEPROM.read(modeAddr);

  for (int i=0;i<modeTop;i++){
    yellowPatterns[i] = EEPROM.read(yellowPatternAddrs[i]);
    yellowIntensities[i] = EEPROM.read(yellowIntensityAddrs[i]);
  }

  analogReference(INTERNAL);

  sleepNow();
  updateSettings();

}

void loop() {

  batteryReading = analogRead(battPin);

  // If battery voltage falls below threshold, turn off PWM pin to prevent overdischarging. (For unprotected Li-ion batteries).
  if (batteryReading < batteryCutoffADC) {
    pinMode(yellowPWMPin, INPUT);
  }

  readButtons();
  yellowStrobe();
  yellowDriver();
}

void yellowStrobe() {


  if (yellowPatterns[modeNumber] >= 0 && yellowPatterns[modeNumber] < 3) { // Pulse.

    if ((millis() - lastYellowToggleTime > yellowHiTime) && yellowState == HIGH) {
      yellowState = LOW;
      lastYellowToggleTime = millis();
      yellowVdesired = yellowLowV;
    }
    else if ((millis() - lastYellowToggleTime > yellowLoTime) && yellowState == LOW) {
      yellowState = HIGH;
      lastYellowToggleTime = millis();
      yellowVdesired = yellowHighV;
    }
  }
  else if (yellowPatterns[modeNumber] == 3) { // Fade.

    if (millis() - lastYellowToggleTime > yellowStepTime) { // If step time expires, increment brightness step.
      lastYellowToggleTime = millis();
      if (yellowStep < yellowStepTop) {
        yellowStep++;
      }
      else {
        yellowStep = 0;
      }
      yellowVdesired = yellowFadeLevels[yellowStep];
    }
  }
  else if (yellowPatterns[modeNumber] > 3 && yellowPatterns[modeNumber] < 7) { // n blinks followed by a pause.

    if (yellowBlinkCt < yellowPulseReps) { // Blink lights n times.
      if ((millis() - lastYellowToggleTime > yellowHiTime) && yellowState == HIGH) {
        yellowState = LOW;
        lastYellowToggleTime = millis();
        yellowVdesired = yellowLowV;
        yellowBlinkCt++;

        if(yellowBlinkCt>=yellowPulseReps){
          yellowVdesired = yellowDimV;
        }
      }
      else if ((millis() - lastYellowToggleTime > yellowLoTime) && yellowState == LOW) {
        yellowState = HIGH;
        lastYellowToggleTime = millis();
        yellowVdesired  = yellowHighV;
      }
    }

    else if (yellowBlinkCt >= yellowPulseReps) { // If pulses are over, pause.
      if ((millis() - lastYellowToggleTime > yellowDelayTime) && yellowState == LOW) {
        lastYellowToggleTime = millis();
        yellowState = HIGH;
        yellowVdesired = yellowHighV;
        yellowBlinkCt = 0;
      }
    }
  }
  else if (yellowPatterns[modeNumber] == 7) { // Solid mode.
    yellowVdesired = yellowHighV;
  }
}

