/*
 *  main.c
 *  PhotoInterrupter
 *
 *  Created by Jens Willy Johannsen on 2010-12.
 *  Copyright 2010. All rights reserved.
 *
 */

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/eeprom.h>
#include <util/delay.h>
#include "Max7219.h"

// Prototypes
void updateDisplay();

// Pin definitions
#define PIN_LOAD	PB4		// PB4 -> Max7219 LOAD pin
#define PORT_LOAD	PORTB
#define PIN_LED		PD6		// PD6 (pin 11) -> debugging LED
#define PORT_LED	PORTD
#define PIN_SWITCH	PD3		// PD3 (pin 7) -> input for program/reset

//wyjscie na DP dla "licznika obrotow"
//#define PIN_DP0		PB0		// PB0 (pin 12) - kropka jednosci
//#define PORT_DP0	PORTPB0
//#define PIN_DP1		PB1		// PB1 (pin 13) - kropka jednosci
//#define PORT_DP1	PORTPB1		


#define TIMER_COMPARE_VALUE 31249	// compare value for 1 Hz: val = F_CPU/prescaler/target freq. - 1: 8000000/256/1 - 1

// Global vars
uint16_t EEMEM eepromCapacity;	// EEPROM var
uint16_t capacity;
int roundsRemaining = 79;
int obroty = 0; // to bedzie odpowiedzialne za zapalanie kropki przy przekreceniu licznika
volatile char fastforward;

// Interrupt handler for INT0 interrupts.
// This interrupt will trigger twice for each interruption of the beam: once when going from 0 (beam uninterrupted) to 1 (beam interrupted)
// and once when going back from 1 to 0. 
// When going high, the LED will light up; when going low, the counter will decrease and the display will be updated.
ISR( INT0_vect )
{
	// What's the current state of INT0?
	int val = (PIND & _BV( PD2 ));
	
	if( val )
	{
		// High (=beam blocked)
		PORT_LED |= _BV( PIN_LED );	// LED on
	}
	else 
	{
		// Low (=beam no longer blocked)
		PORT_LED &= ~_BV( PIN_LED );	// LED off
		
		// Decrease counter
		roundsRemaining--;
		
		// Update display
		updateDisplay();
	}
}

/////////////////kropka
//	if( roundsRemaining >= 100 )
//	{
//		PORT_DP0 = 0x01;
//	}
//	else
//	{
//		PORT_DP0 = 0x00;
//	}


// Interrupt handler for INT1 interrupts.
// This interrupt will trigger when INT1 goes low  i.e. the switch is pressed since the pin's pull-up is active
ISR( INT1_vect )
{
	roundsRemaining = capacity;
	updateDisplay();
}

// Interrupt handler for timer 1 compare match.
// This handler is used in setup mode to indicate when the button has been held down for 1 second
ISR( TIMER1_COMPA_vect )
{
	// Set fastforward
    fastforward = 1;
	
	PORT_LED |= _BV( PIN_LED );
}

// Utility method to shift 8 bits out the USI pins in 3-wire (SPI) mode
static unsigned char shft8bit(unsigned char val) 
{ 
	USIDR = val;			// Set data byte to send
	USISR = (1<<USIOIF);	// Clear Counter Overflow Interrupt Flag (by writing 1 to the USIOIF bit)
	do { 
		USICR = (1<<USIWM0)|(1<<USICS1)|(1<<USICLK)|(1<<USITC);	// 3-wire mode; shift reg. clock source: external pos. edge; 4-bit counter source: USITC; Toggle USICK pin 0->1 or 1->0 (shift register will only clock on 0->1)
	} while ((USISR & (1<<USIOIF)) == 0);						// Check for OIF set which will happen after 16 cycles. I.e. 8 bits since the clock toggles twice per bit.
	return USIDR;			// Return data value
}

void sendToMax7219( char reg, char value )
{
	PORT_LOAD &= ~_BV( PIN_LOAD );	// LOAD low
	shft8bit( reg );
	shft8bit( value );
	PORT_LOAD |= _BV( PIN_LOAD );	// LOAD high
}

void updateDisplay()
{
	if( roundsRemaining < 0 ) 
	{
		// There's less than 0: show ----
		sendToMax7219( max7219_reg_decodeMode, 0x00 );	// no decode 
		sendToMax7219( max7219_reg_digit0, 0x01 );
		sendToMax7219( max7219_reg_digit1, 0x01 );
		sendToMax7219( max7219_reg_digit2, 0x01 );
	//	sendToMax7219( max7219_reg_digit3, 0x01 );
	}
	else 
	{
		// 0 or more: show rounds remaining
		int hundreds = roundsRemaining / 100;
		int tens = (roundsRemaining % 100) / 10;
		int ones = roundsRemaining % 10;
		
		sendToMax7219( max7219_reg_decodeMode, 0x0F );	// decode digits 0-3
		sendToMax7219( max7219_reg_digit0, ones );
		sendToMax7219( max7219_reg_digit1, tens );
		sendToMax7219( max7219_reg_digit2, hundreds );
	//	sendToMax7219( max7219_reg_digit3, 0x0F );	// Blank digit 3
	}
}

void showNumber( int number )
{
	// Show current capacity
	int hundreds = number / 100;
	int tens = (number % 100) / 10;
	int ones = number % 10;
	
	sendToMax7219( max7219_reg_decodeMode, 0x0F );	// decode digits 0-3
	sendToMax7219( max7219_reg_digit0, ones );
	sendToMax7219( max7219_reg_digit1, tens );
	sendToMax7219( max7219_reg_digit2, hundreds );
	//sendToMax7219( max7219_reg_digit3, 0x0F );	// Blank digit 3
}

void setup()
{
	// Flash Pr on display
	/*
	 P	A,B,G,E,F	B.0110.0111	0x67
	 r	E,G			B.0000.0101	0x05
	 G	A,B,C,D,F,G	B.0111.1011	0x7B
	*/
	sendToMax7219( max7219_reg_decodeMode, 0 );	// No decode
//	sendToMax7219( max7219_reg_digit3, 0x00 );
//	sendToMax7219( max7219_reg_digit2, 0x67 );
	sendToMax7219( max7219_reg_digit1, 0x67 ); //P
	sendToMax7219( max7219_reg_digit0, 0x05 ); //r
	
	// Wait a second
	_delay_ms( 1000 );
	
	// Wait for RST/PRG to be released
	while( (PIND & _BV( PIN_SWITCH )) == 0 )
		;
	
	// Initialize capacity to 0
	capacity = 0;

	// Show current capacity
	showNumber( capacity );
	
	// Configure fastforward timer
	fastforward = 0;
	TCCR1A = _BV( COM1A0 );			// toggle OC1A on compare match
	OCR1A = TIMER_COMPARE_VALUE;	// compare value for 1 Hz: val = F_CPU/prescaler/target freq. - 1: 8000000/256/1 - 1
	sei();							// enable interrupts
	TIMSK |= _BV( OCIE1A );			// enable interrupt for timer1 output compare A match
	TCCR1B = 0;						// timer stopped
	
	// Loop
	for( ;; )
	{
		// If RST/PRG is pressed, increase capacity by 1
		if( fastforward || (PIND & _BV( PIN_SWITCH )) == 0 )
		{
			// Debounce 10 ms
			_delay_ms( 10 );
			if( fastforward || (PIND & _BV( PIN_SWITCH )) == 0 )
			{
				// Debounce is good: increase capacity
				capacity++;
			
				// If more than 199, go back to 0
				if( capacity > 199 )
					capacity = 0;
				
				// Update display
				showNumber( capacity );
				
				// Write to EEPROM
				eeprom_write_word( &eepromCapacity, capacity );
				
				// If we're not in fastforward mode, start the 1 second timer now
				if( !fastforward )
					TCCR1B = _BV( WGM12 ) | _BV( CS12 ); // start timer1: waveform generation mode = 4 (TOP=OCR1A), clock source = system clock w/ prescaler 256
					
				// Wait until btn up or fastforward time has passed
				while( !fastforward && (PIND & _BV( PIN_SWITCH )) == 0 )
					;
				
				// If btn is up, reset fastforward, reset timer 1 counter and stop timer
				if( (PIND & _BV( PIN_SWITCH )) != 0 )
				{
					fastforward = 0;
					TCCR1B = 0;
					TCNT1 = 0;
				}
			}
		}
	}
}

int main(void)
{
	/*------------------*/
	/*	CONFIGURATION	*/
	/*------------------*/

	// Configure hardware
	DDRD = _BV( PIN_LED );				// All port D -> INPUT except PD6
	DDRB |= _BV( PB0 ) | _BV( PIN_LOAD ) | _BV( PB7 ) | _BV( PB6 );		// PB0, PB4, DO, UCSK -> OUTPUT
	PORTD |= _BV( PIN_SWITCH );			// Pull-up on INT1/PD3
	
	// Read magazine capacity from EEPROM
	capacity = eeprom_read_word( &eepromCapacity );
	if( capacity == 0xFFFF )	// If uninitialized
	{
		capacity = 180;			// Initialize to 180
		eeprom_write_word( &eepromCapacity, capacity );	// And store it
	}
	roundsRemaining = capacity;
	
	//if( roundsRemaining > 100 )
	//{
//		PORTPB0 = 0x01;
//	}
//	else
//	{
//		PORTPB0 = 0x00;
//	}




	// Init 7219
	sendToMax7219( max7219_reg_scanLimit, 0x07 );	// scan all 8 digits
	sendToMax7219( max7219_reg_decodeMode, 0 );	// No decode
	//sendToMax7219( max7219_reg_digit3, 0x00 );
//	sendToMax7219( max7219_reg_digit2, 0x00 );
	sendToMax7219( max7219_reg_digit1, 0x00 );
	sendToMax7219( max7219_reg_digit0, 0x00 );
	sendToMax7219( max7219_reg_displayTest, 0x00 );	// display test off
	sendToMax7219( max7219_reg_intensity, 0x02 );	// intensity 0-15
	sendToMax7219( max7219_reg_shutdown, 0x01 );	// shutdown off
	
	// Flash LED twice
	PORT_LED |= _BV( PIN_LED );
	_delay_ms( 100 );
	PORT_LED &= ~_BV( PIN_LED );
	_delay_ms( 100 );
	PORT_LED |= _BV( PIN_LED );
	_delay_ms( 100 );
	PORT_LED &= ~_BV( PIN_LED );
	_delay_ms( 100 );
	
	updateDisplay();
	
	// Mode check: run or program?
	if( (PIND & _BV( PIN_SWITCH )) == 0 )
	{
		// PRG/RST switch pressed while powering up: go to setup mode
		setup();	// NOTE: this method will never return
	}
	
	// Run mode: show run
	/*
	 r	E,G				0000.0101	0x05
	 u	C,D,E			0001.1100	0x1C
	 n	C,E,G			0001.0101	0x15

	 9	A,B,C,D,F,G		0111.1011	0x7B
	 5	A,C,D,F,G		0101.1011	0x5B
	 
	*/
	sendToMax7219( max7219_reg_decodeMode, 0 );	// No decode
//	sendToMax7219( max7219_reg_digit3, 0x00 );
//	sendToMax7219( max7219_reg_digit2, 0x05 );	// r
	sendToMax7219( max7219_reg_digit1, 0x7B );	// 9
	sendToMax7219( max7219_reg_digit0, 0x5B );	// 5
	
	// Wait a second
	_delay_ms( 1000 );
	
	updateDisplay();

	/*------------------*/
	/*	BEGIN RUN MODE	*/
	/*------------------*/
	
	// Configure interrupts for INT0
	MCUCR |= _BV( ISC00 );		// Interrupt on any logic change of INT0
	GIMSK = _BV( INT0 );		// Enable INT0 interrupts
	
	// Configure interrupts for INT1
	MCUCR |= _BV( ISC11 );		// Interrupt on falling edge of INT1
	GIMSK |= _BV( INT1 );		// Enable INT1 interrupts
	
	sei();						// Global interrupts enable
	
	// Main loop
	for( ;; )
    {
		// Do nothing  everything is handled in interrupts
    }
    return 0;
}
