
/* ATTiny84 sketch for 78.125 KHz LED driver.
   External oscillator: 20 MHz
   Code contains settings for brightness levels.
   Can enable or disable battery monitor.
   Clicking the power button changes the brightness of the LEDs. Holding and releasing it turns it on or off.
*/

#include <avr/sleep.h>

// Utility macros
#define adc_disable() (ADCSRA &= ~(1<<ADEN)) // disable ADC
#define adc_enable()  (ADCSRA |=  (1<<ADEN)) // re-enable ADC

int onTime = 0;
const int period = 255; // Determines the frequency and resolution.
const int maxOnTime = period; // Keep it equal to period for buck driver.

const int brightnessTop = 4;
const int buckPwmPin = 5;
const int powerButtonPin = 8;
const int feedbackPin = A4;
const int batteryMonitorPin = A0;
const int batteryStatusLEDPin = 6;

int batteryMonitorReading;
int feedbackReading;
int desiredFeedback;
int desiredFeedbackLowBatt;
int desiredFeedbackSelected;
boolean buttonReading;

boolean lowBattery = false;

int brightness;
const int debounceDelay = 300; // Compensated for prescalar selection and the change of clock speed.
int delayCounts = 0;

int deadBatteryADC = 544; // 10.5V + 5% or 11.0V to account for the tolerance of the components.
/*int chargeReminderADC = 603; // 12.24V
  int manualSelectADC = 742; // Manual or 14V*/

unsigned long elapsedMillis = 0;
unsigned long prevMillis = 0;

unsigned long warmUpElapsedMicros = 0;
unsigned long warmUpPrevMicros = 0;

void setup() {
  pinMode(buckPwmPin, INPUT);
  pinMode(powerButtonPin, INPUT_PULLUP);
  pinMode(feedbackPin, INPUT);
  pinMode(batteryMonitorPin, INPUT);
  pinMode(batteryStatusLEDPin, OUTPUT);
  TCCR1A = _BV(COM1B1) | _BV(WGM11);
  TCCR1B = _BV(WGM13) | _BV(WGM12) | _BV(CS10);
  ICR1 = period;
  OCR1B = 0; // Start with minimum on time.
  analogReference(INTERNAL);
  sleepNow();
}

void loop() {
  // put your main code here, to run repeatedly:
  buttonReading = digitalRead(powerButtonPin);
  feedbackReading = analogRead(feedbackPin);

  if (buttonReading == LOW) { // Pressing the button cycles through the drive current levels.
    delayCounts = 0;
    while (buttonReading == LOW) {
      delay(debounceDelay);
      delayCounts++;
      buttonReading = digitalRead(powerButtonPin);
    }
    // Cycles through the brightness levels.
    if (delayCounts <= 1) {
      if (brightness < brightnessTop) {
        brightness++;
      }
      else if (brightness >= brightnessTop) {
        brightness = 0;
      }
    }
    // Go to sleep.
    else if (delayCounts > 1) {

      // Gradually decrease the drive current.
      do {
        onTime--;
        OCR1B = 0;
        delayMicroseconds(1000);
      }
      while (onTime > 0);
      sleepNow();
    }
  }

  // Determines the selected drive current for Rsense of 0.27 ohm.
  if (brightness == 0) {
    desiredFeedback = 0;
  }
  else if (brightness == 1) {
    desiredFeedback = 25; // 100mA
  }
  else if (brightness == 2) {
    desiredFeedback = 85; // 330mA
  }
  else if (brightness == 3) {
    desiredFeedback = 255; // 1A
  }
  else if (brightness == 4) {
    desiredFeedback = 575; // 2.25A
  }

  // Gradually increase the drive current.
  if (desiredFeedbackSelected != desiredFeedback) {
    warmUpElapsedMicros = micros() - warmUpPrevMicros;
    if (warmUpElapsedMicros > 200) {
      warmUpPrevMicros = micros();

      if (desiredFeedbackSelected < desiredFeedback) {
        desiredFeedbackSelected++;
      }
      else if (desiredFeedbackSelected > desiredFeedback && desiredFeedbackSelected > 0) {
        desiredFeedbackSelected--;
      }
    }
  }

  // Keeps the drive current stable.
  if (feedbackReading < desiredFeedbackSelected && onTime < maxOnTime) {
    onTime++;
  }
  else if (feedbackReading > desiredFeedbackSelected && onTime > 0) {
    onTime--;
  }
  OCR1B = onTime;

  batteryMonitorReading = analogRead(batteryMonitorPin);

  if (batteryMonitorReading < deadBatteryADC) { // Battery low. Indicator blinks. Please recharge the battery.
    brightness == 0; // Set the brightness to the minimum.

    // Gradually decrease the drive current.
    do {
      onTime--;
      OCR1B = 0;
      delayMicroseconds(1000);
    } while (onTime > 0);
    digitalWrite(batteryStatusLEDPin, HIGH);
    delay(1000);
    sleepNow();
  }
}

/*void blinkBatteryIndicator() {
  elapsedMillis = millis() - prevMillis;
  if (elapsedMillis > 1000) {
    prevMillis = millis();
    digitalWrite(batteryStatusLEDPin, LOW);
  }
  else if (elapsedMillis > 500) {
    digitalWrite(batteryStatusLEDPin, HIGH);
  }
}*/
