﻿/************************************************************
 Custom Wakeword Engine - Allan Murray 2017
*************************************************************/
using System;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using LattePanda.Firmata;

using Microsoft.Speech.Recognition;
using System.Diagnostics;


namespace HelloSpeech
{


    class Program
    {

        const int PIN_D_BUTTON = 9;//D9
        const int PIN_D_WHITELED = 10;//D10
        const int PIN_D_BLUELED = 11;//D11
        const int PIN_D_ACTIVE = 19;//D19=A1

        const byte DISCONNECT = 1;               // OUT : Ask the AVS client to disconnect from us
        const byte WAKE_WORD_DETECTED = 2;       // OUT : sent to AVS client when a wake word is detected
        const byte PAUSE_WAKE_WORD_ENGINE = 3;   // IN : request to pause the engine and yield the Mic
        const byte RESUME_WAKE_WORD_ENGINE = 4;  // IN : request to resume the engine
        const byte CONFIRM = 5;                  // OUT : sent to AVS client to confirm the engine has stopped
        const byte UNKNOWN = 6;


        static DateTime start = new DateTime();
        static DateTime stop = new DateTime();
        static bool timerOn = false;
        static bool doShutdown = false;

        static bool isConnected = false;
        static bool isPaused = false;
        static bool speechOK = false;
        static bool arduinoOK = false;

        static private System.Net.Sockets.Socket clientSocket = null;
        static private byte[] m_byBuff = new byte[256];    // Recieved data buffer

        static Arduino arduino = null;

       

        static public void processCommand(byte command)
        {
            Console.WriteLine(command);

            switch (command)
            {
                case PAUSE_WAKE_WORD_ENGINE:
                    Console.WriteLine("PAUSED");
                    arduino.digitalWrite(PIN_D_BLUELED, Arduino.HIGH);
                    arduino.digitalWrite(PIN_D_WHITELED, Arduino.LOW);
                    arduino.digitalWrite(PIN_D_ACTIVE, Arduino.HIGH);
                    isPaused = true;
                    break;

                case CONFIRM:
                    Console.WriteLine("SPEECH RESPONSE CONFIRMED");
                    timerOn = false; //Alexa has responded - dont do a timeout
                    break;

                case RESUME_WAKE_WORD_ENGINE:
                    Console.WriteLine("UNPAUSED");
                    arduino.digitalWrite(PIN_D_BLUELED, Arduino.LOW);
                    if (!isPaused)
                    {
                        arduino.digitalWrite(PIN_D_WHITELED, Arduino.LOW);
                        arduino.digitalWrite(PIN_D_ACTIVE, Arduino.LOW);
                        timerOn = false;
                    }
                    else
                    {
                        arduino.digitalWrite(PIN_D_WHITELED, Arduino.HIGH);
                        arduino.digitalWrite(PIN_D_ACTIVE, Arduino.HIGH);
                        timerOn = true;
                        start = DateTime.Now;
                    }
                    isPaused = false;
                    break;


                default:
                    break;
            }

        }


        static public bool Connect()
        {

            // Check if we were sucessfull
            try
            {
                const string DEFAULT_SERVER = "localhost";
                const int DEFAULT_PORT = 5123;
                Console.WriteLine("Trying to connect");
                // Close the socket if it is still open
                if (clientSocket != null && isConnected)
                {
                    clientSocket.Shutdown(SocketShutdown.Both);
                    System.Threading.Thread.Sleep(10);
                    clientSocket.Close();
                }


                clientSocket = new System.Net.Sockets.Socket(System.Net.Sockets.AddressFamily.InterNetwork, System.Net.Sockets.SocketType.Stream, System.Net.Sockets.ProtocolType.Tcp);

                //System.Net.Sockets.SocketInformation clientSocketInfo;
                // The chat client always starts up on the localhost, using the default port 
                IPHostEntry hostInfo = Dns.GetHostByName(DEFAULT_SERVER);
                IPAddress serverAddr = hostInfo.AddressList[0];
                var clientEndPoint = new IPEndPoint(serverAddr, DEFAULT_PORT);

                // Create a client socket and connect it to the endpoint 

                clientSocket.Connect(clientEndPoint);
                SetupRecieveCallback(clientSocket);
                isConnected = true;
                Console.WriteLine("Connected");
            }
            catch
            {
                Console.WriteLine("Unable to connect");
                isConnected = false;

            }

            return isConnected;

        }


        static public void SetupRecieveCallback(Socket sock)
        {

            try
            {
                AsyncCallback recieveData = new AsyncCallback(OnRecievedData);
                sock.BeginReceive(m_byBuff, 0, m_byBuff.Length,
                                   SocketFlags.None, recieveData, sock);
            }
            catch
            {
                Console.WriteLine("Setup Recieve Callback failed!");
            }
        }

        static public void OnRecievedData(IAsyncResult ar)
        {
            // Socket was the passed in object
            Socket sock = (Socket)ar.AsyncState;

            // Check if we got any data
            try
            {
                int nBytesRec = sock.EndReceive(ar);
                if (nBytesRec > 0)
                {

                    if (m_byBuff[nBytesRec - 1] > 0)
                    {
                        Console.WriteLine("Data Recieved");
                        processCommand(m_byBuff[nBytesRec - 1]);
                    }
                    // If the connection is still usable restablish the callback
                    SetupRecieveCallback(sock);
                    
                }
                else
                {
                    // If no data was recieved then the connection is probably dead
                    Console.WriteLine("Client disconnected");
                    sock.Shutdown(SocketShutdown.Both);
                    sock.Close();
                    isConnected = false;
                }
            }
            catch
            {
                // If no data was recieved then the connection is probably dead
                Console.WriteLine("Client disconnected");
                sock.Shutdown(SocketShutdown.Both);
                sock.Close();
                isConnected = false;
            }
        }


        static public bool sendCommand(byte command)
        {

            byte[] byData = new byte[4];// System.Text.Encoding.ASCII.GetBytes(textdata);
            byData[0] = 0;
            byData[1] = 0;
            byData[2] = 0;
            byData[3] = command;

            try
            {
                clientSocket.Send(byData);
            }
            catch
            {
                // If no data was recieved then the connection is probably dead
                Console.WriteLine("Client disconnected");
                clientSocket.Shutdown(SocketShutdown.Both);
                clientSocket.Close();
                isConnected = false;
            }

            return isConnected;
        }


 

        private static void Arduino_digitalPinUpdated(byte pin, byte state)
        {
           // Console.Write("Input pin: ");
           // Console.Write(pin);
           // Console.Write(" state:");
           // Console.WriteLine(state);

            if (isConnected && (state == Arduino.HIGH) && (pin == PIN_D_BUTTON) && !isPaused)
            {
                Console.WriteLine("Button pressed");
                sendCommand(WAKE_WORD_DETECTED);
            }

        }

        private static void sre_SpeechRecognized(object sender, SpeechRecognizedEventArgs e)
        {
            if (isConnected && (e.Result.Confidence >= 0.75) && !isPaused)
            {
                Console.WriteLine("Wake Word detected");
                sendCommand(WAKE_WORD_DETECTED);
            }

        }


        static void Main(string[] args)
        {



            // SETUP ARDUINO
            try
            {
                if (args.Length>0) arduino = new Arduino(args[0]);
                else arduino = new Arduino("COM14");
                arduinoOK = true;
                arduino.pinMode(PIN_D_BUTTON, Arduino.INPUT);   //(active HIGH)

                arduino.pinMode(PIN_D_BLUELED, Arduino.OUTPUT);
                arduino.pinMode(PIN_D_WHITELED, Arduino.OUTPUT);
                arduino.pinMode(PIN_D_ACTIVE, Arduino.OUTPUT);

                arduino.digitalPinUpdated += Arduino_digitalPinUpdated;
 
            }
            catch
            {
                Console.WriteLine("Arduino failed");
                arduinoOK = false;
            }
            arduino.digitalWrite(PIN_D_BLUELED, Arduino.HIGH);
            arduino.digitalWrite(PIN_D_WHITELED, Arduino.HIGH);
            arduino.digitalWrite(PIN_D_ACTIVE, Arduino.LOW);
            // SETUP SPEECH RECOGNITION

            try
            {
                Console.WriteLine("Setup Speech recognition");

                System.Globalization.CultureInfo ci = new System.Globalization.CultureInfo("en-us");
                SpeechRecognitionEngine sre = new SpeechRecognitionEngine(ci);
                sre.SetInputToDefaultAudioDevice();
                sre.SpeechRecognized += new EventHandler<SpeechRecognizedEventArgs>(sre_SpeechRecognized);

                Choices wordChoices = new Choices();
                wordChoices.Add("alexa");

                GrammarBuilder wordGrammarBuilder = new GrammarBuilder();
                wordGrammarBuilder.Append(wordChoices);
                Grammar keyWordsGrammar = new Grammar(wordGrammarBuilder);
                sre.LoadGrammarAsync(keyWordsGrammar);
                sre.RecognizeAsync(RecognizeMode.Multiple);

                speechOK = true;

            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                speechOK = false;
            }

            arduino.digitalWrite(PIN_D_BLUELED, Arduino.LOW);
            arduino.digitalWrite(PIN_D_WHITELED, Arduino.LOW);
            arduino.digitalWrite(PIN_D_ACTIVE, Arduino.LOW);
            // MAIN LOOP
            while (true)
            {
                if (!isConnected)
                {
                    if (!Connect()) Thread.Sleep(3000);
                }

                if (timerOn) { 
                stop = DateTime.Now;
                    if (((stop.Ticks - start.Ticks) / 10000) > 5000)
                    {
                        Console.WriteLine((stop.Ticks - start.Ticks) / 10000);
                        Console.WriteLine("Timer");
                        processCommand(RESUME_WAKE_WORD_ENGINE);
                    }
                }

            }


        } // Main




    }
} 