#include <stdlib.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>

#include "twi/twi.h"
#include "usbconfig.h"
#include "usbdrv/usbdrv.h"

#define WIIEXT_TWI_ADDR 0x52 // this is the I2C slave address of any Wiimote extensions
#define SCROLL_TICK_THRESH 150 // controls speed of mouse wheel scrolling

// this data structure is described by the report descriptor
// see report descriptor below
static struct mouse_report_t
{
	uint8_t buttons; // button mask ( . . . . . M L R )
	int8_t x;        // mouse x movement
	int8_t y;        // mouse y movement
	int8_t v_wheel;  // mouse wheel movement
	int8_t h_wheel;  // mouse wheel movement
} mouse_report;

PROGMEM char usbHidReportDescriptor[61] = {
// make sure the size matches USB_CFG_HID_REPORT_DESCRIPTOR_LENGTH in usbconfig.h
    0x05, 0x01,       // USAGE_PAGE (Generic Desktop)
    0x09, 0x02,       // USAGE (Mouse)
    0xa1, 0x01,       // COLLECTION (Application)
    0x09, 0x01,       //   USAGE (Pointer)
    0xa1, 0x00,       //   COLLECTION (Physical)
    0x05, 0x09,       //     USAGE_PAGE (Button)
    0x19, 0x01,       //     USAGE_MINIMUM (Button 1)
    0x29, 0x03,       //     USAGE_MAXIMUM (Button 3)
    0x15, 0x00,       //     LOGICAL_MINIMUM (0)
    0x25, 0x01,       //     LOGICAL_MAXIMUM (1)
    0x95, 0x03,       //     REPORT_COUNT (3)
    0x75, 0x01,       //     REPORT_SIZE (1)
    0x81, 0x02,       //     INPUT (Data,Var,Abs)
    0x95, 0x01,       //     REPORT_COUNT (1)
    0x75, 0x05,       //     REPORT_SIZE (5)
    0x81, 0x03,       //     INPUT (Cnst,Var,Abs)
    0x05, 0x01,       //     USAGE_PAGE (Generic Desktop)
    0x09, 0x30,       //     USAGE (X)
    0x09, 0x31,       //     USAGE (Y)
    0x09, 0x38,       //     USAGE (Wheel)
    0x15, 0x81,       //     LOGICAL_MINIMUM (-127)
    0x25, 0x7f,       //     LOGICAL_MAXIMUM (127)
    0x75, 0x08,       //     REPORT_SIZE (8)
    0x95, 0x03,       //     REPORT_COUNT (3)
    0x81, 0x06,       //     INPUT (Data,Var,Rel)
    0x05, 0x0c,       //     USAGE_PAGE (Consumer Devices)
    0x0a, 0x38, 0x02, //     USAGE (Undefined)
    0x95, 0x01,       //     REPORT_COUNT (1)
    0x81, 0x06,       //     INPUT (Data,Var,Rel)
    0xc0,             //   END_COLLECTION
    0xc0              // END_COLLECTION
};

static unsigned char idle_rate; // rate of reminder reports even when idle
// ignored

unsigned char usbFunctionSetup(unsigned char data[8])
{
	usbRequest_t *rq = (void *)data;

	// there are several request types but we only need to handle
	// 3 cases of the "class" type
	
	if ((rq->bmRequestType & USBRQ_TYPE_MASK) == USBRQ_TYPE_CLASS) /* class request type */
	{
		if (rq->bRequest == USBRQ_HID_GET_REPORT) /* wValue: ReportType (highbyte), ReportID (lowbyte) */
		{			
			usbMsgPtr = &mouse_report;
			return sizeof(mouse_report);
		}
		else if (rq->bRequest == USBRQ_HID_GET_IDLE)
		{
			usbMsgPtr = &idle_rate;
			return 1;
		}
		else if (rq->bRequest == USBRQ_HID_SET_IDLE)
		{
			idle_rate = rq->wValue.bytes[1];
		}
	}
	else
	{
		// no vendor specific requests implemented
	}
	
	return 0;
}

// since we have set the encryption key to all-zero, we can predict how to decrypt it
// this may or may not be needed, with the clone controller I used, it's not needed
void wiiext_decrypt(unsigned char * array, int size)
{
	for (int i = 0; i < size; i++) {
		array[i] = (array[i] ^ 0x17) + 0x17;
	}
}

int main()
{	
	DDRC &= ~(_BV(5) | _BV(4)); // TWI pins as input
	PORTC |= _BV(5) | _BV(4); // enable pull-ups
	
	twi_init();
	sei();
	
	_delay_ms(25); // power up delay
	
	uint8_t twiBuffer[8];
	
	// initialize the Wii Classic Controller
	// make decryption predictable
	
	twiBuffer[0] = 0x40; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 7, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	twiBuffer[0] = 0x46; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 7, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	twiBuffer[0] = 0x4C; twiBuffer[1] = 0x00; twiBuffer[2] = 0x00; twiBuffer[3] = 0x00; twiBuffer[4] = 0x00; twiBuffer[5] = 0x00; twiBuffer[6] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 5, 1);
	_delay_us(500); // the nunchuk needs some time to process
	
	// retrieve center value of sticks
	
	twiBuffer[0] = 0x00;
	twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 1, 1);
	twi_readFrom(WIIEXT_TWI_ADDR, twiBuffer, 6);
	//wiiext_decrypt(twiBuffer, 6);
	
	int centerLeftX = (int)((twiBuffer[0] & 0x3F) << 2);
	int centerLeftY = (int)((twiBuffer[1] & 0x3F) << 2);
	int centerRightX = (int)((twiBuffer[0] & 0xC0) | ((twiBuffer[1] & 0xC0) >> 2) | ((twiBuffer[2] & 0x80) >> 4));
	int centerRightY = (int)((twiBuffer[2] & 0x1F) << 3);

	// large counters for scrolling behaviour
	int h_scroll_cnt = 0;
	int v_scroll_cnt = 0;
	
	cli();
	
	usbInit(); // start v-usb
	usbDeviceDisconnect(); // enforce USB re-enumeration, do this while interrupts are disabled!
	_delay_ms(250);
	usbDeviceConnect();
	
	sei(); // enable interrupts
	
	while (1) // forever loop
	{
		usbPoll();
		
		// read raw values
		
		twiBuffer[0] = 0x00;
		twi_writeTo(WIIEXT_TWI_ADDR, twiBuffer, 1, 1);
		twi_readFrom(WIIEXT_TWI_ADDR, twiBuffer, 6);
		//wiiext_decrypt(twiBuffer, 6);
		
		int rawLeftX = (int)((twiBuffer[0] & 0x3F) << 2);
		int rawLeftY = (int)((twiBuffer[1] & 0x3F) << 2);
		int rawRightX = (int)((twiBuffer[0] & 0xC0) | ((twiBuffer[1] & 0xC0) >> 2) | ((twiBuffer[2] & 0x80) >> 4));
		int rawRightY = (int)((twiBuffer[2] & 0x1F) << 3);
		
		// prepare report
		
		// use left stick for mouse movement
		mouse_report.x = (rawLeftX - centerLeftX) / 6;
		mouse_report.y = -(rawLeftY - centerLeftY) / 6;
		
		// use right stick or D-pad for scroll wheel
		mouse_report.h_wheel = 0;
		mouse_report.v_wheel = 0;
		h_scroll_cnt += rawRightX - centerRightX;
		v_scroll_cnt += rawRightY - centerRightY;
		if (h_scroll_cnt > SCROLL_TICK_THRESH || bit_is_clear(twiBuffer[4], 7)) {
			h_scroll_cnt = 0;
			mouse_report.h_wheel = 1;
		}
		else if (h_scroll_cnt < -SCROLL_TICK_THRESH || bit_is_clear(twiBuffer[5], 1)) {
			h_scroll_cnt = 0;
			mouse_report.h_wheel = -1;
		}
		if (v_scroll_cnt > SCROLL_TICK_THRESH || bit_is_clear(twiBuffer[5], 0)) {
			v_scroll_cnt = 0;
			mouse_report.v_wheel = 1;
		}
		else if (v_scroll_cnt < -SCROLL_TICK_THRESH || bit_is_clear(twiBuffer[4], 6)) {
			v_scroll_cnt = 0;
			mouse_report.v_wheel = -1;
		}
		
		mouse_report.buttons = 0;
		if (bit_is_clear(twiBuffer[5], 6) || bit_is_clear(twiBuffer[5], 5)) { // B or Y button as L
			mouse_report.buttons |= (1 << 0);
		}
		if (bit_is_clear(twiBuffer[5], 4)) { // A button as R
			mouse_report.buttons |= (1 << 1);
		}
		if (bit_is_clear(twiBuffer[5], 3)) { // X button as M
			mouse_report.buttons |= (1 << 2);
		}
		
		// loop until ready to send
		while (1) {
			usbPoll();
			
			// send if ready to send
			if (usbInterruptIsReady()) {
				usbSetInterrupt((unsigned char *)(&mouse_report), sizeof(mouse_report));
				break;
			}
		}
	}
	
	return 0;
}