;---------------------------------------------------------
;
;  This code makes the following assumptions about
;  the interface of the GP2D02 to the 16F84:
;
;    - A<0> wired (through diode) to GP2D02 ir sensor Vin
;    - A<1> wired to LCD display
;    - A<2> wired to transmitter
;    - A<3> wired to heater
;    - B<1> wired to DS1820 temp sensor
;    - B<7> wired directly to GP2D02 Vout
;
;---------------------------------------------------------

;__CONFIG 3FF1h	;_CP_OFF _XT_OSC
__CONFIG _CP_OFF & _PWRTE_ON & _WDT_OFF  & _XT_OSC

		LIST 	P=16F84		; Set the chip 
					; type
		RADIX	DEC
#include "p16F84.inc"			; include register 
					; defs

CONSTANT	VIN=0	; output. PORTA bit 0 used for GP2D02 Vin
CONSTANT	LCD=1	; output, PORTA bit 1 used for LCD display
CONSTANT 	TX=2	; output. PORTA bit 2 used for Linx xmit
CONSTANT	TEMP=3	; output, PORTA bit 3 used for heater control
CONSTANT	DSBIT=1	; input/output, PORTB bit 1 used for DS1820 DQ
CONSTANT	VOUT=7  ; input,  PORTB bit 7 used for GP2D02 Vout
;CONSTANT	TLO=0x24 ; 36d, 18C or 64F low temp limit
CONSTANT	TLO=0x28 ; 40d, 20C or 68F low temp limit
CONSTANT	THI=0x2A ; 42d, 21C or 70F hi temp limit


SPR  		EQU     0x20		; register base 

;
; Delay routine registers
;

outer		EQU     SPR+0		
inner		EQU	SPR+1		
longDly		EQU	SPR+2

;
; Sharp GP2D02 range sensor
;

bitCount	EQU	SPR+3			
gp2d02Data	EQU	SPR+4 

;
; Dallas DS1820 1-Wire Digital Thermometer
;

DSMAG		EQU	SPR+6		; temperature magnitude
DSSIGN		EQU	SPR+7		; temperature sign
DSDLYF		EQU	SPR+8		; for DS1820 delay macro
DSTMP0		EQU	SPR+9		; scratch
DSTMP1		EQU	SPR+10
DSTMP2		EQU	SPR+11

;
; Serial xmit registers
;

serData		EQU 	SPR+12
serLoop		EQU 	SPR+13
serTime		EQU 	SPR+14

;
; Parity registers
;

parTemp1	equ	SPR+21
parTemp2	equ	SPR+22
serialStatus	equ	SPR+23

;
; Manchester encoding registers
;

manchTemp1	equ	SPR+25
manchBCount	equ	SPR+26
manchL		equ	SPR+27
manchH		equ	SPR+28

;
; misc
;

xmitRepeat	equ	SPR+29

;---------------------------------------------------------

            	org       0   		; define the reset    
            	goto      io_init	; vector

;---------------------------------------------------------
io_init
	      	org     H'50'

		movlw	b'00000011'	; initialize
		movwf	PORTA		; port A with the
					; the detector
					; bit off 
					; (<0> high)

		clrf	PORTB		; B is zeros

		bsf	STATUS, RP0	; select bank 1

		clrf	TRISA		; configure A as
					; all outputs

		movlw	b'10000010'	; configure B as
		movwf	TRISB		; all outputs
					; but <1><7>
 
		bcf	STATUS, RP0	; select bank 0

;---------------------------------------------------------

		call	SerInit		; setup TX output
		call    LCD_CLR		; clear LCD
		call    LCD_LINE1	; set cursor to
					; beg. of line 1
		
main_loop
		call	GP2D02		; take range reading
		;xorlw   0FFH		; invert gp2d02 data
		;movwf 	gp2d02Data	; save data
		movf	gp2d02Data, W	; setup for LCD_VAL
		call    LCD_CLR		; clear LCD
		call    LCD_LINE1	; set cursor to
					; beg. of line 1
		call	LCD_VAL		; show reading

		movlw	.5		; admire for 1.25 sec
		call	Delay250Ms


		call    LCD_LINE2	; set cursor to
					; beg. of line 2
		call	GETEMP		; readout DS1820 temp
		movf	DSSIGN,W	; display sign
		call	LCD_VAL
		movf	DSMAG,W		; display magnitude
		call	LCD_VAL

		movlw	.4		; admire for 1 sec
		call	Delay250Ms

		
send_packet
		movlw	0xFF		; packet header
		call	ToManch
		movf	manchL, W
		call	XmitSerChar	; serial xmit
		movlw	.8
		call	DelayMs
		movf	manchH, W
		call	XmitSerChar	; serial xmit
		movlw	.8
		call	DelayMs

		movlw .3
		movwf	xmitRepeat
range_data 
		movf	gp2d02Data, W	; send GP2D02 data 3x
		call	ToManch
		movf	manchL, W
		call	XmitSerChar	; serial xmit
		movlw	.8
		call	DelayMs
		movf	manchH, W
		call	XmitSerChar	; serial xmit
		movlw	.8
		call	DelayMs
		decfsz  xmitRepeat,F
		goto 	range_data
		movlw	.1		; for 3 copy check
		call	DelayMs


		movlw .3
		movwf	xmitRepeat
temp_data
		movf	DSMAG,W		; send temp data 3x
		call	ToManch
		movf	manchL, W
		call	XmitSerChar	; serial xmit
		movlw	.8
		call	DelayMs
		movf	manchH, W
		call	XmitSerChar	; serial xmit
		movlw	.8
		call	DelayMs
		decfsz  xmitRepeat,F
		goto 	temp_data
		movlw	.1		; for 3 copy check
		call	DelayMs

check_temp:
		movf	DSMAG,W		; temperature <= low temp limit?
		sublw	TLO	
		btfsc	STATUS,C
		bsf	PORTA,TEMP	; heater on
		
		movlw	THI		; temperature >= hi temp limit?
		subwf	DSMAG,W
		btfsc	STATUS,C
		bcf	PORTA,TEMP	; heater off

		goto	main_loop


;---------------------------------------------------------
; delay_1_ms subroutine
;
; This routine takes up 1ms of clock

delay_1_ms  	movlw   0xC7		; Set the inner
		movwf	inner		; count
inner_loop	nop			; 5 cycles loop
		nop			; until done
            	decfsz  inner,f
            	goto    inner_loop
            	return

;---------------------------------------------------------
; DelayMs subroutine
;
; This routine stalls (W) ms

DelayMs		movwf	outer		; loop (w) times
outer_loop  	call	delay_1_ms	; at 1ms each
		decfsz  outer,f
		goto	outer_loop
            	return

;---------------------------------------------------------
; Delay250Ms subroutine
;
; This routine stalls (W) * 250ms

Delay250Ms	movwf	longDly
loop		movlw   .250		; loop (w) times
		call	DelayMs		; at 250ms each
		decfsz	longDly,F
		goto	loop
		return

;---------------------------------------------------------
; gp2d02 subroutine
; 
; This routine takes a reading from the GP2D02 and stores
; the result in file register gp2d02Data on return.

GP2D02
		bcf	PORTA,VIN	; turn on detector
		nop

wait_for_reading
		btfss	PORTB,VOUT	; wait until done
		goto 	wait_for_reading; with measurement
		bsf	PORTA,VIN	; bring detector 
					; high
		clrf	gp2d02Data	; clear old value

		movlw	.8		; set up to clock
		movwf	bitCount	; out data

		bcf	STATUS,C	; ensure carry
					; bit clear
					; for rotates
		nop			; clock delay

read_bit	bcf	PORTA,VIN	; bring clock low
		nop			; clock delay
		nop
		rlf	gp2d02Data,F	; roll out prev.
					; bit
		btfsc	PORTB,VOUT	; check bit on 
					; output
		bsf	gp2d02Data,0	; set if output 1
		bsf	PORTA,VIN	; bring clock high
		nop			; clock delay
		nop
		decfsz  bitCount,F	; all 8 bits done?
		goto 	read_bit

		;movfw	current		; get return value
		return

;---------------------------------------------------------
; xmitSerChar subroutine
;
; Send out character in W using serial NRZ. Baud is set
; by the serial delay loops.
 

TRUE	equ	1
FALSE	equ	0

#define _PARITY_ENABLE  FALSE           ; NO Parity
#define _ODD_PARITY     FALSE           ; EVEN Parity, if Parity enabled
#define	_parityBit	serialStatus,7
_DataBits       set     8               ; 8 bit data, can be 1 to 8


SerInit					; this is already done by i/o init????
        	bcf 	STATUS, RP1
        	bsf 	STATUS, RP0     ; switch to bank 1
        	bcf 	TRISA, TX       ; make TX an output
        	bcf 	STATUS, RP0     ; back to data bank 0
		return

XmitSerChar
        	movwf   serData
if _PARITY_ENABLE
		call	GenParity	; If Parity is used, then Generate Parity Bit
endif  
        	movlw   .9
        	movwf   serLoop
        	bcf     STATUS, C       ; set C to 0, start bit

ser_out
        	btfsc   STATUS, C                               
        	bsf     PORTA, TX       ; send a one            
        	btfss   STATUS, C                               
        	bcf     PORTA, TX       ; or a zero             
 
		call	SerDelay
        
        	rrf     serData, F  	; least sign bit now in C       
        	decfsz  serLoop, F  	; does not affect status        
        	goto    ser_out        ; next character                

if _PARITY_ENABLE
		btfss	_parityBit
		bcf	PORTA, TX
		btfsc	_parityBit
        	bsf	PORTA, TX
		call	SerDelay
endif

        	bsf     PORTA, TX       ; send stop bit
		call	SerDelay
        	return


;---------------------------------------------------------
; SerDelay subroutine
;
; Delay for to set baud rate. One "136" loop for 2400 baud,
; two "136" loops for 1200 baud.

SerDelay
	movlw   .136		;1
      	movwf	serTime		; bit time delay, 408uS at 2400 baud ;1
serloop
	decfsz  serTime, F	;1 cycles
      	goto 	serloop		;2 cycles

	movlw   .136		;1
      	movwf	serTime     	; bit time delay, 408uS at 2400 baud ;1
serloop1
	decfsz  serTime, F	;1 cycles
      	goto 	serloop1	;2 cycles
	return

; -------------------------------------------
; SUBROUTINE: CONVERT BYTE TO MANCHESTER WORD
; -------------------------------------------
; Data to convert is in W
; Result is in manchH and manchL
;

ToManch 	movwf manchTemp1        ; save data byte
        	movlw 8h
        	movwf manchBCount       ; 8 bits per byte

man_loop 	rlf manchTemp1
        	btfsc STATUS,C
        	goto bit_is_one
;
; Bit to convert = 0 - place 01 into manchester bytes
;
        	bcf STATUS,C
        	rlf manchL
        	rlf manchH
        	bsf STATUS,C
        	rlf manchL
        	rlf manchH
        	goto dec_m_loop
;
; Bit to convert = 1 - place 10 into manchester bytes
;
bit_is_one	bsf STATUS,C
        	rlf manchL
        	rlf manchH
        	bcf STATUS,C
        	rlf manchL
        	rlf manchH

dec_m_loop      decfsz manchBCount  ; do for 8 bits
        	goto man_loop
        	return

;*********************************************************************************************************
;				Generate Parity for the Value in WREG
;
;  The parity bit is set in _parityBit (serialStatus,7)
;  Common Routine For Both Transmission & Reception
;
;  Program Memory :	16 locations
;  Cycles	  :	72
;
;*********************************************************************************************************
if _PARITY_ENABLE

GenParity:
	movwf	temp2				;save data
	movlw	_DataBits
	;movf	BitCount,W			;save bitcount
	movwf	parTemp1  
Parityloop
        rrf	parTemp2,F
	btfss	STATUS,C			;put data in carry bit
	goto	NotOne	
	xorlw	00h				;parity calculated by XORing all data bits
	goto	OneDone
NotOne
	xorlw	01h
OneDone
	decfsz	parTemp1,F
	goto	Parityloop			;decrement count
	movwf	parTemp1			; Parity bit is in Bit 0 of parTemp1
						;
    if _ODD_PARITY
	bsf	_parityBit
	btfsc	parTemp1,0
        bcf	_parityBit
    else
	bcf	_parityBit
	btfsc	parTemp1,0
        bsf	_parityBit
    endif

	return
endif    


#include "ds1820.asm"        
#include "lcd1ctrl.asm"
	end



