/*
RoboDolly
........................................
.%%%%%....%%%%...%%%%%....%%%%..........
.%%..%%..%%..%%..%%..%%..%%..%%.........
.%%%%%...%%..%%..%%%%%...%%..%%..%%%%%%.
.%%..%%..%%..%%..%%..%%..%%..%%.........
.%%..%%...%%%%...%%%%%....%%%%..........
........................................
.%%%%%....%%%%...%%......%%......%%..%%.
.%%..%%..%%..%%..%%......%%.......%%%%..
.%%..%%..%%..%%..%%......%%........%%...
.%%..%%..%%..%%..%%......%%........%%...
.%%%%%....%%%%...%%%%%%..%%%%%%....%%...
........................................

Operates the hardware of the RoboDolly.
Switches between three modes:
  1) Scrolling Text
  2) Flashing Text
  3) Etch-Sketch Mode
Please follow pin configuration information in code.
All configurations are consistent with PCB templates.
Edit message under "message settings" at beginning of code.
  Use only capital letters and numbers.  All other chars ignored.

THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY 
KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
PARTICULAR PURPOSE.

Copyright 2012
by Andre Aboulian (contact@andreaboulian.com)
*/

//BEGIN CODE

//message settings
const char message[] = "TEST TEXT IS FOR TESTING ONLY  ";  //must be capital letters and numbers
int messageLength;  //number of chars in message to display

//pin setup
int xPotPin = A2;  //analog pin
int yPotPin = A3;  //analog pin

//pin setup
int matrixSerialPin = 8;
int matrixLatchPin = 7;
int matrixClockPin = 6;

int eyePins[] = {11,12,10};  //ordered R, G, B pins

int resetButtonPin = 3;
int modeButtonPin = 4;

//matrix declaration
const int numRows = 9;
const int numCols = 7;
boolean pixels[numRows][numCols];
byte rowData;  //cathodes 1-8
byte colData;  //all anodes and cathode 0

int delayDisplay = 20; //refresh delay in microseconds

//cursor position
int x, y;

//insertion point
long lastBlinkOffTime = millis();
long lastBlinkOnTime = 0;
int timeDiff;
int blinkTime = 300;  //set blink timing in milliseconds
boolean state = 0;
int lastX, lastY;

//switch logic
boolean hasBeenCleared = false;
int lastMode;
int mode = 0;
boolean isModeButtonPressed = false;

//text blinking
int letterPosition;
int flashLength = 70;  //number of times to scan each letter; speed
int timesTextFlashed;
int flashingTimingSlowest = 450;
int flashingTimingFastest = 30;
int timeTextOff;
double offFactor = 0.12;
int offLength = (int)(offFactor*(double)flashLength) + 1;

//text scanning
int textPosition;
int displayIterations = 90;  //number of times to scan frame
int timesScanned;  //number of itmes current frame has been scanned
boolean pendingLetter[5][5];
int scanTimingSlowest = 150;
int scanTimingFastest = 30;//must be >=10

//eye flashing
int eyeTimingIterations = 40;
int eyeTimingElapsed = 0;
int eyeColor = 0;
int eyeTimingSlowest = 400;
int eyeTimingFastest = 15;

 //letters A-Z in indicies 0-25 
 //blank space in index 26 
 //filled space in index 27
 boolean alpha[][5][5] = {    {     {       0,0,1,0,0    }     ,     {       0,1,0,1,0    }     ,     {       1,0,0,0,1    }     ,     {       1,1,1,1,1    }     ,     {       1,0,0,0,1    }   }   ,    {     {       1,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       1,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       1,1,1,1,0    }   }   ,    {     {       0,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       1,0,0,0,0    }     ,     {       1,0,0,0,1    }     ,     {       0,1,1,1,0    }   }   ,    {     {       1,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       1,0,0,0,1    }     ,     {       1,0,0,0,1    }     ,     {       1,1,1,1,0    }   }   ,    {     {       1,1,1,1,1    }     ,     {       1,0,0,0,0    }     ,     {       1,1,1,1,0    }     ,     {       1,0,0,0,0    }     ,     {       1,1,1,1,1    }   }   ,    {     {       1,1,1,1,1    }     ,     {       1,0,0,0,0    }     ,     {       1,1,1,1,0    }     ,     {       1,0,0,0,0    }     ,     {       1,0,0,0,0    }   }   ,    {     {       0,1,1,1,1    }     ,     {       1,0,0,0,0    }     ,     {       1,0,0,1,1    }     ,     {       1,0,0,0,1    }     ,     {       0,1,1,1,1    }   }   ,    {     {       1,0,0,0,1    }     ,     {       1,0,0,0,1    }     ,     {       1,1,1,1,1    }     ,     {       1,0,0,0,1    }     ,     {       1,0,0,0,1    }   }   ,    {     {       0,1,1,1,0    }     ,     {       0,0,1,0,0    }     ,     {       0,0,1,0,0    }     ,     {       0,0,1,0,0    }     ,     {       0,1,1,1,0    }   }   ,    {     {       0,0,1,1,1    }     ,     {       0,0,0,0,1    }     ,     {       0,0,0,0,1    }     ,     {       1,0,0,0,1    }     ,     {       0,1,1,1,0    }   }   ,    {     {       1,0,0,1,0    }     ,     {       1,0,1,0,0    }     ,     {       1,1,0,0,0    }     ,     {       1,0,1,0,0    }     ,     {       1,0,0,1,0    }   }   ,    {     {       1,0,0,0,0    }     ,     {       1,0,0,0,0    }     ,     {       1,0,0,0,0    }     ,     {       1,0,0,0,0    }     ,     {       1,1,1,1,1    }   }   ,    {     {       1,0,0,0,1    }     ,     {       1,1,0,1,1    }     ,     {       1,0,1,0,1    }     ,     {       1,0,1,0,1    }     ,     {       1,0,0,0,1    }   }   ,    {     {       1,0,0,0,1    }     ,     {       1,1,0,0,1    }     ,     {       1,0,1,0,1    }     ,     {       1,0,0,1,1    }     ,     {       1,0,0,0,1    }   }   ,    {     {       0,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       1,0,0,0,1    }     ,     {       1,0,0,0,1    }     ,     {       0,1,1,1,0    }   }   ,    {     {       1,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       1,1,1,1,0    }     ,     {       1,0,0,0,0    }     ,     {       1,0,0,0,0    }   }   ,    {     {       0,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       1,0,0,0,1    }     ,     {       0,1,1,1,0    }     ,     {       0,0,0,1,1    }   }   ,    {     {       1,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       1,1,1,1,0    }     ,     {       1,0,1,0,0    }     ,     {       1,0,0,1,1    }   }   ,    {     {       0,1,1,1,1    }     ,     {       1,0,0,0,0    }     ,     {       0,1,1,1,0    }     ,     {       0,0,0,0,1    }     ,     {       1,1,1,1,0    }   }   ,    {     {       1,1,1,1,1    }     ,     {       0,0,1,0,0    }     ,     {       0,0,1,0,0    }     ,     {       0,0,1,0,0    }     ,     {       0,0,1,0,0    }   }   ,    {     {       1,0,0,0,1    }     ,     {       1,0,0,0,1    }     ,     {       1,0,0,0,1    }     ,     {       1,0,0,0,1    }     ,     {       0,1,1,1,0    }   }   ,    {     {       1,0,0,0,1    }     ,     {       1,0,0,0,1    }     ,     {       0,1,0,1,0    }     ,     {       0,1,0,1,0    }     ,     {       0,0,1,0,0    }   }   ,    {     {       1,0,0,0,1    }     ,     {       1,0,1,0,1    }     ,     {       1,0,1,0,1    }     ,     {       1,1,0,1,1    }     ,     {       1,0,0,0,1    }   }   ,    {     {       1,0,0,0,1    }     ,     {       0,1,0,1,0    }     ,     {       0,0,1,0,0    }     ,     {       0,1,0,1,0    }     ,     {       1,0,0,0,1    }   }   ,    {     {       1,0,0,0,1    }     ,     {       0,1,0,1,0    }     ,     {       0,0,1,0,0    }     ,     {       0,0,1,0,0    }     ,     {       0,0,1,0,0    }   }   ,    {     {       1,1,1,1,1    }     ,     {       0,0,0,1,0    }     ,     {       0,0,1,0,0    }     ,     {       0,1,0,0,0    }     ,     {       1,1,1,1,1    }   }   ,    {     {       0,0,0,0,0    }     ,     {       0,0,0,0,0    }     ,     {       0,0,0,0,0    }     ,     {       0,0,0,0,0    }     ,     {       0,0,0,0,0    }   }   ,    {     {       1,1,1,1,1    }     ,     {       1,1,1,1,1    }     ,     {       1,1,1,1,1    }     ,     {       1,1,1,1,1    }     ,     {       1,1,1,1,1    }   }  };  boolean numerica[][5][5] =  {    {     {       0,1,1,1,0    }     ,     {       1,0,0,1,1    }     ,     {       1,0,1,0,1    }     ,     {       1,1,0,0,1    }     ,     {       0,1,1,1,0    }   }   ,    {     {       0,0,1,0,0    }     ,     {       0,1,1,0,0    }     ,     {       0,0,1,0,0    }     ,     {       0,0,1,0,0    }     ,     {       0,0,1,0,0    }   }   ,    {     {       1,1,1,1,0    }     ,     {       0,0,0,0,1    }     ,     {       0,1,1,1,0    }     ,     {       1,0,0,0,0    }     ,     {       1,1,1,1,1    }   }   ,    {     {       1,1,1,1,0    }     ,     {       0,0,0,0,1    }     ,     {       0,1,1,1,0    }     ,     {       0,0,0,0,1    }     ,     {       1,1,1,1,0    }   }   ,    {     {       0,0,0,1,1    }     ,     {       0,0,1,0,1    }     ,     {       0,1,0,0,1    }     ,     {       1,1,1,1,1    }     ,     {       0,0,0,0,1    }   }   ,    {     {       1,1,1,1,1    }     ,     {       1,0,0,0,0    }     ,     {       1,1,1,1,0    }     ,     {       0,0,0,0,1    }     ,     {       1,1,1,1,0    }   }   ,    {     {       0,1,1,1,0    }     ,     {       1,0,0,0,0    }     ,     {       1,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       0,1,1,1,0    }   }   ,    {     {       1,1,1,1,1    }     ,     {       0,0,0,0,1    }     ,     {       0,0,0,1,0    }     ,     {       0,0,1,0,0    }     ,     {       0,1,0,0,0    }   }   ,    {     {       0,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       0,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       0,1,1,1,0    }   }   ,    {     {       0,1,1,1,0    }     ,     {       1,0,0,0,1    }     ,     {       0,1,1,1,1    }     ,     {       0,0,0,0,1    }     ,     {       0,1,1,1,0    }   }  };

//--------------------------------------------

void setup(){
  Serial.begin(9600);

  //shift register pins
  pinMode(matrixLatchPin, OUTPUT);
  pinMode(matrixSerialPin, OUTPUT);
  pinMode(matrixClockPin, OUTPUT);

  //switch pins
  pinMode(resetButtonPin, INPUT);
  pinMode(modeButtonPin, INPUT);
  
  //pot pins
  pinMode(xPotPin, INPUT);
  pinMode(yPotPin, INPUT);
  
  //eye pins
  pinMode(eyePins[0], OUTPUT);
  pinMode(eyePins[1], OUTPUT);
  pinMode(eyePins[2], OUTPUT);
  
//  analogReference(EXTERNAL);

  messageLength = (String(message)).length();
  
  resetTextScanning();
  resetTextFlashing();
  resetSketch();
  clearPixels();
  lastMode = mode;
  
  setEyeColor(1,0,0);
}

void loop(){
  checkAndSwitchMode();

  //mode
  //0 - scrolling text
  //1 - flashing text
  //2 - sketch
  if(mode != lastMode){
    Serial.println("switch");
    clearPixels();
    //switching from scrolling text to flashing text
    if(lastMode == 0){
      resetTextFlashing();
    }
    //switching from flashing text to sketch
    else if(lastMode == 1){
      resetSketch();
    }
    //switching from sketch to scrolling text
    else if(lastMode == 2){
      resetTextScanning();
    }
    refreshDisplay();
    lastMode = mode;
  }

  //scrolling text mode
  if(mode == 0){
    shiftText();
    refreshDisplay();
  }
  //flashing text mode
  else if(mode == 1){
    flashMessage();
    refreshDisplay();
  }
  //sketch mode
  else if(mode == 2){
    recordPosition();
    refreshDisplayWithBlinkingInsertion();
  }
  flashEyes();
  checkSwitchAndClear();
}

void flashEyes(){
  if(eyeTimingElapsed % 10 == 0){
    //determine timing
    if(mode == 0 || mode == 1){
      eyeTimingIterations = map(analogRead(yPotPin), 0, 1023, eyeTimingSlowest, eyeTimingFastest);
    }
    else if(mode == 2){
      eyeTimingIterations = 30;
    }
  }
  
  if(++eyeTimingElapsed >= eyeTimingIterations){
    eyeTimingElapsed = 0;
    //switch color
    if(++eyeColor > 6){
      eyeColor = 0;
    }
        
    //set color
    switch(eyeColor){
      case(0): setEyeColor(1,0,0); break;
      case(1): setEyeColor(0,1,0); break;
      case(2): setEyeColor(0,0,1); break;
      case(3): setEyeColor(1,1,0); break;
      case(4): setEyeColor(0,1,1); break;
      case(5): setEyeColor(1,0,1); break;
      case(6): setEyeColor(1,1,1); break;
      default: break;
    }
  }
}

void setEyeColor(boolean r, boolean g, boolean b){
  setEyes(0, r);
  setEyes(1, g);
  setEyes(2, b);
}

void setEyes(int pin, boolean state){
  if(state){
    digitalWrite(eyePins[pin], HIGH);
  }
  else{
    digitalWrite(eyePins[pin], LOW);
  }
}

void shiftText(){
  if(timesScanned % 10 == 0){
    displayIterations = map(analogRead(xPotPin), 0, 1023, scanTimingSlowest, scanTimingFastest);
  }
  if(++timesScanned >= displayIterations){
    shiftAllColsLeft();
    timesScanned = 0;
    if(++textPosition >= 5){
      textPosition = -1;
      if(++letterPosition >= messageLength){
        letterPosition = 0;
      }
      setPendingLetter(message[letterPosition]);
    }
    else{
      for(int row = 0; row != 5; row++){
        pixels[row+2][numCols-1] = pendingLetter[row][textPosition];
      }
    }
  }
}

void shiftAllColsLeft(){
  for(int row = 0; row != numRows; row++){
    for(int col = 0; col != numCols - 1; col++){
      pixels[row][col] = pixels[row][col+1];
    }
    pixels[row][numCols-1] = 0;
  }
}

void resetTextScanning(){
  displayIterations = map(analogRead(xPotPin), 0, 1023, scanTimingSlowest, scanTimingFastest);
  textPosition = -1;
  timesScanned = 0;
  setPendingLetter(message[0]);
  letterPosition = 0;
  clearPixels();
}

void checkAndSwitchMode(){
  boolean modeButtonRead = digitalRead(modeButtonPin);

  if(modeButtonRead && !isModeButtonPressed){
    if(++mode > 2){
      mode = 0;
    }
    isModeButtonPressed = true;
  }
  else if(isModeButtonPressed && !modeButtonRead){
    isModeButtonPressed = false;
  }
}

void recordPosition(){
//  x = map(analogRead(xPotPin), 0, 1023, 0, 6);
//  y = map(analogRead(yPotPin), 0, 1023, 0, 8);
//  
//  if(x > 6){
//    x = 6;
//  }
//  if(y > 8){
//    y = 8;
//  }
  x = customMap(xPotPin, 6);
  y = customMap(yPotPin, 8);  
  
  pixels[y][x] = 1;
}

void checkSwitchAndClear(){
  boolean clearButton = digitalRead(resetButtonPin);
  if(clearButton & !hasBeenCleared){
    switch(mode){
      case(0): resetTextScanning(); break;
      case(1): resetTextFlashing(); break;
      case(2): resetSketch(); break;
      default: break;
    }
    hasBeenCleared = true;
  }
  else if(!clearButton & hasBeenCleared){
    hasBeenCleared = false;
  }
}

void resetSketch(){
  clearPixels();
  recordPosition();
  lastX = x;
  lastY = y;
}

void refreshDisplayWithBlinkingInsertion(){
  if(lastX != x || lastY != y){
    pixels[lastY][lastX] = 1;
    lastX = x;
    lastY = y;
  }

  if(!state){
    timeDiff = millis() - lastBlinkOffTime;
    if(timeDiff > blinkTime || timeDiff < 0){
      lastBlinkOnTime = millis();
      state = 1;
      pixels[y][x] = 1;
    }
    else{
      pixels[y][x] = 0;
    }
  }
  else{
    timeDiff = millis() - lastBlinkOnTime;
    if(timeDiff > blinkTime || timeDiff < 0){
      lastBlinkOffTime = millis();
      state = 0;
      pixels[y][x] = 0;
    }
    else{
      pixels[y][x] = 1;
    }
  }

  refreshDisplay();
}

void refreshDisplay(){
  //shift data one col at a time
  //transistor logic - cols on when col low and row high
  for(int col = 0; col != numCols; col++){
    colData = 0;
    rowData = 0;
    //turn on col
    for(int i = 0; i != numCols; i++){
      if (i != col){
        colData |= (1 << i);
      }
    }

    //row adjustment for shift register on col register
    if(pixels[0][col]){
      colData |= (1 << 7);
    }

    //rows shifted with row register
    for(int row = 1; row != numRows; row++){
      if(pixels[row][col]){
        rowData |= (1 << row - 1);
      }
    }
    shiftOutData();
  }
}

void shiftOutData(){
  digitalWrite(matrixLatchPin, LOW);
  shiftOut(matrixSerialPin, matrixClockPin, MSBFIRST, B00000000);
  shiftOut(matrixSerialPin, matrixClockPin, MSBFIRST, B00000000);
  shiftOut(matrixSerialPin, matrixClockPin, MSBFIRST, rowData);
  shiftOut(matrixSerialPin, matrixClockPin, MSBFIRST, colData);
  digitalWrite(matrixLatchPin, HIGH);
  
  delayMicroseconds(delayDisplay);
}

void clearPixels(){
  for(int row = 0; row != numRows; row++){
    for(int col = 0; col != numCols; col++){
      pixels[row][col] = 0;
    }
  }
}

void fillPixels(){
  for(int row = 0; row != numRows; row++){
    for(int col = 0; col != numCols; col++){
      pixels[row][col] = 1;
    }
  }
}

// returns >100+index if number and index if letter
int retrieveCharIndex(char c){
  int asciiVal = (int) c;
  if(asciiVal >=65 && asciiVal <= 90){
    return asciiVal - 65;
  }
  else if(asciiVal == 32){  //space
    return 26;
  }
  else if(asciiVal >= 48 && asciiVal <= 57){
    return asciiVal - 48 + 100;  //flag for number (100+index)
  }
  else{  //filled pixels
    return 27;
  }
}

void placeLetter(char c){
  clearPixels();
  int index = retrieveCharIndex(c);

  for(int col = 0; col != 5; col++){
    for(int row = 0; row != 5; row++){
      if(index < 100){
        pixels[row + 2][col + 1] = alpha[index][row][col];
      }
      else{
        pixels[row + 2][col + 1] = numerica[index-100][row][col];
      }
    }
  }
}

void flashMessage(){
  if(timesTextFlashed == 0){
    if(++timeTextOff >= offLength){
      timesTextFlashed = 1;
    }
  }
  else if(++timesTextFlashed >= flashLength){
    letterPosition++;
    timesTextFlashed = 0;
    timeTextOff = 0;
    clearPixels();
  }
  else{
    if(letterPosition >= messageLength){
      letterPosition = 0;
     }
     placeLetter(message[letterPosition]);
  }
  if(timesTextFlashed % 10 == 0){
    flashLength = map(analogRead(xPotPin), 0, 1023, flashingTimingSlowest, flashingTimingFastest);
  }
}

void setPendingLetter(char c){
  int index = retrieveCharIndex(c);

  for(int col = 0; col != 5; col++){
    for(int row = 0; row != 5; row++){
      if(index < 100){
        pendingLetter[row][col] = alpha[index][row][col];
      }
      else{
        pendingLetter[row][col] = numerica[index-100][row][col];
      }
    }
  }
}

void resetTextFlashing(){
  letterPosition = 0;
  timesTextFlashed = 0;
  flashLength = map(analogRead(xPotPin), 0, 1023, flashingTimingSlowest, flashingTimingFastest);
  timeTextOff = 0;
  offLength = (int)(offFactor*(double)flashLength) + 1;
}

int customMap(int analogPin, int toNum){
  int scaled = (int) (((double) analogRead(analogPin)) / 1023.0 * ((double)toNum + 1.0));
  if(scaled > toNum){
    return toNum;
  }
    return scaled;
}

//END CODE
