/*
  SIM900Modem.h - SIM900 GSM module library for DefendLineII
  Copyright (c) 2011 Dmitry Pakhomenko.  All right reserved.

  http://atmega.magictale.com

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <avr/interrupt.h>
#include <util/delay.h>
#include <string.h>

#include "hardwareserial.h"
#include "Board.h"
#include "Messages.h"
#include "SIM900Modem.h"


// Initialize Class Variables //////////////////////////////////////////////////

// Constructors ////////////////////////////////////////////////////////////////

SIM900Modem::SIM900Modem(HardwareSerial* modem, HardwareSerial* debug)
{
    mdm = modem;
    dbg = debug;
}

// Public Methods //////////////////////////////////////////////////////////////
uint8_t SIM900Modem::init()
{
    if (sendATCmdWaitResp_p(PSTR("Z"))) 
    {
        //dbg->print_p(ALREADY_POWERED_UP);
        return true;
    }
    //dbg->print_p(POWERING_UP);
    mdm->flush();
    PWRK_1;
    _delay_ms(500);
    _delay_ms(500);
    PWRK_0;
    mdm->write('A');

    for (uint8_t i = 0; i < MDM_TIMEOUT; i++)
    {
        _delay_ms(500);
        if (sendATCmdWaitResp_p(PSTR("Z"))) return true;
    }
    return false;
}

uint8_t findSubstr(const ring_buffer* buf, uint16_t len, const prog_char expectedResp[],
    uint16_t* posFound)
{
    prog_char* ptrnStr = (prog_char*)expectedResp;
    int tmpTail = buf->tail;
    char srcChar;
    char ptrnChar;
    for (uint16_t i = 0; i < len; i++)
    {
        srcChar = buf->buffer[tmpTail];
        ptrnChar = pgm_read_byte(ptrnStr++);
        if (!ptrnChar) 
        {
            if (posFound != NULL) *posFound = i;
            return true;
        }
        if (srcChar != ptrnChar) ptrnStr = (prog_char*)expectedResp;
        tmpTail = (tmpTail + 1) % buf->rx_buffer_size;
    }
    return false;
}

uint8_t SIM900Modem::powerOff()
{
    dbg->print_p(GSM_MODEM_NAME);
    dbg->print_p(POWERING_OFF);
    return sendATCmdWaitResp_p(PSTR("+CPOWD=1"), PSTR("POWER DOWN"), 50);
}

void SIM900Modem::forcedPowerOff()
{
    dbg->print_p(FORCING_POWER_OFF);
    PWRK_1;
    _delay_ms(500);
    _delay_ms(500);
    PWRK_0;
}

uint8_t SIM900Modem::atCmdWaitResp(const prog_char expectedResp[], uint8_t delayCntr)
{
    for (uint8_t i = 0; i < MDM_TIMEOUT; i++)
    {
        _delay_ms(100);
        for (uint8_t j = 0; j < delayCntr; j++)
        {
            if (mdm->available() >= 0) break;
            _delay_ms(100);
        }
        if (mdm->available() >= 0) break;
    }
    if (mdm->available() == 0) 
    {
        return false;
    }
    ring_buffer* buf = mdm->getBuffer();
    for (uint8_t i = 0; i < MDM_TIMEOUT; i++)
    {
        if (expectedResp == NULL)
        {
            if (findSubstr(buf, mdm->available(), MDM_OK_REPLY, NULL))
                return true;
            if (findSubstr(buf, mdm->available(), MDM_ERROR_REPLY, NULL))
                break;
        }else
        {
            if (findSubstr(buf, mdm->available(), expectedResp, NULL))
                return true;
        }
        _delay_ms(100);
        for (uint8_t j = 0; j < delayCntr; j++)
            _delay_ms(100);
    }
    return false;
}


uint8_t SIM900Modem::sendATCmdWaitResp(const char* cmd)
{
    mdm->flush();
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->write((uint8_t*)cmd, strlen(cmd));
    return atCmdWaitResp(NULL, 0);
}

uint8_t SIM900Modem::sendATCmdWaitResp(const char* cmd, const prog_char expectedResp[], uint8_t delayCntr)
{
    mdm->flush();
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->write((uint8_t*)cmd, strlen(cmd));
    return atCmdWaitResp(expectedResp, delayCntr);
}

uint8_t SIM900Modem::sendATCmdWaitResp_p(const prog_char cmd[])
{
    mdm->flush();
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->println_p(cmd);
    return atCmdWaitResp(NULL, 0);
}

uint8_t SIM900Modem::sendATCmdWaitResp_p(const prog_char cmd[], const prog_char expectedResp[], uint8_t delayCntr)
{
    mdm->flush();
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->println_p(cmd);
    return atCmdWaitResp(expectedResp, delayCntr);
}

uint8_t SIM900Modem::extractReply(char* outBuf, uint16_t len)
{
    return extractReply(outBuf, len, 2);
}

uint8_t SIM900Modem::extractReply(char* outBuf, uint16_t len, uint8_t line)
{
    uint8_t lfFound = 0;
    ring_buffer* buf = mdm->getBuffer();
    int tmpTail = buf->tail;
    int outBufIdx = 0;
    char srcChar;
    for (uint8_t i = 0; i < mdm->available(); i++)
    {
        srcChar = buf->buffer[tmpTail];
        if (lfFound == line && srcChar != 0xA)
        {
            if (outBufIdx < len) outBuf[outBufIdx++] = srcChar;
            else break;
        }
        if (srcChar == 0xD) lfFound++;
        tmpTail = (tmpTail + 1) % buf->rx_buffer_size;
    }
    if (outBufIdx > 0) outBuf[outBufIdx] = 0x0;
    return outBufIdx;
}

uint8_t SIM900Modem::getManufacturerID(char* buf, uint16_t len)
{
    if (!sendATCmdWaitResp_p(PSTR("+GMI"))) return 0;
    return extractReply(buf, len);
}

uint8_t SIM900Modem::getModelID(char* buf, uint16_t len)
{
    if (!sendATCmdWaitResp_p(PSTR("+GMM"))) return 0;
    return extractReply(buf, len);
}

uint8_t SIM900Modem::getTARevision(char* buf, uint16_t len)
{
    if (!sendATCmdWaitResp_p(PSTR("+GMR"))) return 0;
    return extractReply(buf, len);
}

uint8_t SIM900Modem::getIMEI(char* buf, uint16_t len)
{
    if (!sendATCmdWaitResp_p(PSTR("+GSN"))) return 0;
    return extractReply(buf, len);
}

uint8_t SIM900Modem::getNetworkRegistration()
{
    if (!sendATCmdWaitResp_p(PSTR("+CGREG?"))) return false;
    //Registered in home network?
    if (findSubstr(mdm->getBuffer(), mdm->available(), PSTR(",1"), NULL))
        return true;
    else 
    //Well... maybe registered in roaming?
        return (findSubstr(mdm->getBuffer(), mdm->available(), PSTR(",5"), NULL));
}

uint8_t SIM900Modem::waitForNetworkRegistration()
{
    for (uint8_t i = 0; i < MDM_TIMEOUT; i++)
    {
        if (getNetworkRegistration()) return true;
        _delay_ms(500);
        _delay_ms(500);
        _delay_ms(500);
        _delay_ms(500);
        _delay_ms(500);
        _delay_ms(500);
    }
    //Failure to register issue - reset modem and try again
    sendATCmdWaitResp_p(PSTR("+CFUN=1,1"));

    for (uint8_t i = 0; i < MDM_TIMEOUT; i++)
    {
        if (getNetworkRegistration()) return true;
        _delay_ms(500);
        _delay_ms(500);
        _delay_ms(500);
        _delay_ms(500);
        _delay_ms(500);
        _delay_ms(500);
    }
    return false;
}

uint8_t SIM900Modem::getNetworkOperator(char* buf, uint16_t len)
{
    if (!sendATCmdWaitResp_p(PSTR("+COPS?"))) return false;
    return extractReply(buf, len);
}

uint8_t SIM900Modem::enableRTC()
{
    return sendATCmdWaitResp_p(PSTR("+CLTS=1"));
}

uint8_t SIM900Modem::getRTC(RtcTimeType* rtc)
{
    if (!sendATCmdWaitResp_p(PSTR("+CCLK?"))) return false;

    ring_buffer* rec_buf = mdm->getBuffer();

    //Searching for the datetime in the
    //following form: '+CCLK: "11/05/27,10:13:04"'
    uint16_t posFound;
    uint8_t temp = false;
    if (findSubstr(rec_buf, mdm->available(), PSTR("+CCLK: \""), &posFound))
    {

        temp = true;
        for (uint16_t i = 0; i < posFound; i++)
        {
            if (!(mdm->available() > 0))
            {
                temp = false;
                break;
            }
            mdm->read();
        }
        if (temp)
        {
            // At this point we should be at the very beginning of timestamp,
            // means at 'year'

            if (!(mdm->available() > 0) || (mdm->available() < 20))
            {
                temp = false;
            }else
            {   // Parsing received timestamp into Rtc structure
                rtc->year = mdm->read() - '0';
                rtc->year = rtc->year * 16 + (mdm->read() - '0');
                mdm->read();
                    
                rtc->month = mdm->read() - '0';
                rtc->month = rtc->month * 16 + (mdm->read() - '0');
                mdm->read();

                rtc->day = mdm->read() - '0';
                rtc->day = rtc->day * 16 + (mdm->read() - '0');
                mdm->read();

                rtc->hours = mdm->read() - '0';
                rtc->hours = rtc->hours * 16 + (mdm->read() - '0');
                mdm->read();

                rtc->minutes = mdm->read() - '0';
                rtc->minutes = rtc->minutes * 16 + (mdm->read() - '0');
                mdm->read();

                rtc->seconds = mdm->read() - '0';
                rtc->seconds = rtc->seconds * 16 + (mdm->read() - '0');
            }
        }
        return true;

    }else return false;
}

//=========================================== SMS related commands 

uint8_t SIM900Modem::setTxtSMSMsgFormat()
{
    return sendATCmdWaitResp_p(PSTR("+CMGF=1"));
}

uint8_t SIM900Modem::setGSMCharSet()
{
    return sendATCmdWaitResp_p(PSTR("+CSCS=\"GSM\""));
}

uint8_t SIM900Modem::sendSMS(char* destAddr, char* msg)
{
    mdm->flush();
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->print_p(PSTR("+CMGS=\""));
    mdm->write((uint8_t*)destAddr, strlen(destAddr));
    mdm->println_p(PSTR("\""));
    if (!atCmdWaitResp(PSTR(">"), 0)) return false;
    mdm->write((uint8_t*)msg, strlen(msg));
    mdm->print(0x1A, BYTE);
    return atCmdWaitResp(NULL, 100);
}

//=========================================== GPRS related commands

uint8_t SIM900Modem::openGPRSContext(char* buf)
{
    if (!sendATCmdWaitResp_p(PSTR("+SAPBR=3,1,\"Contype\",\"GPRS\""))) return false;
    mdm->flush();
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->print_p(PSTR("+SAPBR=3,1,\"APN\",\""));
    mdm->write((uint8_t*)buf, strlen(buf));
    mdm->println_p(PSTR("\""));
    if (!atCmdWaitResp(NULL, 0)) return false;
    return sendATCmdWaitResp_p(PSTR("+SAPBR=1,1"), NULL, 50);
}

uint8_t SIM900Modem::closeGPRSContext()
{
    return sendATCmdWaitResp_p(PSTR("+SAPBR=0,1"));
}

uint8_t SIM900Modem::queryGPRSContext(char* buf, uint16_t len)
{
    if (!sendATCmdWaitResp_p(PSTR("+SAPBR=2,1"))) return false;
    return extractReply(buf, len);
}

//=========================================== HTTP related commands

uint8_t SIM900Modem::httpInit()
{
    if (!sendATCmdWaitResp_p(PSTR("+HTTPINIT"))) return false;
    return sendATCmdWaitResp_p(PSTR("+HTTPPARA=\"CID\",1"));
}

uint8_t SIM900Modem::httpSetURL(char* buf)
{
    mdm->flush();
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->print_p(PSTR("+HTTPPARA=\"URL\",\""));
    mdm->write((uint8_t*)buf, strlen(buf));
    mdm->println_p(PSTR("\""));
    return atCmdWaitResp(NULL, 50);
}

uint8_t SIM900Modem::httpTerm()
{
    return sendATCmdWaitResp_p(PSTR("+HTTPTERM"));
}

uint8_t SIM900Modem::httpSetData(char* buf, uint16_t len, uint16_t timeout)
{
    mdm->flush();
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->print_p(PSTR("+HTTPDATA="));
    mdm->print(len, DEC);
    mdm->print_p(PSTR(","));
    mdm->println(timeout, DEC);
    if (!atCmdWaitResp(PSTR("DOWNLOAD"), 0)) return false;
    mdm->write((uint8_t*)buf, len);
    mdm->println();
    return atCmdWaitResp(NULL, 0);
}

uint8_t SIM900Modem::httpAction(uint8_t isPostMethod, uint16_t* responseLen)
{
    uint8_t res = false;
    if (isPostMethod)
        res = sendATCmdWaitResp_p(PSTR("+HTTPACTION=1"), PSTR(":1,200,"), 50);
    else 
        res = sendATCmdWaitResp_p(PSTR("+HTTPACTION=0"), PSTR(":1,200,"), 50);
    if (!res) return false;
    //TODO: extract response length
    return res;
}

uint8_t SIM900Modem::httpRead(char* buf, uint16_t startIdx, uint16_t len)
{
    //TODO: add implementation
    return false;
}

//=========================================== TCP related commands
uint8_t SIM900Modem::startSingleIPConnMode()
{
    return sendATCmdWaitResp_p(PSTR("+CIPMUX=0"));
}

uint8_t SIM900Modem::startIPConn(uint8_t isTCP, char* buf, uint16_t port)
{
    mdm->flush();
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->print_p(PSTR("+CIPSTART="));
    if (isTCP)
        mdm->print_p(PSTR("\"TCP\",\""));
    else
    mdm->print_p(PSTR("\"UDP\",\""));
    mdm->write((uint8_t*)buf, strlen(buf));
    mdm->print_p(PSTR("\",\""));
    mdm->print(port, DEC);
    mdm->println_p(PSTR("\""));
    return atCmdWaitResp(PSTR("CONNECT OK"), 10);
}

uint8_t SIM900Modem::sendIPConn(char* buf, uint16_t len, uint16_t delayCntr)
{
    mdm->flush();
    mdm->setOverrideFlag(true);
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->print_p(PSTR("+CIPSEND="));
    mdm->println(len, DEC);
    if (!atCmdWaitResp(PSTR(">"), 0)) return false;
    mdm->write((uint8_t*)buf, len);
    uint8_t res = atCmdWaitResp(PSTR("SEND OK"), delayCntr);
    mdm->setOverrideFlag(false);
    return res;
}

uint8_t SIM900Modem::sendAndCheckHTTPRespIPConn(char* buf, uint16_t len, uint16_t  bufsize, 
    uint16_t delayCntr, RtcTimeType* rtc)
{
    ring_buffer* rec_buf = mdm->getBuffer();
    unsigned char* oldBufRef = rec_buf->buffer;
    uint16_t oldBifSize = rec_buf->rx_buffer_size;
    mdm->flush();
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->print_p(PSTR("+CIPSEND="));
    mdm->println(len, DEC);
    if (!atCmdWaitResp(PSTR(">"), 0)) return false;
    mdm->write((uint8_t*)buf, len);
    // Replacing receiving buffer with a bigger one
    // which is used for data sending
    cli();
    rec_buf->head = 0;
    rec_buf->tail = 0;
    rec_buf->buffer = (unsigned char*)buf;
    rec_buf->rx_buffer_size = bufsize;
    //mdm->setOverrideFlag(true);
    sei();
    uint8_t temp = false;
    for (uint8_t i = 0; i < MDM_TIMEOUT; i++)
    {
        if (findSubstr(rec_buf, mdm->available(), PSTR("HTTP/1.1 200 OK"), NULL))
        {
            temp = true;
            break;
        }
        if (findSubstr(rec_buf, mdm->available(), PSTR("HTTP/1.0 200 OK"), NULL))
        {
            temp = true;
            break;
        }
        _delay_ms(100);
        for (uint8_t j = 0; j < delayCntr; j++)
            _delay_ms(100);
	}
    if (temp) 
    {
        //Searching for the datetime on server side in the
        //following form: "Date: Sun, 05 Jun 2011 08:50:52 GMT"
        uint16_t posFound;
        if (!findSubstr(rec_buf, mdm->available(), PSTR("Date: "), &posFound))
        {
            temp = false;
        }else{
            //Skipping day of week
            posFound += 5;  
            temp = true;
            for (uint16_t i = 0; i < posFound; i++)
            {
                if (!(mdm->available() > 0))
                {
                    temp = false;
                    break;
                }
                mdm->read();
            }
            if (temp)
            {
                // At this point we should be at the very beginning of timestamp,
                // means at 'day of month'

                if (!(mdm->available() > 0) || (mdm->available() < 20))
                {
                    temp = false;
                }else
                {   // Parsing received timestamp into Rtc structure
                    rtc->day = mdm->read() - '0';
                    rtc->day = rtc->day * 16 + (mdm->read() - '0');
                    mdm->read();
                    
                    rtc->month = 1;
                    if (findSubstr(rec_buf, 4, JANUARY, NULL))
                        rtc->month = 0x1;
                    else if (findSubstr(rec_buf, 4, FEBRUARY, NULL))
                        rtc->month = 0x2;
                    else if (findSubstr(rec_buf, 4, MARCH, NULL))
                        rtc->month = 0x3;
                    else if (findSubstr(rec_buf, 4, APRIL, NULL))
                        rtc->month = 0x4;
                    else if (findSubstr(rec_buf, 4, MAY, NULL))
                        rtc->month = 0x5;
                    else if (findSubstr(rec_buf, 4, JUNE, NULL))
                        rtc->month = 0x6;
                    else if (findSubstr(rec_buf, 4, JULY, NULL))
                        rtc->month = 0x7;
                    else if (findSubstr(rec_buf, 4, AUGUST, NULL))
                        rtc->month = 0x8;
                    else if (findSubstr(rec_buf, 4, SEPTEMBER, NULL))
                        rtc->month = 0x9;
                    else if (findSubstr(rec_buf, 4, OCTOBER, NULL))
                        rtc->month = 0x10;
                    else if (findSubstr(rec_buf, 4, NOVEMBER, NULL))
                        rtc->month = 0x11;
                    else if (findSubstr(rec_buf, 4, DECEMBER, NULL))
                        rtc->month = 0x12;

                    dbg->println(mdm->read());
                    dbg->println(mdm->read());
                    dbg->println(mdm->read());
                    dbg->println(mdm->read());

                    rtc->year = mdm->read() - '0';
                    rtc->year = rtc->year * 16 + (mdm->read() - '0');
                    rtc->year = rtc->year * 16 + (mdm->read() - '0');
                    rtc->year = rtc->year * 16 + (mdm->read() - '0');
                    mdm->read();

                    rtc->hours = mdm->read() - '0';
                    rtc->hours = rtc->hours * 16 + (mdm->read() - '0');
                    mdm->read();

                    rtc->minutes = mdm->read() - '0';
                    rtc->minutes = rtc->minutes * 16 + (mdm->read() - '0');
                    mdm->read();

                    rtc->seconds = mdm->read() - '0';
                    rtc->seconds = rtc->seconds * 16 + (mdm->read() - '0');
                }
            }
        }
    }
    // Restoring the stored receiving buffer
    cli();
    rec_buf->head = 0;
    rec_buf->tail = 0;
    rec_buf->buffer = oldBufRef;
    rec_buf->rx_buffer_size = oldBifSize;
	//mdm->setOverrideFlag(false);
    sei();
    return temp;
}

/*uint8_t SIM900Modem::closeIPConn()
{
    //This command causing the module to become unresponsive. Don't use it for now
    return sendATCmdWaitResp_p(PSTR("+CIPCLOSE"), PSTR("CLOSE OK"), 50);
}*/

uint8_t SIM900Modem::deactGPRSPDPContext()
{
    return sendATCmdWaitResp_p(PSTR("+CIPSHUT"), PSTR("SHUT OK"), 50);
}

uint8_t SIM900Modem::setGPRSPDPAPN(char* apn, char* username, char* pwd)
{
    mdm->flush();
    mdm->print_p(MDM_CMD_PREFIX);
    mdm->print_p(PSTR("+CSTT=\""));
    mdm->write((uint8_t*)apn, strlen(apn));
    mdm->print_p(PSTR("\",\""));
    if (username != NULL)
        mdm->write((uint8_t*)username, strlen(username));
    mdm->print_p(PSTR("\",\""));
    if (pwd != NULL)
        mdm->write((uint8_t*)pwd, strlen(pwd));
    mdm->println_p(PSTR("\""));
    return atCmdWaitResp(NULL, 0);
}

uint8_t SIM900Modem::getLocalIP(char* buf, uint16_t len)
{
    if (!sendATCmdWaitResp_p(PSTR("+CIFSR"), PSTR("CIFSR"), 50)) return 0;
    return extractReply(buf, len);
}

uint8_t SIM900Modem::getGPRSPDPStatus(char* buf, uint16_t len)
{
    if (!sendATCmdWaitResp_p(PSTR("+CIPSTATUS"), PSTR("STATE:"), 0)) return 0;
    return extractReply(buf, len, 4);
}

uint8_t SIM900Modem::actGPRSPDPContext()
{
    return sendATCmdWaitResp_p(PSTR("+CIICR"), NULL, 50);
}

// Preinstantiate Objects //////////////////////////////////////////////////////

SIM900Modem GsmModem = SIM900Modem(&Serial3, &Serial);
