/*
  ModemHelper.c - high level functionality for GSM modem
  Copyright (c) 2011 Dmitry Pakhomenko.  All right reserved.

  http://atmega.magictale.com

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this    library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <string.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>

#include "Board.h"
#include "Messages.h"
#include "HardwareSerial.h"
#include "SIM900Modem.h"
#include "MicroSD.h"
#include "ModemHelper.h"
#include "Utils.h"

inline void fillStrHTTPParams(char** buf, StrEventType* evnt)
{
    setStrHTTPParam(buf, XX_EXTTEMP, evnt->exTemp);
    setStrHTTPParam(buf, XX_SUBJECT, evnt->subject);
    setStrHTTPParam(buf, XX_BODY, evnt->body);
    setStrHTTPParam(buf, XX_DEVICE_ID, evnt->deviceID);
    setStrHTTPParam(buf, XX_SOFT_VER, evnt->softVer);
    setStrHTTPParam(buf, XX_DETECTOR_ID, evnt->detectorID);
}

inline void setStrHTTPParam(char** buf, const prog_char name[], char* value)
{
    if (value == NULL) return;
    *buf += sprintf_P(*buf, DASH_DASH);
    *buf += sprintf_P(*buf, BOUNDARY_VAL);
    *buf += sprintf_P(*buf, CRLF);
    *buf += sprintf_P(*buf, CONTENT_DISPOSITION);
    *buf += sprintf_P(*buf, name);
    *buf += sprintf_P(*buf, QUOT);
    *buf += sprintf_P(*buf, CRLF);
    *buf += sprintf_P(*buf, CRLF);
    *buf += sprintf(*buf, value);
    *buf += sprintf_P(*buf, CRLF);
}

inline void getHostPort(char** buf, uint8_t len)
{
    eeprom_busy_wait();
    uint8_t bt;
    uint8_t* eeprPntr = (uint8_t*)eepromParams.server_HTTP_URL;
    for (uint8_t i = 0; i < len; i++)
    {
        if ((bt = eeprom_read_byte(eeprPntr++)) == '/') break;
        *buf += sprintf_P(*buf, CHAR_FORMAT, (const char*)bt);
    }
}

inline void getUri(char** buf, uint8_t len)
{
    eeprom_busy_wait();
    uint8_t bt;
    uint8_t found = false;
    uint8_t* eeprPntr = (uint8_t*)eepromParams.server_HTTP_URL;
    for (uint8_t i = 0; i < len; i++)
    {
        if ((bt = eeprom_read_byte(eeprPntr++)) == '/') found = true;
        if (found)
        {
            if (bt == 0x0) return;
            *buf += sprintf_P(*buf, CHAR_FORMAT, (const char*)bt);
        }
    }
}

inline void fillHTTPHeader(char** buf, uint32_t contentLen)
{
    //===HTTP header===
    *buf += sprintf_P(*buf, POST);
    getUri(buf, SERVER_HTTP_URL_LEN);
    *buf += sprintf_P(*buf, HTTP11);
    *buf += sprintf_P(*buf, CRLF);

    *buf += sprintf_P(*buf, HOST);
    getHostPort(buf, SERVER_HTTP_URL_LEN);
    *buf += sprintf_P(*buf, CRLF);

    *buf += sprintf_P(*buf, USER_AGENT);
    eeprom_busy_wait();
    eeprom_read_block(*buf, eepromParams.device_name, SENSOR_DESCR_LEN);
    *buf += strlen(*buf);
    *buf += sprintf_P(*buf, CRLF);

    *buf += sprintf_P(*buf, CONTENT_TYPE);
    *buf += sprintf_P(*buf, MUTLIPART_CONT_TYPE);
    *buf += sprintf_P(*buf, BOUNDARY_EQ);
    *buf += sprintf_P(*buf, BOUNDARY_VAL);
    *buf += sprintf_P(*buf, CRLF);

    *buf += sprintf_P(*buf, CONTENT_LENGTH);
    *buf += sprintf_P(*buf, DECIMAL_FORMAT, contentLen);
    *buf += sprintf_P(*buf, CRLF);
    *buf += sprintf_P(*buf, CRLF);
    //===End of HTTP header===
}

inline void fillFinalBoundaryHTTP(char** buf)
{
    *buf += sprintf_P(*buf, CRLF);
    *buf += sprintf_P(*buf, DASH_DASH);
    *buf += sprintf_P(*buf, BOUNDARY_VAL);
    *buf += sprintf_P(*buf, DASH_DASH);
    *buf += sprintf_P(*buf, CRLF);
    *buf += sprintf_P(*buf, CRLF);
}

inline void fillBinHTTPParam(char** buf, char* fileName, uint32_t* len)
{
    flashCard.fileLen(fileName, len);
    *buf += sprintf_P(*buf, DASH_DASH);
    *buf += sprintf_P(*buf, BOUNDARY_VAL);
    *buf += sprintf_P(*buf, CRLF);
    *buf += sprintf_P(*buf, CONTENT_DISPOSITION);
    *buf += sprintf_P(*buf, UPLOAD_FILE);
    *buf += sprintf_P(*buf, FILE_NAME_PAR);
    *buf += sprintf(*buf, fileName);
    *buf += sprintf_P(*buf, QUOT);
    *buf += sprintf_P(*buf, CRLF);
    *buf += sprintf_P(*buf, CONTENT_TYPE);
    *buf += sprintf_P(*buf, JPEG_CONT_TYPE);
    *buf += sprintf_P(*buf, CRLF);
    *buf += sprintf_P(*buf, CRLF);
}

inline uint8_t sendHTTPRequest(char* buf, uint16_t bufLen, StrEventType* evnt)
{
    char* tmpBufRef = buf;
    uint32_t len = 0;
    fillBinHTTPParam(&tmpBufRef, evnt->fileName, &len);
    len += strlen(buf);

    tmpBufRef = buf;
    fillStrHTTPParams(&tmpBufRef, evnt);
    len += strlen(buf);

    tmpBufRef = buf;
    fillFinalBoundaryHTTP(&tmpBufRef);
    len += strlen(buf);

    tmpBufRef = buf;
    fillHTTPHeader(&tmpBufRef, len);

    if (!GsmModem.sendIPConn(buf, strlen(buf), 10))
    {
        Serial.print_p(FAILED_TO);
        Serial.print_p(SEND);
        Serial.println_p(DATA);
        return false;
    }else{
        Serial.print_p(DATA);
        Serial.print_p(IS);
        Serial.println_p(SENT);
    }

    tmpBufRef = buf;
    fillStrHTTPParams(&tmpBufRef, evnt);

    if (!GsmModem.sendIPConn(buf, strlen(buf), 10))
    {
        Serial.print_p(FAILED_TO);
        Serial.print_p(SEND);
        Serial.println_p(DATA);
        return false;
    }else{
        Serial.print_p(DATA);
        Serial.print_p(IS);
        Serial.println_p(SENT);
    }

    tmpBufRef = buf;
    fillBinHTTPParam(&tmpBufRef, evnt->fileName, &len);

    if (!GsmModem.sendIPConn(buf, strlen(buf), 10))
    {
        Serial.print_p(FAILED_TO);
        Serial.print_p(SEND);
        Serial.println_p(DATA);
        return false;
    }else{
        Serial.print_p(DATA);
        Serial.print_p(IS);
        Serial.println_p(SENT);
    }

    uint8_t oldMdmMode = mdmMode;
    // Suppressing modem to console redirection output for binary data
    mdmMode = false;
    struct fat_file_struct* fd = flashCard.openFileInDir(flashCard.fs, flashCard.dd, evnt->fileName);
    if(fd)
    {
        uint32_t totalBytesRead = 0;
        uint32_t bytesRead;
        while((bytesRead = fat_read_file(fd, (uint8_t*)buf, bufLen/2)) > 0)
        {
            if (!GsmModem.sendIPConn(buf, bytesRead, 5))
            {
                Serial.print_p(FAILED_TO);
                Serial.print_p(SEND);
                Serial.println_p(DATA);
                break;
            }else{
                totalBytesRead += bytesRead;
                Serial.print(totalBytesRead, DEC);
                Serial.print_p(BYTES);
                //Serial.print_p(DATA);
                Serial.print_p(IS);
                Serial.println_p(SENT);
            }
        }
        fat_close_file(fd);

        Serial.print_p(TOTAL_BYTES_SENT);
        Serial.println(totalBytesRead, DEC);
        Serial.print_p(BYTES_SUPPOSED_TO_BE_SENT);
        Serial.println(len, DEC);

        if (len != totalBytesRead) return false;

        // Restoring modem to console redirection output
        mdmMode = oldMdmMode;

        tmpBufRef = buf;
        fillFinalBoundaryHTTP(&tmpBufRef);
        RtcTimeType rtc;
        if (!GsmModem.sendAndCheckHTTPRespIPConn(buf, strlen(buf), AUX_BUF_LEN, 5, &rtc))
        {
            Serial.print_p(FAILED_TO);
            Serial.print_p(SEND);
            Serial.println_p(DATA);
            return false;
        }else{
            Serial.print_p(DATA);
            Serial.print_p(IS);
            Serial.println_p(SENT);

            //if (!isTimeSync())
            {
                //Synchronizing local RTC
                eeprom_busy_wait();
                uint8_t bt = eeprom_read_byte(&eepromParams.time_zone_shift);
                for (uint8_t i = 0; i < bt; i++) incHour(&rtc);
            
                RTClock.setTime(PCF8563_I2C_ADDR, &rtc);
            }

            return true;
        }
    } 
    else 
        return false;
}


uint8_t getModemInfo()
{
    Serial.print_p(GSM_MODEM_NAME);
    if (!GsmModem.init())
    {
        Serial.print_p(COMMA);
        Serial.print_p(NOT);
        Serial.println_p(DETECTED);
        return false;
    }
    Serial.print_p(COMMA);
    Serial.println_p(DETECTED);

    if (!GsmModem.getManufacturerID((char*)aux_buffer, AUX_BUF_LEN) > 0)
    {
        Serial.print_p(MANUFACTURER_ID);
        Serial.print_p(NOT);
        Serial.println_p(DETECTED);
        return false;
    }
    Serial.print_p(MANUFACTURER_ID);
    Serial.print_p(DASH);
    Serial.println((char*)aux_buffer);

    if (!GsmModem.getModelID((char*)aux_buffer, AUX_BUF_LEN) > 0)
    {
        Serial.print_p(TA_MODEL_ID);
        Serial.print_p(NOT);
        Serial.println_p(DETECTED);
        return false;
    }
    Serial.print_p(TA_MODEL_ID);
    Serial.print_p(DASH);
    Serial.println((char*)aux_buffer);

    if (!GsmModem.getTARevision((char*)aux_buffer, AUX_BUF_LEN) > 0)
    {
        Serial.print_p(TA_REVISION);
        Serial.print_p(NOT);
        Serial.println_p(DETECTED);
        return false;
    }
    Serial.print_p(TA_REVISION);
    Serial.print_p(DASH);
    Serial.println((char*)aux_buffer);

    if (!GsmModem.getIMEI((char*)aux_buffer, AUX_BUF_LEN) > 0)
    {
        Serial.print_p(IMEI);
        Serial.print_p(NOT);
        Serial.println_p(DETECTED);
        return false;
    }
    Serial.print_p(IMEI);
    Serial.print_p(DASH);
    Serial.println((char*)aux_buffer);

    Serial.print_p(GSM_NETWORK_STATUS);
    if (!GsmModem.waitForNetworkRegistration())
    {
        Serial.print_p(NOT);
        Serial.println_p(REGISTERED);
        return false;
    }
    Serial.println_p(REGISTERED);

    if (!GsmModem.getNetworkOperator((char*)aux_buffer, AUX_BUF_LEN) > 0)
    {
        Serial.print_p(NETWORK_OPERATOR);
        Serial.print_p(NOT);
        Serial.println_p(DETECTED);
        return false;
    }
    Serial.print_p(NETWORK_OPERATOR);
    Serial.print_p(DASH);
    Serial.println((char*)aux_buffer);

    return true;
}

uint8_t transmitEventTCP(uint8_t finalPowerOff, StrEventType* evnt)
{
    uint8_t res = true;
    uint8_t* apn_login[SENSOR_DESCR_LEN];
    uint8_t* apn_pwd[SENSOR_DESCR_LEN];

    apn_login[0] = 0;
    apn_pwd[0] = 0;
                
    if (!getModemInfo()) return false;

    if (!GsmModem.startSingleIPConnMode())
    {
        Serial.print_p(FAILED_TO);
        Serial.print_p(SET);
        Serial.println_p(SINGLE_IP_CONN_MODE);
        return false;
    }
    Serial.print_p(SINGLE_IP_CONN_MODE);
    Serial.print_p(IS);
    Serial.println_p(SET);

    eeprom_busy_wait();
    eeprom_read_block(aux_buffer, eepromParams.gsm_apn_name, SENSOR_DESCR_LEN);


    eeprom_busy_wait();
    eeprom_read_block(apn_login, eepromParams.gsm_apn_login, SENSOR_DESCR_LEN);

    eeprom_busy_wait();
    eeprom_read_block(apn_pwd, eepromParams.gsm_apn_pwd, SENSOR_DESCR_LEN);


    if (!GsmModem.setGPRSPDPAPN((char*)aux_buffer, (char*)apn_login, (char*)apn_pwd))
    {
        Serial.print_p(FAILED_TO);
        Serial.print_p(SET);
        Serial.println_p(APN);
        return false;
    }
    Serial.print_p(APN);
    Serial.print_p(IS);
    Serial.println_p(SET);

    if (GsmModem.actGPRSPDPContext())
    {
        Serial.print_p(GPRS_CONTEXT);
        Serial.print_p(IS);
        Serial.println_p(OPENED);

        if (GsmModem.getLocalIP((char*)aux_buffer, AUX_BUF_LEN) > 0)
        {
            Serial.print_p(LOCAL_IP);
            Serial.print_p(DASH);
            Serial.println((char*)aux_buffer);

            if (GsmModem.getGPRSPDPStatus((char*)aux_buffer, AUX_BUF_LEN) > 0)
            {
                Serial.print_p(GPRS_STATUS);
                Serial.print_p(DASH);
                Serial.println((char*)aux_buffer);

                char* tmpBufRef = (char*)aux_buffer;
                getHostPort(&tmpBufRef, SERVER_HTTP_URL_LEN);

                if (GsmModem.startIPConn(true, (char*)aux_buffer, 80))
                {
                    Serial.print_p(SOCKET_CONNECTION);
                    Serial.print_p(IS);
                    Serial.println_p(OPENED);

                    res = sendHTTPRequest((char*)aux_buffer, AUX_BUF_LEN, evnt);

                    /*if (!GsmModem.closeIPConn())
                    {
                        Serial.print_p(FAILED_TO);
                        Serial.print_p(CLOSE);
                        Serial.println_p(SOCKET_CONNECTION);
                    }else{
                        Serial.print_p(SOCKET_CONNECTION);
                        Serial.print_p(IS);
                        Serial.println_p(CLOSED);
                    }*/
                }else
                {
                    Serial.print_p(FAILED_TO);
                    Serial.print_p(OPEN);
                    Serial.println_p(SOCKET_CONNECTION);
                    res = false;
                }
            }else
            {
                Serial.print_p(FAILED_TO);
                Serial.print_p(GET);
                Serial.println_p(GPRS_STATUS);
                res = false;
            }
        }else
        {
            Serial.print_p(FAILED_TO);
            Serial.print_p(GET);
            Serial.println_p(LOCAL_IP);
            res = false;
        }
        if (!GsmModem.deactGPRSPDPContext())
        {
            Serial.print_p(FAILED_TO);
            Serial.print_p(CLOSE);
            Serial.println_p(GPRS_CONTEXT);
            res = false;
        }else
        {
            Serial.print_p(GPRS_CONTEXT);
            Serial.print_p(IS);
            Serial.println_p(CLOSED);
        }
    }else
    {
        Serial.print_p(FAILED_TO);
        Serial.print_p(OPEN);
        Serial.println_p(GPRS_CONTEXT);
        return false;
    }

    if (finalPowerOff)
        if (!GsmModem.powerOff()) GsmModem.forcedPowerOff();

    return res;
}

/*uint8_t transmitEventHTTP()
{
        Serial.print_p(GSM_MODEM_NAME);
        if (!GsmModem.init())
        {
                Serial.print_p(COMMA);
                Serial.print_p(NOT);
                Serial.println_p(DETECTED);
                return false;
        }
        Serial.print_p(COMMA);
        Serial.println_p(DETECTED);

        if (!GsmModem.getManufacturerID((char*)aux_buffer, AUX_BUF_LEN) > 0)
        {
                Serial.print_p(MANUFACTURER_ID);
                Serial.print_p(NOT);
                Serial.println_p(DETECTED);
                return false;
        }
        Serial.print_p(MANUFACTURER_ID);
        Serial.print_p(DASH);
        Serial.println((char*)aux_buffer);

        if (!GsmModem.getModelID((char*)aux_buffer, AUX_BUF_LEN) > 0)
        {
                Serial.print_p(TA_MODEL_ID);
                Serial.print_p(NOT);
                Serial.println_p(DETECTED);
                return false;
        }
        Serial.print_p(TA_MODEL_ID);
        Serial.print_p(DASH);
        Serial.println((char*)aux_buffer);

        if (!GsmModem.getTARevision((char*)aux_buffer, AUX_BUF_LEN) > 0)
        {
                Serial.print_p(TA_REVISION);
                Serial.print_p(NOT);
                Serial.println_p(DETECTED);
                return false;
        }
        Serial.print_p(TA_REVISION);
        Serial.print_p(DASH);
        Serial.println((char*)aux_buffer);

        if (!GsmModem.getIMEI((char*)aux_buffer, AUX_BUF_LEN) > 0)
        {
                Serial.print_p(IMEI);
                Serial.print_p(NOT);
                Serial.println_p(DETECTED);
                return false;
        }
        Serial.print_p(IMEI);
        Serial.print_p(DASH);
        Serial.println((char*)aux_buffer);

        Serial.print_p(GSM_NETWORK_STATUS);
        if (!GsmModem.waitForNetworkRegistration())
        {
                Serial.print_p(NOT);
                Serial.println_p(REGISTERED);
                return false;
        }
        Serial.println_p(REGISTERED);

        if (!GsmModem.getNetworkOperator((char*)aux_buffer, AUX_BUF_LEN) > 0)
        {
                Serial.print_p(NETWORK_OPERATOR);
                Serial.print_p(NOT);
                Serial.println_p(DETECTED);
                return false;
        }
        Serial.print_p(NETWORK_OPERATOR);
        Serial.print_p(DASH);
        Serial.println((char*)aux_buffer);


        eeprom_busy_wait();
        eeprom_read_block(aux_buffer, eepromParams.gsm_apn_name, SENSOR_DESCR_LEN);

        if (!GsmModem.openGPRSContext((char*)aux_buffer))
        {
                Serial.print_p(FAILED_TO);
                Serial.print_p(OPEN);
                Serial.println_p(GPRS_CONTEXT);
                return false;
        }

        Serial.print_p(GPRS_CONTEXT);
        Serial.print_p(IS);
        Serial.println_p(OPENED);

        if (!GsmModem.queryGPRSContext((char*)aux_buffer, AUX_BUF_LEN) > 0)
        {
                Serial.print_p(GPRS_CONTEXT);
                Serial.print_p(NOT);
                Serial.println_p(OPENED);
                return false;
        }
        Serial.print_p(DETECTED);
        Serial.print_p(IS);
        Serial.print_p(COLUMN);
        Serial.println((char*)aux_buffer);



        if (!GsmModem.httpInit())
        {
                Serial.print_p(FAILED_TO);
                Serial.print_p(INITIATE);
                Serial.print_p(HTTP);
                Serial.println_p(SERVICE);
                return false;
        }
        Serial.print_p(HTTP);
        Serial.print_p(SERVICE);
        Serial.print_p(IS);
        Serial.println_p(INITIATED);


        eeprom_busy_wait();
        eeprom_read_block(aux_buffer, eepromParams.server_HTTP_URL, SERVER_HTTP_URL_LEN);

        if (!GsmModem.httpSetURL((char*)aux_buffer))
        {
                Serial.print_p(FAILED_TO);
                Serial.print_p(SET);
                Serial.print_p(HTTP);
                Serial.println_p(URL);
                return false;
        }

        Serial.print_p(HTTP);
        Serial.print_p(URL);
        Serial.print_p(IS);
        Serial.println_p(SET);


        //Setting up HTTP parameters

        //char* paramBuf = (char*)aux_buffer;
        //fillTemperature(&paramBuf);
        //fillSubject(&paramBuf);
        //fillBody(&paramBuf);
        //fillDeviceID(&paramBuf);
        //fillSoftVersion(&paramBuf);
        sprintf_P((char*)aux_buffer, PSTR("XX-exttemp=24XX-subject=Test_subject&XX-body=Test_body&XX-deviceid=DLII&XX-softversion=V_1.0.0.1"));

        if (!GsmModem.httpSetData((char*)aux_buffer, strlen((char*)aux_buffer), 10000))
        {
                Serial.println_p(PSTR("Failed to set http data"));

        }else 
                Serial.println_p(PSTR("http data is set"));

        uint16_t responseLen;
        if (!GsmModem.httpAction(true, &responseLen))
        {
                Serial.print_p(FAILED_TO);
                Serial.print_p(SET);
                Serial.print_p(HTTP);
                Serial.println_p(ACTION);
                return false;
        }
        Serial.print_p(HTTP);
        Serial.print_p(ACTION);
        Serial.print_p(IS);
        Serial.println_p(SET);

        // TODO:httpRead() if necessary

        if (!GsmModem.httpTerm())
        {
                Serial.print_p(FAILED_TO);
                Serial.print_p(TERMINATE);
                Serial.print_p(HTTP);
                Serial.println_p(SERVICE);
        }else{
                Serial.print_p(HTTP);
                Serial.print_p(SERVICE);
                Serial.print_p(IS);
                Serial.println_p(TERMINATED);
        }


        if (!GsmModem.closeGPRSContext())
        {
                Serial.print_p(FAILED_TO);
                Serial.print_p(CLOSE);
                Serial.println_p(GPRS_CONTEXT);
        }else{
                Serial.print_p(GPRS_CONTEXT);
                Serial.println_p(IS);
                Serial.println_p(CLOSED);
        }
        if (!GsmModem.powerOff()) GsmModem.forcedPowerOff();
 
}
*/



