/*
  MicroSD.c - SDCard+FAT wrapper around Roland Riegel library for DefendLineII
  Copyright (c) 2010 Dmitry Pakhomenko.  All right reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "MicroSD.h"
// Initialize Class Variables //////////////////////////////////////////////////

// Constructors ////////////////////////////////////////////////////////////////

MicroSD::MicroSD(HardwareSerial* serial)
{
	ser = serial;
}

// Public Methods //////////////////////////////////////////////////////////////
uint8_t MicroSD::initCard()
{
	return sd_raw_init();
}

uint8_t MicroSD::initFilesystem(void)
{
	//setup sd card slot 
	if(!sd_raw_init())
	{
		ser->println_p(PSTR("MMC/SD initialization failed"));
		return 0;
	}
	ser->println_p(PSTR("MMC/SD successfully initialized"));
   /* open first partition */
    partition = partition_open(sd_raw_read,
 	   sd_raw_read_interval,
#if SD_RAW_WRITE_SUPPORT
       sd_raw_write,
       sd_raw_write_interval,
#else
       0,
	   0,
#endif
	   0
	);

	if(!partition)
	{
		/* If the partition did not open, assume the storage device
		* is a "superfloppy", i.e. has no MBR.
		*/
		partition = partition_open(sd_raw_read,
			sd_raw_read_interval,
#if SD_RAW_WRITE_SUPPORT
			sd_raw_write,
			sd_raw_write_interval,
#else
			0,
			0,
#endif
			-1
			);
		if(!partition)
		{
			ser->println_p(PSTR("opening partition failed"));
			return 0;
		}
	}

	/* open file system */
	fs = fat_open(partition);
	if(!fs)
        {
		ser->println_p(PSTR("opening filesystem failed"));
		return 0;
	}

	/* open root directory */
	struct fat_dir_entry_struct directory;
	fat_get_dir_entry_of_path(fs, "/", &directory);

	dd = fat_open_dir(fs, &directory);
	if(!dd)
	{
        ser->println_p(PSTR("opening root directory failed"));
		return 0;
	}

   	/* print some card information as a boot message */
   	//print_disk_info(fs);


	return 1;
}


void MicroSD::ls(void)
{
	/* print directory listing */
	struct fat_dir_entry_struct dir_entry;
	while(fat_read_dir(flashCard.dd, &dir_entry))
	{
		uint8_t spaces = sizeof(dir_entry.long_name) - strlen(dir_entry.long_name) + 4;
		ser->write((uint8_t*)&dir_entry.long_name, (size_t)strlen(dir_entry.long_name));
		ser->print(dir_entry.attributes & FAT_ATTRIB_DIR ? '/' : ' ');
		while(spaces--)
			ser->print(' ');
		ser->print(dir_entry.file_size, DEC);
        ser->println();
	}
}


uint8_t MicroSD::diskInfo(void)
{
    if(!fs)
        return 0;

    struct sd_raw_info disk_info;
    if(!sd_raw_get_info(&disk_info))
        return 0;

    ser->print_p(PSTR("manuf:  0x")); ser->println(disk_info.manufacturer, HEX);
    ser->print_p(PSTR("oem:    ")); ser->println((char*) disk_info.oem);
    ser->print_p(PSTR("prod:   ")); ser->println((char*) disk_info.product);
    ser->print_p(PSTR("rev:    ")); ser->println(disk_info.revision, HEX);
    ser->print_p(PSTR("serial: 0x")); ser->println(disk_info.serial, HEX);
    ser->print_p(PSTR("date:   ")); ser->print((uint16_t)disk_info.manufacturing_month); ser->print('/');
		ser->println((uint16_t)disk_info.manufacturing_year);

    ser->print_p(PSTR("size:   ")); ser->print((uint32_t)disk_info.capacity / 1024 / 1024); 
		ser->println_p(PSTR("MB"));

    ser->print_p(PSTR("copy:   ")); ser->println((uint16_t)disk_info.flag_copy);

    ser->print_p(PSTR("wr.pr.: ")); ser->print((uint16_t)disk_info.flag_write_protect_temp); 
		ser->print('/');	ser->println((uint16_t)disk_info.flag_write_protect); 

    ser->print_p(PSTR("format: ")); ser->println((uint16_t)disk_info.format);

    ser->print_p(PSTR("free:   ")); ser->print((uint32_t)fat_get_fs_free(fs)); 
		ser->print('/'); ser->println((uint32_t)fat_get_fs_size(fs));

    return 1;
}

uint8_t MicroSD::cd(char* dirName)
{
	/* change directory */
	struct fat_dir_entry_struct subdir_entry;
	if(findFileInDir(fs, dd, dirName, &subdir_entry))
	{
		struct fat_dir_struct* dd_new = fat_open_dir(fs, &subdir_entry);
		if(dd_new)
		{
			fat_close_dir(dd);
			dd = dd_new;
			return 1;
		}
	}

	ser->print_p(PSTR("directory not found: "));
	ser->print(dirName);
	ser->println();
	return 0;
}

uint8_t MicroSD::rm(char* fileName)
{
	struct fat_dir_entry_struct file_entry;
	if(findFileInDir(fs, dd, fileName, &file_entry))
	{
		if(fat_delete_file(fs, &file_entry))
		return 1;
	}

	ser->print_p(PSTR("error deleting file: "));
	ser->print(fileName);
	ser->println();
	return 0;
}

struct fat_file_struct* MicroSD::openFileInDir(struct fat_fs_struct* fs, struct fat_dir_struct* dd, 
	const char* name)
{
    struct fat_dir_entry_struct file_entry;
    if(!findFileInDir(fs, dd, name, &file_entry))
        return 0;

    return fat_open_file(fs, &file_entry);
}

uint8_t MicroSD::findFileInDir(struct fat_fs_struct* fs, struct fat_dir_struct* dd, 
	const char* name, struct fat_dir_entry_struct* dir_entry)
{
    while(fat_read_dir(dd, dir_entry))
    {
        if(strcmp(dir_entry->long_name, name) == 0)
        {
            fat_reset_dir(dd);
            return 1;
        }
    }
    return 0;
}


uint8_t MicroSD::mkdir(char* dirName)
{
	struct fat_dir_entry_struct dir_entry;
	if(!fat_create_dir(dd, dirName, &dir_entry))
	{
		ser->print_p(PSTR("error creating directory: "));
		ser->print(dirName);
		ser->println();
		return 0;
	}
	return 1;
}

uint8_t MicroSD::sync()
{
	if(!sd_raw_sync())
	{
		ser->println_p(PSTR("error syncing disk"));
		return 0;
	}
	return 1;
}
	
uint8_t MicroSD::touch(char* fileName)
{
	struct fat_dir_entry_struct file_entry;
	if(!fat_create_file(dd, fileName, &file_entry))
	{
		ser->print_p(PSTR("error creating file: "));
		ser->print(fileName);
		ser->println();
		return 0;
	}
	return 1;
}

uint8_t MicroSD::cat(char* fileName)
{
	/* search file in current directory and open it */
	struct fat_file_struct* fd = openFileInDir(fs, dd, fileName);
	if(!fd)
	{
		ser->print_p(PSTR("error opening "));
		ser->print(fileName);
		ser->println();
		return 0;
	}

	/* print file contents */
	uint8_t buffer[8];
	uint32_t offset = 0;
	uint8_t isText = 0;

	if(strstr_P(fileName, PSTR(".txt")) != NULL) isText = 1;

	intptr_t len;
	while((len = fat_read_file(fd, buffer, sizeof(buffer))) > 0)
	{
		if (isText)
		{
			ser->write((uint8_t*)buffer, len);
		}else{
			ser->print(offset, HEX);
			ser->print(':');
			for(uint8_t i = 0; i < 8; ++i)
			{
				ser->print(' ');
				ser->print(buffer[i], HEX);
			}
			ser->println();
			offset += 8;
		}
	}
	fat_close_file(fd);
	return 1;
}


uint8_t MicroSD::closeFilesystem(void)
{
   	/* close file system */
	fat_close(fs);

	/* close partition */
	if (!partition_close(partition))
	{
		ser->println_p(PSTR("error closing partition"));
		return 0;
	}
	return 1;
}


struct fat_file_struct* MicroSD::fileOpen(char* fileName)
{
	struct fat_file_struct* fd = flashCard.openFileInDir(flashCard.fs, flashCard.dd, fileName);
	if(!fd)
	{
		Serial.print_p(PSTR("error opening "));
		Serial.println(fileName);
		return 0;
	}
	return fd;
}

void MicroSD::fileClose(struct fat_file_struct* fd)
{
	fat_close_file(fd);
}

uint8_t MicroSD::fileWrite(struct fat_file_struct* fd, uint8_t* buffer, uint16_t len)
{
	intptr_t res = fat_write_file(fd, buffer, len);
	if(res != len)
	{
		ser->print_p(PSTR("error writing to file"));
		ser->println(res, DEC);
		return 0;
	}
	return 1;
}

uint8_t MicroSD::fileLen(char* fileName, uint32_t* fileSize)
{
	struct fat_dir_entry_struct file_entry;
	if(findFileInDir(fs, dd, fileName, &file_entry))
	{
        *fileSize = file_entry.file_size;
		return 1;
	}
	return 0;
}



// Preinstantiate Objects //////////////////////////////////////////////////////

MicroSD flashCard = MicroSD(&Serial);
