/*
  DefendLineII.cpp - main module
  Copyright (c) 2011 Dmitry Pakhomenko.  All right reserved.

  http://atmega.magictale.com

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "Utils.h"


volatile uint8_t termCmd;
volatile uint8_t ctrlC;
volatile uint8_t* aux_buffer[AUX_BUF_LEN];
volatile uint8_t mdmMode;
volatile uint8_t gpsMode;
volatile uint8_t cameraMode;
volatile uint8_t kbdInt;
volatile uint8_t sysState;
volatile uint16_t sysCntr;
volatile uint8_t refreshLCD;
volatile uint8_t rdyPage;
volatile uint8_t lastSensorStates;
volatile uint8_t currSensorStates;
volatile uint8_t dailyRepSent;

EepromParamsType eepromParams EEMEM = 
    {"Sensor 0", "Sensor 1", "Sensor 2", "Sensor 3", "Not defined", 
    "internet.net", "magictale.com/dl/dlLog.php", DEF_SENSORS_MSK, DEF_REPORT_HOUR,
     DEF_TIMEZONE, DEF_MODEM_POWER_OFF, DEF_MODEM_POWER_INTERVAL}; 

void (*boot_loader_start)(void) = (void(*)(void))0x1F000; 

#ifdef LCD
LiquidCrystal lcd(23, 24, 25, 26, 27, 28, 29);
//                RS  RW  EN  D0  D1  D2  D3
volatile uint8_t lcdBacklight;
volatile uint8_t lcdOnTime;
#endif

//NokiaLCD nokiaLcd(26, 24, 25, 23);
//		    mosi sclk cs rst



void getSensorDescr(uint8_t sensorStates, char* sensorDescr)
{
    //For now fill in no more than one sensor description
    //If more sensors triggered simultaneously sensorStates
    //should be analysed for detailed info.
    uint8_t idx;

    sensorDescr[0] = 0;
    for (idx = 0; idx < 5; idx++)
    {
        if (sensorStates & ( 1 << idx ) != 0)
        {
            if (idx == 4)
                sprintf_P(sensorDescr, TAMPERING);
            else
            {
                uint8_t* senDescAddr = eepromParams.sensor0_descr;
                senDescAddr += idx * SENSOR_DESCR_LEN;

                eeprom_busy_wait();
                eeprom_read_block(sensorDescr, (void*)senDescAddr, SENSOR_DESCR_LEN);
                return;
            }
        }
    }
}

uint8_t sendPicture(char* fileName, char* subject, char* body, char* detectorID)
{
    // Sending picture...
    uint8_t attempts;

    mdmMode = true;

    StrEventType strEvent;
    char* currTemp[12];
    char* tmpPntr = (char*)&currTemp;
    fillTemperature(&tmpPntr);

    char* softVer[16];
    tmpPntr = (char*)&softVer;
    fillSoftVer(&tmpPntr);
    
    char* na[4];
    char* deviceName[SENSOR_DESCR_LEN];

    if (subject == NULL)
    {
        sprintf_P((char*)&na, NA);
        strEvent.subject = (char*)na;
    }else
        strEvent.subject = subject;
    
    strEvent.exTemp = (char*)currTemp;

    if (body == NULL)
        strEvent.body = (char*)na;
    else
        strEvent.body = body;

    if (detectorID == NULL)
        strEvent.detectorID = (char*)na;
    else
        strEvent.detectorID = detectorID;

    eeprom_busy_wait();
    eeprom_read_block(deviceName, eepromParams.device_name, SENSOR_DESCR_LEN);
    //Check php script and database field on server side - it expects no more
    //than 4 characters in deviceID
    strEvent.deviceID = (char*)deviceName;


    //Temporary limits maximum field length - for database compatibility sake
    //on server side
    deviceName[4]  = 0;

    strEvent.softVer = (char*)softVer;
    
    strEvent.fileName = fileName;

    for (attempts = 0; attempts < MDM_REPEATS; attempts++)
    {
   
#ifdef LCD
        lcd.clear();
        lcd.print_p(SENDING_EVENT);
#endif

        if (transmitEventTCP(true, &strEvent))
        {
            Serial.println_p(SUCCESS);
#ifdef LCD
            lcd.setCursor(0, 1);
            lcd.print_p(OK);
#endif
            break;
        }else
        {
            Serial.println_p(FAILURE);
#ifdef LCD
            lcd.setCursor(0, 1);
            lcd.print_p(FAILURE);
#endif
        }
    }
    mdmMode = false;
}

inline void handleStatusEvent()
{
    RtcTimeType rtc;
    char buffer[30];
    uint16_t resCode;
    char* subject[sizeof(ALARM+1)];
    char status[STATUS_LEN];
    char* tmpPntr = (char*)&status;

    if (RTClock.init(PCF8563_I2C_ADDR) == 0)
    {
        RTClock.getTime(PCF8563_I2C_ADDR, &rtc);

        sprintf_P(buffer, PICTURE_NAME_FORMAT, rtc.year, rtc.month, 
            rtc.day, rtc.hours, rtc.minutes, rtc.seconds);

#ifdef LCD
        lcd.clear();
        lcd.print_p(TAKING_PICTURE);
#endif
        if (!takePicture(buffer, resCode))
        {
            Serial.print_p(FAILURE);
            Serial.println_p(TAKING_PICTURE);
#ifdef LCD
            lcd.setCursor(0, 1);
            lcd.print_p(FAILURE);
#endif
            return;
        }

        sprintf_P((char*)&subject, STATUS);
        fillStatus(&tmpPntr);

        sendPicture(buffer, (char*)subject, (char*)&status, NULL);
    }
}

inline void handleSensorEvent(uint8_t sensorStates)
{
    RtcTimeType rtc;
    char buffer[30];
    uint16_t resCode;
    char* subject[sizeof(STATUS+1)];
    char sensorDescr[SENSOR_DESCR_LEN];
    char* sensorID[4];

    if (RTClock.init(PCF8563_I2C_ADDR) == 0)
    {
        RTClock.getTime(PCF8563_I2C_ADDR, &rtc);

        sprintf_P(buffer, PICTURE_NAME_FORMAT, rtc.year, rtc.month, 
            rtc.day, rtc.hours, rtc.minutes, rtc.seconds);


#ifdef LCD
        lcd.clear();
        lcd.print_p(TAKING_PICTURE);
#endif

        if (!takePicture(buffer, resCode))
        {
            Serial.print_p(FAILURE);
            Serial.println_p(TAKING_PICTURE);

#ifdef LCD
            lcd.setCursor(0, 1);
            lcd.print_p(FAILURE);
#endif
            return;
        }

        sprintf_P((char*)&subject, ALARM);
        getSensorDescr(sensorStates, (char*)&sensorDescr);
        sprintf_P((char*)&sensorID, TWO_POS_HEX_FORMAT, sensorStates);

        sendPicture(buffer, (char*)subject, (char*)&sensorDescr, (char*)sensorID);
    }

}

inline volatile void onSerialReceive(unsigned char c, ring_buffer* buf)
{
    if (gpsMode)
    {
        //Redirect input data to GPS
        Serial2.print(c);
        //Serial.print(c);
    }else if (cameraMode)
    {
        //Redurect input data to Serial Camera
        Serial1.print(c);
    }else if (!termCmd)
        {
                Serial.print(c);
                if (c == 0xD) 
                {
                        buf->buffer[buf->head-1] = 0;
                        Serial.println();
                        termCmd = true;
                }
        else if (c == 0x3)      ctrlC = true;   
        }
}

inline volatile void onModemReceive(unsigned char c, ring_buffer* buf)
{
        if (mdmMode) Serial.print(c);
}

inline volatile void onGpsReceive(unsigned char c, ring_buffer* buf)
{
        if (gpsMode) Serial.print(c);
}

inline volatile void onCameraReceive(unsigned char c, ring_buffer* buf)
{
        if (cameraMode) Serial.print(c);
}


inline void updateSensorStates(void)
{
#ifdef LCD
        lcdBacklight = MAX_BRIGHTNESS;
        lcdOnTime = 10;
        analogWrite(LCD_BACKLIGHT_PIN, lcdBacklight);
#endif
        eeprom_busy_wait();
        uint8_t senMsk = eeprom_read_byte(&eepromParams.sensor_mask);

        if (currSensorStates == lastSensorStates &&
            (currSensorStates != (inb(PINK) & senMsk)))
        {
                currSensorStates = inb(PINK) & senMsk;
        }
}

SIGNAL(PCINT2_vect) 
{
        kbdInt = 2;
        updateSensorStates();
}

SIGNAL(PCINT1_vect) 
{
        kbdInt = 1;
        updateSensorStates();
}

SIGNAL(PCINT0_vect) 
{
        kbdInt = 0;
        updateSensorStates();
}

bool isTimeSync()
{
    RtcTimeType rtc;
    if (RTClock.getTime(PCF8563_I2C_ADDR, &rtc))
    {
        if (rtc.year > MIN_YEAR) return true;
    }
    return false;
}

//===============================================================
//===============================================================
/* main program starts here */
int main(void)
{    
        set_sleep_mode(SLEEP_MODE_IDLE);

        //temporary disabling interrupts from Timer3
        cbi(TIMSK3,TOIE3);

        cbi(LED_PORT, LED_ORANGE);
        sbi(LED_DDR, LED_ORANGE);

        cbi(LED_PORT, LED_BLUE);
        sbi(LED_DDR, LED_BLUE);

        cbi(CTS0_PORT, CTS0);
        sbi(CTS0_DDR, CTS0);

        cbi(RELAY_PORT, RELAY_0);
        sbi(RELAY_DDR, RELAY_0);

        cbi(RELAY_PORT, RELAY_1);
        sbi(RELAY_DDR, RELAY_1);

        //Hardware bug, allowing SD card to be interfaced;
        cbi(PORTB, 1);
        sbi(DDRB, DDB0);

        //Setting as input to handle events from buttons
        cbi(DDRK, BUTTON1);
        cbi(DDRK, BUTTON2);
        sbi(PORTK, BUTTON1);
        sbi(PORTK, BUTTON2);

        //Setting as input to handle event from sensors
        cbi(SENSOR_DDR, SENSOR0);
        cbi(SENSOR_DDR, SENSOR1);
        cbi(SENSOR_DDR, SENSOR2);
        cbi(SENSOR_DDR, SENSOR3);
        sbi(SENSOR_PORT, SENSOR0);
        sbi(SENSOR_PORT, SENSOR1);
        sbi(SENSOR_PORT, SENSOR2);
        sbi(SENSOR_PORT, SENSOR3);

        eeprom_busy_wait();
        uint8_t senMsk = eeprom_read_byte(&eepromParams.sensor_mask);

        lastSensorStates = inb(PINK) & senMsk;
        currSensorStates = lastSensorStates;

        uint8_t kbd = inb(PINK) & BTN1_MSK;
        if (kbd == 0) {

                Serial.println_p(PSTR("test point 1"));
                boot_loader_start();
        }

        //Enabling interrupts from sensors
        sbi(PCMSK2, PCINT16);
        sbi(PCMSK2, PCINT17);
        sbi(PCMSK2, PCINT18);
        sbi(PCMSK2, PCINT19);

        //Enabling interrupts from buttons
        sbi(PCMSK2, PCINT21);
        sbi(PCMSK2, PCINT22);

        kbdInt = 0;

        sbi(PCMSK1, PCINT8);

        PCIFR = 0xFF;
        sbi(PCICR, PCIE2);
        sbi(PCICR, PCIE1);
        sbi(PCICR, PCIE0);

        //GSM modem control
        //setting as input
        cbi(MODEM_DDR, STAT);
        sbi(MODEM_PORT, STAT);
        cbi(MODEM_DDR, MDM_CTS);
        sbi(MODEM_PORT, MDM_CTS);
        // setting as output
        cbi(MODEM_PORT, PWRK);
        sbi(MODEM_DDR, PWRK);
        cbi(MODEM_PORT, MDM_RTS);
        sbi(MODEM_DDR, MDM_RTS);

        PWRK_0;
        MDM_RTS_0;

        sei();
        init();

        sysState = sysRelayTest;
        termCmd = false;
        ctrlC = false;
        mdmMode = false;
        gpsMode = false;
        cameraMode = false;
        dailyRepSent = false;
        Wire.begin();           // join i2c bus (address optional for master)
        Serial.begin(115200);   // start serial for output

        Serial.attachRxEvent(onSerialReceive);

        Serial1.begin(115200);
        Serial1.attachRxEvent(onCameraReceive);
        Serial1.setOverrideFlag(false);

        Serial3.begin(57600);
        Serial3.attachRxEvent(onModemReceive);
        Serial3.setOverrideFlag(false);

        //sbi(PORTF, 2);
        //sbi(DDRF, 2);

        sbi(PORTD, 7);
        sbi(DDRD, 7);

        Serial2.begin(38400);
        Serial2.attachRxEvent(onGpsReceive);
        Serial2.setOverrideFlag(false);

        flashCard.initFilesystem();

        printHelp();
        printMenu();

#ifdef LCD
    lcdBacklight = MAX_BRIGHTNESS;
    lcdOnTime = 0;
    analogWrite(LCD_BACKLIGHT_PIN, lcdBacklight);

    // set up the LCD's number of rows and columns: 
    lcd.begin(16, 2);
    lcd.print_p(ASTERISK);
    lcd.print_p(PRODUCT_NAME);
    lcd.print_p(ASTERISK);

    lcd.setCursor(0, 1);
    lcd.print_p(FIRMWARE_REV);
    lcd.print_p(SPACE_CHAR);
    lcd.print_p(FIRMWARE_DATE);
#endif

#ifdef SND
        initSpeaker(SND_PIN);
        playStartSequence();
#endif

    IRInit(false);


    //Init ADC
    sbi(ADCSRA, ADEN);
    cbi(PRR0, 0);
    analogReference(INTERNAL);


        uint8_t lastKbdVal =  inb(PINK);


    //Setting up system timer (3)
    sbi(TIFR3, TOV3);
    TCCR3A = 0;
    outb(TCCR3B, (inb(TCCR3B) & ~TIMER_PRESCALE_MASK) | TIMER_CLK_DIV64);
    cbi(TCCR3B, WGM32);
    cbi(TCCR3B, WGM33);
    TCNT3 = SYS_TIMER_CNTR;
    sysCntr = 0;
    sbi(TIMSK3,TOIE3);

#ifdef LCD
    lcdOnTime = 60;
#endif







    //Serial.println_p(PSTR("Init display..."));
    //nokiaLcd.reset();
    //Serial.println_p(PSTR("Fill display..."));
    //nokiaLcd.fill(10, 10, 80, 20, 0x0F0);
    //nokiaLcd.fill(20, 20, 40, 40, 0xF00);

    //nokiaLcd.pixel(30, 30, 0x00F);
    //Serial.println_p(PSTR("Text output..."));


    //nokiaLcd.locate(0, 10);
    //nokiaLcd.write('D');
    //nokiaLcd.foreground(0xFF0);
    //nokiaLcd.print("Hello World!");


    /*Serial.println_p(PSTR("Hello World! on display"));

    uint8_t x, y;
    x = 0;
    y = 0;
    nokiaLcd.fill(x, y, 10, 132, 0xFFF);
    x += 10;
    y += 10;
    nokiaLcd.fill(x, y, 10, 132, 0x000);
    x += 10;
    y += 10;
    nokiaLcd.fill(x, y, 10, 132, 0xF00);
    x += 10;
    y += 10;
    nokiaLcd.fill(x, y, 10, 132, 0x0F0);
    x += 10;
    y += 10;
    nokiaLcd.fill(x, y, 10, 132, 0x00F);
    x += 10;
    y += 10;
    nokiaLcd.fill(x, y, 10, 132, 0x0FF);
    x += 10;
    y += 10;
    nokiaLcd.fill(x, y, 10, 132, 0xF0F);
    x += 10;
    y += 10;
    nokiaLcd.fill(x, y, 10, 132, 0xFF0);
    x += 10;
    y += 10;
    nokiaLcd.fill(x, y, 10, 132, 0xB22);
    x += 10;
    y += 10;
    nokiaLcd.fill(x, y, 10, 132, 0xFA0);
    x += 10;
    y += 10;
    nokiaLcd.fill(x, y, 10, 132, 0xF6A);
    x += 10;
    y += 10;

    nokiaLcd.locate(0, 1);
    nokiaLcd.foreground(0xFF0);
    nokiaLcd.print("*DefendLineII*");

*/
        





        //Main application loop
        while(1)
        {

        if (sysState == sysDispDeviceName)
        {
#ifdef LCD
            lcd.clear();
            lcd.print_p(DEVICE_NAME_FLD);
            lcd.setCursor(0, 1);
            eeprom_busy_wait();
            eeprom_read_block(aux_buffer, eepromParams.device_name, 16);
            lcd.print((char*)aux_buffer);
#endif

            sysState = sysDispDeviceNameWait;
        }else if (sysState == sysRTCTest)
        {
#ifdef LCD
            lcd.clear();
            lcd.print_p(RTC_CHIP_SHORT);
            lcd.setCursor(0, 1);
            if (Thermometer.init(DS1631_I2C_ADDR) == 0)
                    lcd.print_p(OK);
            else
            {
                lcd.print_p(FAILURE);
                sysCntr = 0;
            }
#endif
            sysState = sysRTCTestWait;
        }else if (sysState == sysTempSensorTest)
        {
#ifdef LCD
            lcd.clear();
            lcd.print_p(TEMP_SHORT);
            lcd.print_p(SPACE_CHAR);
            lcd.print_p(SENSOR_SHORT);
            lcd.setCursor(0, 1);
            if (RTClock.init(PCF8563_I2C_ADDR) == 0)
                lcd.print_p(OK);
            else
            {
                lcd.print_p(NOT);
                lcd.print_p(DETECTED);
                sysCntr = 0;
            }
#endif
            sysState = sysTempSensorTestWait;
        }else if (sysState == sysMicroSDTest)
        {
#ifdef LCD
            lcd.clear();
            lcd.print_p(MICRO_SD_NAME);
            lcd.setCursor(0, 1);
            if(sd_raw_init())
                lcd.print_p(OK);
            else
            {
                lcd.print_p(NOT);
                lcd.print_p(DETECTED);
                sysCntr = 0;
            }
#endif
            sysState = sysMicroSDTestWait;
        }else if (sysState == sysCameraTest)
        {
#ifdef LCD
            lcd.clear();
            lcd.print_p(SERIAL_CAMERA);
            lcd.print_p(DASH);
            lcd.setCursor(0, 1);

            uint16_t res;
                if (Camera.sync(&res))
                lcd.print_p(OK);
            else
            {
                lcd.print_p(NOT);
                lcd.print_p(DETECTED);
                sysCntr = 0;
            }
#endif
            sysState = sysCameraTestWait;
        }else if (sysState == sysModemInitTest)
        {
#ifdef LCD
            lcd.clear();
            lcd.print_p(GSM_MODEM_NAME);
            lcd.setCursor(0, 1);
#endif

                if (GsmModem.init())
            {
#ifdef LCD
                lcd.print_p(OK);
#endif
                sysState = sysModemInitTestWait;
            }else
            {
#ifdef LCD
                lcd.print_p(NOT);
                lcd.print_p(DETECTED);
                sysCntr = 0;
#endif
                //sysState = sysReady;
                sysState = sysGuarding;
            }

            
        }else if (sysState == sysModemNameTest)
        {
#ifdef LCD
            lcd.clear();
            lcd.print_p(MODEM_MODEL);
            lcd.setCursor(0, 1);

                GsmModem.getModelID((char*)aux_buffer, 16);
            stripCRLF((char*)aux_buffer);
            lcd.print((char*)aux_buffer);

#endif
            sysState = sysModemNameTestWait;
        }else if (sysState == sysModemIMEITest)
        {
#ifdef LCD
            lcd.clear();

            lcd.print_p(IMEI);
            lcd.print_p(DASH);

            lcd.setCursor(0, 1);

                GsmModem.getIMEI((char*)aux_buffer, 16);
            stripCRLF((char*)aux_buffer);
            lcd.print((char*)aux_buffer);

#endif
            sysState = sysModemIMEITestWait;
        }else if (sysState == sysModemNwkReg)
        {
#ifdef LCD
            lcd.clear();

            lcd.print_p(JOINING_NETWORK);

            lcd.setCursor(0, 1);

                if (GsmModem.waitForNetworkRegistration())
            {
                lcd.print_p(OK);
                GsmModem.enableRTC();
            }else
            {
                lcd.print_p(FAILURE);
                sysCntr = 0;
            }
#endif
            
            RtcTimeType rtc;            
            if (GsmModem.getRTC(&rtc))
            {
                RTClock.setTime(PCF8563_I2C_ADDR, &rtc);

                sysState = sysModemNwkRegWait;
            }else
            {
                //Prevent the timer from switching modem off
                //sysState = sysReady;
                sysState = sysGuarding;
            }

        }

        else if (sysState == sysReady || sysState == sysGuarding)
        {

#ifdef LCD
            if (refreshLCD)
            {
                char* aux_ptr;
                char** buf = &aux_ptr;

                if (rdyPage)
                {
                    int adcVal;
                    //'Mode: inactive /'
                    //'Mode: guarding  '
                    //'M:10.7V, B:10.0V'

                    lcd.clear();

                    lcd.print_p(MODE);
                    lcd.print_p(COLUMN);
                    lcd.print_p(SPACE_CHAR);
                    if (sysState == sysGuarding)
                        lcd.print_p(GUARDING);
                    else
                        lcd.print_p(INACTIVE);

                    lcd.setCursor(0, 1);


                    aux_ptr = (char*)&aux_buffer;
                    buf = &aux_ptr;

                    adcVal = analogRead(PMON_PIN);
                    *buf += sprintf_P(*buf, MAIN_PWR_FLD);
                    fillVoltage(buf, get_voltage(adcVal));

                    adcVal = analogRead(BMON_PIN);
                    *buf += sprintf_P(*buf, BAT_PWR_FLD);
                    fillVoltage(buf, get_voltage(adcVal));


                    lcd.print((char*)aux_buffer);

                }else
                {
                    //'03.02.2011 22:40'
                    //'T: +21.5 C'

                    RtcTimeType rtc;
 
                    aux_ptr = (char*)&aux_buffer;
                    buf = &aux_ptr;

                    /*if (!isTimeSync() && GsmModem.getRTC(&rtc))
                    {
                                    if (rtc.year > MIN_YEAR)
                                    {   
                            RTClock.setTime(PCF8563_I2C_ADDR, &rtc);
                            PWRK_1;
                            _delay_ms(500);
                            _delay_ms(500);
                            PWRK_0;
                        }
                        //GsmModem.enableRTC();
                    }*/


                    *buf += sprintf_P(*buf, TEMP_SHORT);
                    *buf += sprintf_P(*buf, COLUMN);
                    fillTemperature(buf);
                    *buf += sprintf_P(*buf, C_UNITS);

                    lcd.clear();
                    lcd.setCursor(0, 1);
                    lcd.print((const char*)aux_buffer);
    
                    aux_ptr = (char*)&aux_buffer;
                    buf = &aux_ptr;
                    lcd.setCursor(0, 0);
    
                            RTClock.getTime(PCF8563_I2C_ADDR, &rtc);
    
                            *buf += sprintf_P(*buf, TWO_POS_HEX_FORMAT, rtc.day);
                    *buf += sprintf_P(*buf, SLASH);
                            *buf += sprintf_P(*buf, TWO_POS_HEX_FORMAT, rtc.month);
                    *buf += sprintf_P(*buf, SLASH);
                            *buf += sprintf_P(*buf, FOUR_POS_HEX_FORMAT, rtc.year);
                    *buf += sprintf_P(*buf, SPACE_CHAR);

                            *buf += sprintf_P(*buf, TWO_POS_HEX_FORMAT, rtc.hours);
                    *buf += sprintf_P(*buf, COLUMN);
                            *buf += sprintf_P(*buf, TWO_POS_HEX_FORMAT, rtc.minutes);
                            lcd.print((char*)aux_buffer);


                    if (sysState == sysGuarding)
                    {
                        eeprom_busy_wait();
                        uint8_t repHour = eeprom_read_byte(&eepromParams.report_hour);
                        if (repHour == rtc.hours && !dailyRepSent)
                        {
                            dailyRepSent = true;
                            //Sending daily report
                            handleStatusEvent();
                        }else if (dailyRepSent && rtc.hours != repHour)
                            dailyRepSent = false;
                    }
                }
                refreshLCD = false;
            }
#endif
        }
        if (sysState == sysGuarding)
        {
            if (currSensorStates != lastSensorStates)
            {
                Serial.println(currSensorStates, HEX);
                if (currSensorStates != 0)
                    handleSensorEvent(currSensorStates);
                lastSensorStates = currSensorStates;
            }
        }           

        waitForInput();
        IRHandle();


                /*kbd = inb(PINK);
                if (kbd != lastKbdVal)
                {
                        Serial.println(kbd, HEX);
                        lastKbdVal = kbd;
                }

                if (kbd & BTN1_MSK == 0)
                {
                        boot_loader_start();
                }*/

                if (kbdInt == 2)
                {
                        kbd = inb(PINK);
                        if (kbd != lastKbdVal)
                        {
                                //Serial.println(kbd, HEX);
//                              Serial.print_p(EXT_INTERRUPT);
//                if (sysState == sysGuarding)
//                                  handleSensorEvent();


                                lastKbdVal = kbd;
                        }
                        kbdInt = 0;
                }

                kbd = inb(PINK) & BTN1_MSK;
                if (kbd == 0) {
                        boot_loader_start();
                }

                //set_sleep_mode(SLEEP_MODE_STANDBY);
                sleep_mode();
        }
}


/* 10Hz system timer */
ISR(TIMER3_OVF_vect)
{
    TCNT3 = SYS_TIMER_CNTR;

    sysCntr++;
    //10 seconds counter
    if (sysCntr >= 100) sysCntr = 0;
   

    //Relay 0 and 1 startup test
    if (sysState == sysRelayTest)
    {
        if (sysCntr == 1) 
        {
            set_relay(0, 1);
            //And starting GSM modem simultaneously
            PWRK_1;
        }
        else if (sysCntr == 2) set_relay(1, 1);
        else if (sysCntr == 11) 
        {
            set_relay(0, 0);
            //Set PWK line to its initial state
            PWRK_0;
        }
        else if (sysCntr == 12) set_relay(1, 0);
        else if (sysCntr == 20) sysState = sysDispDeviceName;
        
    }else if (sysState == sysDispDeviceNameWait)
    {
        if (sysCntr == 40) 
            sysState = sysRTCTest;
    }
    else if (sysState == sysRTCTestWait)
    {
        if (sysCntr == 60) 
            sysState = sysTempSensorTest;
    }
    else if (sysState == sysTempSensorTestWait)
    {
        if (sysCntr == 80) 
            sysState = sysMicroSDTest;
    }
    else if (sysState == sysMicroSDTestWait)
    {
        if (sysCntr == 99) 
            sysState = sysCameraTest;
    }
    else if (sysState == sysCameraTestWait)
    {
        if (sysCntr >= 20 && sysCntr < 99) 
        {
            sysCntr = 0;
            sysState = sysModemInitTest;
        }
    }else if (sysState == sysModemInitTestWait)
    {
        if (sysCntr >= 20 && sysCntr < 99) 
        {
            sysCntr = 0;
            sysState = sysModemNameTest;
        }
    }else if (sysState == sysModemNameTestWait)
    {
        if (sysCntr >= 20 && sysCntr < 40) 
        {
            sysCntr = 0;
            sysState = sysModemIMEITest;
        }
    }else if (sysState == sysModemIMEITestWait)
    {
        if (sysCntr >= 20 && sysCntr < 40) 
        {
            sysCntr = 0;
            sysState = sysModemNwkReg;
        }
    }else if (sysState == sysModemNwkRegWait)
    {
        if (sysCntr >= 60 && sysCntr < 99) 
        {
            sysCntr = 0;
            sysState = sysModemPowerOff;
        }
    }else if (sysState == sysModemPowerOff)
    {
        if (sysCntr == 1) 
        {
            //And stopping GSM modem simultaneously
            //PWRK_1;
        }
        else if (sysCntr == 11) 
        {
            //Set PWK line to its initial state
            //PWRK_0;
            //sysState = sysReady;
            sysState = sysGuarding;
        }
    }

    //Handling blue LED
    if (sysState == sysGuarding || sysState == sysReady)
    {
        if (sysCntr == 50) rdyPage = !rdyPage;

        if (sysCntr == 0 || sysCntr == 2) LED_BLUE_1;
        else if (sysCntr == 1 || sysCntr == 3) LED_BLUE_0;
    }else{
        if (sysCntr & 1) LED_BLUE_1;
        else LED_BLUE_0;
    }

    //Handling orange LED
    if (sysState == sysGuarding)
    {
        if (sysCntr == 50 || sysCntr == 52) LED_ORANGE_1;
        else if (sysCntr == 51 || sysCntr == 53) LED_ORANGE_0;
    }else if (sysState == sysReady)
    {
        if (sysCntr > 50 && sysCntr <= 60) LED_ORANGE_1;
        else if (sysCntr == 61) LED_ORANGE_0;
    }

    //Handling LCD backlight
#ifdef LCD
    if ((sysCntr % 10 == 0))
    {
        if (sysState == sysGuarding || sysState == sysReady)
        {
            if (lcdOnTime > 0) lcdOnTime--;
        }
    }

    //Request LCD content refresh
    if ((sysCntr % 40 == 0)) refreshLCD = true;

    if (lcdOnTime == 0 && lcdBacklight > 0)
    {
        if (lcdBacklight > MAX_BRIGHTNESS / 2) lcdBacklight--;
        if (lcdBacklight > MAX_BRIGHTNESS / 4) lcdBacklight--;
        lcdBacklight--;
        analogWrite(LCD_BACKLIGHT_PIN, lcdBacklight);
    }
#endif
}

