/*
  DS1621.h - Digital Thermometer and Thermostat library for DefendLineII
  Copyright (c) 2010 Dmitry Pakhomenko.  All right reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "DS1621.h"

// Initialize Class Variables //////////////////////////////////////////////////

// Constructors ////////////////////////////////////////////////////////////////

DS1621::DS1621()
{
}

// Public Methods //////////////////////////////////////////////////////////////
uint8_t DS1621::init(uint8_t i2cAddr)
{
	// set a default configuration
	// (1-shot mode, T_OUT active high)
	return setConfig(i2cAddr, DS1621_CONFIG_1SHOT | DS1621_CONFIG_POL);
}

uint8_t DS1621::setConfig(uint8_t i2cAddr, uint8_t config)
{
	uint8_t buffer[2];
	// write the DS1621 configuration byte
	buffer[0] = DS1621_CMD_ACCESSCONFIG;
	buffer[1] = config;

	Wire.beginTransmission(i2cAddr);
	Wire.send(buffer, 2);
	return Wire.endTransmission();
}

void DS1621::startConvert(uint8_t i2cAddr)
{
	Wire.beginTransmission(i2cAddr);
	Wire.send(DS1621_CMD_STARTCONV);
	Wire.endTransmission();
}

void DS1621::stopConvert(uint8_t i2cAddr)
{
	Wire.beginTransmission(i2cAddr);
	Wire.send(DS1621_CMD_STOPCONV);
	Wire.endTransmission();
}

void DS1621::setTH(uint8_t i2cAddr, uint16_t value)
{
	Wire.beginTransmission(i2cAddr);
	Wire.send(DS1621_CMD_ACCESSTH);
	Wire.send((uint8_t)(value >> 8 & 0xFF));
	Wire.send((uint8_t)(value & 0xFF));
	Wire.endTransmission();
}

void DS1621::setTL(uint8_t i2cAddr, uint16_t value)
{
	Wire.beginTransmission(i2cAddr);
	Wire.send(DS1621_CMD_ACCESSTL);
	Wire.send((uint8_t)(value >> 8 & 0xFF));
	Wire.send((uint8_t)(value & 0xFF));
	Wire.endTransmission();
}

uint16_t DS1621::readTempReg(uint8_t i2cAddr, uint8_t cmd)
{
	uint8_t buffer[2];
	uint16_t T;

	// read the temperature value from the requested register
	Wire.beginTransmission(i2cAddr);
	Wire.send(&cmd, 1);
	Wire.endTransmission();

	Wire.requestFrom(i2cAddr, (uint8_t)2);

	buffer[0] = 0;
	buffer[1] = 0;
	if(2 <= Wire.available())    // if two bytes were received
	{
    	buffer[0] = Wire.receive();
		buffer[1] = Wire.receive();
  	}

	// pack bytes
	T = (uint16_t)((buffer[0]<<8) | buffer[1]);
	// return result
	return T;
}

uint16_t DS1621::readTemp(uint8_t i2cAddr)
{
	// read the Temperature register and return the result
	return readTempReg(i2cAddr, DS1621_CMD_READTEMP);
}


uint16_t DS1621::getTH(uint8_t i2cAddr)
{
	return readTempReg(i2cAddr, DS1621_CMD_ACCESSTH);
}

uint16_t DS1621::getTL(uint8_t i2cAddr)
{
	return readTempReg(i2cAddr, DS1621_CMD_ACCESSTL);
}


uint8_t DS1621::readByte(uint8_t i2cAddr, uint8_t cmd)
{
	uint8_t res = -1;

	// read the temperature value from the requested register
	Wire.beginTransmission(i2cAddr);
	Wire.send(&cmd, 1);
	Wire.endTransmission();

	Wire.requestFrom(i2cAddr, (uint8_t)1);

	if(1 <= Wire.available())    // if two bytes were received
	{
    	res = Wire.receive();
  	}
	return res;
}

uint8_t DS1621::getConfig(uint8_t i2cAddr)
{
	return readByte(i2cAddr, DS1621_CMD_ACCESSCONFIG);
}

uint8_t DS1621::getCountRemain(uint8_t i2cAddr)
{
	return readByte(i2cAddr, DS1621_CMD_READCNTR);
}

uint8_t DS1621::getCountPerC(uint8_t i2cAddr)
{
	return readByte(i2cAddr, DS1621_CMD_READSLOPE);
}


// Preinstantiate Objects //////////////////////////////////////////////////////

DS1621 Thermometer = DS1621();
