/*
  SerialCamera.h - C328 serial camera library for DefendLineII
  Copyright (c) 2010 Dmitry Pakhomenko.  All right reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef C328Camera_h
#define C328_h

#include <inttypes.h>

#define CAMERA_TIMEOUT 20 //reply timeout in hundreds of milliseconds
#define CAM_PREFIX 0xAA
#define CAM_INITIAL 0x01
#define CAM_GET_PICTURE 0x04
#define CAM_SNAPSHOT 0x05
#define CAM_SET_PACKAGE_SIZE 0x06
#define CAM_SET_BAUDRATE 0x07
#define CAM_RESET 0x08
#define CAM_POWER_OFF 0x09
#define CAM_DATA 0x0A
#define CAM_SYNC 0x0D
#define CAM_ACK 0x0E
#define CAM_NACK 0x0F
#define CAM_LIGHT_FREQ 0x13

#define CAM_CMD_LEN 6
#define CAM_MAX_SYNC_NUM 60
#define CAM_DEF_PACKAGE_SIZE 64

typedef enum
{
	CCT_2BGRAY 		= 0x1,
	CCT_4BGRAY 		= 0x2,
	CCT_8BGRAY 		= 0x3,
	CCT_12BCOLOR 	= 0x5,
	CCT_16BCOLOR 	= 0x6,
	CCT_JPEG 		= 0x7
} eCameraColorType;

typedef enum
{
	CPR_80X60		= 0x1,
	CPR_160X120		= 0x3
} eCameraPreviewRes;

typedef enum
{
	CJR_80X64		= 0x1,
	CJR_160X128		= 0x3,
	CJR_320X240		= 0x5,
	CJR_640X480		= 0x7
} eCameraJPEGRes;

typedef enum
{
	CST_COMPRESSED = 0x0,
	CST_UNCOMPRESSED = 0x1
} eCameraSnapshotType;

typedef enum
{
	CPT_SHAPSHOT 	= 0x1,
	CPT_PREVIEW		= 0x2,
	CPT_JPEG_PREVIEW	= 0x5
} eCameraPictureType;

typedef enum
{
	CBD_7200 		= 0x1FF,
	CBD_9600		= 0x1BF,
	CBD_14400		= 0x17F,
	CBD_19200		= 0x15F,
	CBD_28800		= 0x13F,
	CBD_38400		= 0x12F,
	CBD_57600		= 0x11F,
	CBD_115200		= 0x10F
} eCameraBaudRate;

typedef enum
{
	CRT_WHOLE_SYSTEM = 0x0,
	CRT_STATE_MACHINES = 0x1
} eCameraResetType;

typedef enum
{
	CFT_50HZ		= 0x0,
	CFT_60HZ		= 0x1
} eCameraLFreqType;


class SerialCamera
{
  private:
  	HardwareSerial* cam;
  	HardwareSerial* dbg;
	volatile uint16_t currPackageSize;
  public:
  	SerialCamera(HardwareSerial* cam, HardwareSerial* dbg);

	uint8_t getReply(uint8_t cmdId, uint16_t* result);
	uint8_t singleSync(uint16_t* result);
	void ack(uint8_t cmdId, uint8_t ackCntr, uint16_t packId);
	uint8_t sync(uint16_t* result);
	uint8_t initial(eCameraColorType colorType, eCameraPreviewRes rawRes, eCameraJPEGRes jpegRes, uint16_t* result);

//	uint8_t getPicture(eCameraPictureType pictType, uint16_t* result);

	uint8_t getPicture(eCameraPictureType pictType, uint16_t* result, 
		uint8_t* aux_buffer, const uint16_t buf_size, volatile void (*function)(uint8_t*, uint16_t,
		struct fat_file_struct*), struct fat_file_struct* fd );


	uint8_t snapshot(eCameraSnapshotType snapshotType, uint16_t skipFrameCntr, uint16_t* result);
	uint8_t setPackageSize(uint16_t packageSize, uint16_t* result);
	uint8_t setBaudRate(eCameraBaudRate divider, uint16_t* result);
	uint8_t reset(eCameraResetType resetType, uint8_t specialReset, uint16_t* result);
	uint8_t powerOff(uint16_t* result);
	uint8_t lightFrequency(eCameraLFreqType freqType, uint16_t* result);
};

extern SerialCamera Camera;

#endif

