/*
  C328Camera.cpp - C328 serial camera library for DefendLineII
  Copyright (c) 2010 Dmitry Pakhomenko.  All right reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <util/delay.h>

#include "hardwareserial.h"
#include "C328Camera.h"
#include "Messages.h"


// Initialize Class Variables //////////////////////////////////////////////////

// Constructors ////////////////////////////////////////////////////////////////

SerialCamera::SerialCamera(HardwareSerial* camera, HardwareSerial* debug)
{
        cam = camera;
        dbg = debug;
}

// Public Methods //////////////////////////////////////////////////////////////
uint8_t SerialCamera::getReply(uint8_t cmdId, uint16_t* result)
{
        for (uint8_t i = 0; i < CAMERA_TIMEOUT; i++)
        {
                _delay_ms(10);
                if (cam->available() >= CAM_CMD_LEN) break;
        }

        if (cam->available() >= CAM_CMD_LEN && cam->read() == CAM_PREFIX)
        {
                uint8_t cmd = cam->read();
                //command id/0
                uint8_t recCmdId = cam->read();
                if (recCmdId != cmdId) return false;

                //ACK/NACK counter - not used
                cam->read();
                //Package id (if presented)/Err number
                *result = cam->read() | cam->read()<<8;

                switch (cmd)
                {
                        case CAM_ACK: 
                                return true;

                        case CAM_NACK:
                                return false;
                }
        }
        cam->flush();
        return false;
}

uint8_t SerialCamera::singleSync(uint16_t* result)
{
        cam->flush();

        cam->write(CAM_PREFIX);
        cam->write(CAM_SYNC);
        //Par1
        cam->write((uint8_t)0);
        //Par2
        cam->write((uint8_t)0);
        //Par3
        cam->write((uint8_t)0);
        //Par4
        cam->write((uint8_t)0);

        return getReply(CAM_SYNC, result);
}


void SerialCamera::ack(uint8_t cmdId, uint8_t ackCntr, uint16_t packId)
{
        cam->write(CAM_PREFIX);
        cam->write(CAM_ACK);
        //Par1
        cam->write(cmdId);
        //Par2
        cam->write(ackCntr);
        //Par3
        cam->write((uint8_t)(packId & 0xFF));
        //Par4
        cam->write((uint8_t)(packId >> 8 & 0xFF));
}

uint8_t SerialCamera::sync(uint16_t* result)
{
        currPackageSize = CAM_DEF_PACKAGE_SIZE;

        reset(CRT_WHOLE_SYSTEM, true, result);
        _delay_ms(10);

        for (int i = 0; i < CAM_MAX_SYNC_NUM; i++)
        {
                if (singleSync(result))
                {
                        //Waiting for SYNC from camera
                        for (uint8_t i = 0; i < CAMERA_TIMEOUT; i++)
                        {
                                _delay_ms(10);
                                if (cam->available() >= CAM_CMD_LEN) break;
                        }

                        if (cam->available() >= CAM_CMD_LEN && cam->read() == CAM_PREFIX)
                        {
                                uint8_t cmd = cam->read();
                                uint8_t fail = false;
                                uint8_t ackCntr = 0;

                                if (cmd == CAM_SYNC){
                                        for (int i = 0; i < 4; i++)
                                        {
                                                if (cam->read() != 0)
                                                {
                                                        fail = true;
                                                        break;
                                                }
                                        }
                                        if (!fail) 
                                        {
                                                ack(CAM_SYNC, ackCntr, 0);
                                                return true;
                                        }       
                                }
                        }
                        cam->flush();
                        return false;
                }
        }
        return false;
}

uint8_t SerialCamera::initial(eCameraColorType colorType, eCameraPreviewRes rawRes, eCameraJPEGRes jpegRes, uint16_t* result)
{
        cam->write(CAM_PREFIX);
        cam->write(CAM_INITIAL);
        //Par1
        cam->write((uint8_t)0);
        //Par2
        cam->write(colorType);
        //Par3
        cam->write(rawRes);
        //Par4
        cam->write(jpegRes);
 
        return getReply(CAM_INITIAL, result);
}

uint8_t SerialCamera::snapshot(eCameraSnapshotType snapshotType, uint16_t skipFrameCntr, uint16_t* result)
{
        cam->write(CAM_PREFIX);
        cam->write(CAM_SNAPSHOT);
        //Par1
        cam->write(snapshotType);
        //Par2
        cam->write((uint8_t)skipFrameCntr & 0xFF);
        //Par3
        cam->write((uint8_t)(skipFrameCntr >> 8 & 0xFF));
        //Par4
        cam->write((uint8_t)0);
 
        return getReply(CAM_SNAPSHOT, result);
}

uint8_t SerialCamera::setPackageSize(uint16_t packageSize, uint16_t* result)
{
        cam->write(CAM_PREFIX);
        cam->write(CAM_SET_PACKAGE_SIZE);
        //Par1
        cam->write(0x8);
        //Par2
        cam->write((uint8_t)(packageSize & 0xFF));
        //Par3
        cam->write((uint8_t)(packageSize >> 8 & 0xFF));
        //Par4
        cam->write((uint8_t)0);
 
        if (getReply(CAM_SET_PACKAGE_SIZE, result))
        {
                currPackageSize = packageSize;
                return true;
        }else
                return false;
}

uint8_t SerialCamera::setBaudRate(eCameraBaudRate divider, uint16_t* result)
{
        cam->write(CAM_PREFIX);
        cam->write(CAM_SET_BAUDRATE);
        //Par1
        cam->write((uint8_t)divider & 0xFF);
        //Par2
        cam->write((uint8_t)(divider >> 8 & 0xFF));
        //Par3
        cam->write((uint8_t)0);
        //Par4
        cam->write((uint8_t)0);
 
        if (getReply(CAM_SET_BAUDRATE, result))
        {       
                currPackageSize = CAM_DEF_PACKAGE_SIZE;
                return true;
        }
        return false;
}

uint8_t SerialCamera::reset(eCameraResetType resetType, uint8_t specialReset, uint16_t* result)
{
        cam->write(CAM_PREFIX);
        cam->write(CAM_RESET);
        //Par1
        cam->write(resetType);
        //Par2
        cam->write((uint8_t)0);
        //Par3
        cam->write((uint8_t)0);
        //Par4
        if (specialReset)
                cam->write(0xFF);
        else
                cam->write((uint8_t)0);
 
        //if (getReply(CAM_RESET, result)){
        //        currPackageSize = CAM_DEF_PACKAGE_SIZE;
                return true;
        //}
        //return false;
}

uint8_t SerialCamera::powerOff(uint16_t* result)
{
        cam->write(CAM_PREFIX);
        cam->write(CAM_POWER_OFF);
        //Par1
        cam->write((uint8_t)0);
        //Par2
        cam->write((uint8_t)0);
        //Par3
        cam->write((uint8_t)0);
        //Par4
        cam->write((uint8_t)0);
 
        return getReply(CAM_POWER_OFF, result);
}

uint8_t SerialCamera::lightFrequency(eCameraLFreqType freqType, uint16_t* result)
{
        cam->write(CAM_PREFIX);
        cam->write(CAM_LIGHT_FREQ);
        //Par1
        cam->write(freqType);
        //Par2
        cam->write((uint8_t)0);
        //Par3
        cam->write((uint8_t)0);
        //Par4
        cam->write((uint8_t)0);
 
        return getReply(CAM_LIGHT_FREQ, result);
}


uint8_t SerialCamera::getPicture(eCameraPictureType pictType, uint16_t* result, 
        uint8_t* aux_buffer, const uint16_t buf_size, 
        volatile void (*function)(uint8_t*, uint16_t, struct fat_file_struct*),
        struct fat_file_struct* fd )
{
        cam->write(CAM_PREFIX);
        cam->write(CAM_GET_PICTURE);
        //Par1
        cam->write(pictType);
        //Par2
        cam->write((uint8_t)0);
        //Par3
        cam->write((uint8_t)0);
        //Par4
        cam->write((uint8_t)0);

        if (getReply(CAM_GET_PICTURE, result))
        {
                uint16_t buf_len;

                for (uint8_t i = 0; i < CAMERA_TIMEOUT; i++)
                {
                        _delay_ms(10);
                        if (cam->available() >= CAM_CMD_LEN) break;
                }
                if (cam->read() == CAM_PREFIX)
                {
                        if (cam->read() == CAM_DATA)
                        {
                                //data type
                                uint8_t dataType = cam->read();
                                if (dataType != pictType) return false;
        
                                //Data length
                                uint32_t dataLen = cam->read() | cam->read()<<8 | cam->read()*65536;

                                ack(CAM_DATA, 0, 0);

                                uint16_t totalPackages = 1 + dataLen/(currPackageSize-6);

                                dbg->print_p(DATA_LEN);
                                dbg->println(dataLen, HEX);

                                dbg->print_p(PACKAGE_SIZE);
                                dbg->println(currPackageSize, HEX);

                                dbg->print_p(TOTAL_PACKAGES);
                                dbg->println(totalPackages, HEX);

                                dbg->println_p(GETTING_PICTURE);

                                uint8_t failed = false;
                                buf_len = 0;
                                for (uint16_t i = 0; i < totalPackages; i++)
                                {
                                        for (uint8_t j = 0; j < CAMERA_TIMEOUT; j++)
                                        {
                                                _delay_ms(10);
                                                if (cam->available() >= currPackageSize || i == (totalPackages - 1)) break;
                                        }

                                        if (cam->available() >= currPackageSize || i == (totalPackages - 1))
                                        {
                                                uint16_t packageId = cam->read() | cam->read()<<8;
                                                uint16_t dataSize = cam->read() | cam->read()<<8;
                                                uint8_t checkSum = (packageId & 0xFF) + (packageId >> 8 & 0xFF) +
                                                        (dataSize & 0xFF) + (dataSize >> 8 & 0xFF);

                                                for (uint16_t j = 0; j < dataSize; j++)
                                                {
                                                        uint8_t pictByte = cam->read();
                                                        checkSum += pictByte;

                                                        aux_buffer[buf_len] = pictByte;
                                                        buf_len++;

                                                        if (buf_len >= buf_size)
                                                        {
                                                                if (function) function(aux_buffer, buf_len, fd);
                                                                buf_len = 0;
                                                        }
                                                }
                                                uint16_t recCheckSum = cam->read() | cam->read()<<8;
                                                packageId++;
                                                if (checkSum == recCheckSum)
                                                {
                                                        ack(0, 0, packageId);
                                                }else{
                                                        failed = true;
                                                        break;
                                                }

                                        }else{
                                                failed = true;
                                                break;
                                        }
                                }

                                if (buf_len > 0)
                                {
                                        if (function) function(aux_buffer, buf_len, fd);
                                        buf_len = 0;
                                }

                                if (!failed) return true;
                        }
                }
                cam->flush();
                return false;
        }
        return false;
}


// Preinstantiate Objects //////////////////////////////////////////////////////

SerialCamera Camera = SerialCamera(&Serial1, &Serial);
