	TITLE           "RS232 Communications : Half Duplex : PIC16C6x/7x/8x"
	SUBTITLE        "Software Implementation : Interrupt Driven"

;*********************************************************************************************************
;                       Software Implementation Of RS232 Communications Using PIC16CXX
;                                              Half-Duplex 
;
;  These routines are intended to be used with PIC16C6X/7X family. These routines can be
;  used with processors in the 16C6X/7X family which do not have on board Hardware Async
;  Serial Port.
;  MX..
;
;  Description :
;               Half Duplex RS-232 Mode Is implemented in Software.
;               Both Reception & Transmission are Interrupt driven
;               Only 1 peripheral (RTCC) used for both transmission & reception
;               RTCC is used for both timing generation (for bit transmission & bit polling)
;               and Start Bit Detection in reception mode.
;               This is explained in more detail in the Interrupt Subroutine.
;               Programmable Baud Rate (speed depnding on Input Clock Freq.), programmable
;               #of bits, Parity enable/disable, odd/even parity is implemented.
;                       Parity & Framing errors are detected on Reception
;
;                               RS-232 Parameters
;
;The RS-232 Parameters are defined as shown below:
;
;               _ClkIn          :       Input Clock Frequency of the processor
;                                       (NOTE : RC Clock Mode Is Not Suggested due to wide variations)
;               _BaudRate       :       Desired Baud Rate. Any valid value can be used.
;                                       The highest Baud Rate achievable depends on Input Clock Freq.
;                                       300 to 4800 Baud was tested using 4 Mhz Input Clock
;                                       300 to 19200 Baud was tested using 10 Mhz Input Clock
;                                       Higher rates can be obtained using higher Input Clock Frequencies.
;                                       Once the _BaudRate & _ClkIn are specified the program
;                                       automatically selectes all the appropiate timings
;               _DataBits       :       Can specify 1 to 8 Bits.
;               _StopBits       :       Limited to 1 Stop Bit. Must set it to 1.
;               _PARITY_ENABLE  :       Parity Enable Flag. Set it to TRUE or FALSE. If PARITY
;                                       is used, then set it to TRUE, else FALSE. See "_ODD_PARITY" flag
;                                       description below
;               _ODD_PARITY     :       Set it to TRUE or FALSE. If TRUE, then ODD PARITY is used, else
;                                       EVEN Parity Scheme is used.
;                                       This Flag is ignored if _PARITY_ENABLE is set to FALSE.
;                
;
;  Usage :
;               An example is given in the main program on how to Receive & Transmit Data
;               In the example, the processor waits until a command is received. The command is interpreted
;               as the A/D Channel Number of PIC16C71. Upon reception of a command, the desired A/D channel
;               is selected and after A/D conversion, the 8 Bit A/D data is transmitted back to the Host.
;
;                       The RS-232 Control/Status Reg's bits are explained below :
;
;       "SerialStatus"          : RS-232 Status/Control Register
;
;       Bit 0   :       _txmtProgress   (1 if transmission in progress, 0 if transmission is complete)
;                                       After a byte is transmitted by calling "PutChar" function, the
;                                       user's code can poll this bit to check if transmission is complete.
;                                       This bit is reset after the STOP bit has been transmitted
;       Bit 1   :       _txmtEnable     Set this bit to 1 on initialization to enable transmission.
;                                       This bit can be used to Abort a transmission while the transmitter
;                                       is in progress (i.e when _txmtProgress = 1)
;       Bit 2   :       _rcvProgress    Indicates that the receiver is in middle of reception.It is reset when
;                                       a byte is received.
;       Bit 3   :       _rcvOver        This bit indicates the completion of Reception of a Byte. The user's
;                                       code can poll this bit after calling "GetChar" function. Once "GetChar"
;                                       function is called, this bit is 1 and is set to 0 after reception of
;                                       a complete byte (parity bit if enabled & stop bit)
;       Bit 4   :       _ParityErr      A 1 indicates Parity Error on Reception (for both even & odd parity) 
;       Bit 5   :       _FrameErr       A 1 indicates Framing Error On Reception
;
;       Bit 6   :       _unused_        Unimplemented Bit
;
;       Bit 7   :       _parityBit      The 9 th bit of transmission or reception (status of PARITY bit
;                                       if parity is enabled)
;
;       To Transmit A Byte Of Data :
;                       1) Make sure _txmtProgress & _rcvOver bits are cleared
;                       2) Load TxReg with data to be transmitted
;                       3) CALL  PutChar function                   
;
;       To Receive A Byte Of Data :
;                       1) Make sure _txmtProgress & _rcvOver bits are cleared
;                       2) CALL GetChar function
;                       3) The received Byte is in TxReg after _rcvOver bit is cleared
;
;
;
;       Program:          RS232.ASM 
;       Revision Date:   
;	                  May 17,1994 Scott Fink    (Rev 2)
;	                        	Corrected 7 bit and parity operation, corrected stop bit generation, corrected
;                       		receive prescaler settings.  Protected against inadvertant WDT reset.
;                         1-16-97      Compatibility with MPASMWIN 1.40
;
;			  12-04-01     check for FF header in RangeCount and re-sync, remove
;				       ALARM_ON in Calibrate
;
;*********************************************************************************************************

		Processor       16F84
		Radix   DEC
		EXPAND

		include         "p16F84.inc"

;__CONFIG	3FF1h
__CONFIG _CP_OFF & _PWRTE_ON & _WDT_OFF  & _XT_OSC


_ResetVector	set	0x00
_IntVector	set	0x04

TRUE	equ	1
FALSE	equ	0

;*********************************************************************************************************
;                               Setup RS-232 Parameters
;*********************************************************************************************************

_ClkIn          equ     4000000         ; Input Clock Frequency is 4 Mhz
_BaudRate       set     1200            ; Baud Rate (bits per second) is 1200
_DataBits       set     8               ; 8 bit data, can be 1 to 8
_StopBits       set     1               ; 1 Stop Bit, 2 Stop Bits is not implemented

#define _PARITY_ENABLE  FALSE           ; NO Parity
#define _ODD_PARITY     FALSE           ; EVEN Parity, if Parity enabled
#define _USE_RTSCTS     FALSE           ; NO Hardware Handshaking is Used

		include         "rs232.h"

;*********************************************************************************************************
;

		ORG     _ResetVector
		goto    Start
;
	
		ORG     _IntVector
		goto    Interrupt
;
;*********************************************************************************************************
;                                       Main Program Loop
;
;*********************************************************************************************************

Start:
		clrf	ZeroCal		; used to zero IR sensor output
		clrf	BtnTemp		; setup button registers
		movlw	0x3F		; setup alarm level address in W
		READ_EEPROM		; data read in W
		movwf	BtnCount
		call	_INITDISP	; setup led display pins RA2:RA0
		call	InitPortB	; setup button pins, enable PORTB change interrupts
		call  	InitSerialPort	; setup RX pin
;
WaitForNextSel:
		GET_CHAR
;;;;;;;;;;
		movf	RxReg, W
		movwf   ManchL
		GET_CHAR
		movf	RxReg, W
		movwf	ManchH
		call	FromManch
		movf	Temp1, W
		movwf	RxReg
;;;;;;
		movlw 	0x30		; _ParityErr OR _FrameErr mask
		andwf 	SerialStatus, F
		btfss	STATUS, Z	; skip retry if SerialStatus 0 (no errors)
		goto	WaitForNextSel
		clrf	SerialStatus
;
		movf	RxReg, W
		xorlw	0xFF		; skip 0xFF packet header
		btfsc	STATUS, Z
		;sublw	0xC8		; skip display if > 200 (in-exact match) 
		;btfss	STATUS, C
		goto  	WaitForNextSel  ; wait for next selection (command from Serial Port)
;
		movf	RxReg,W	
		movwf	cp1		; range sample 1
		GET_CHAR
;;;;;;;;;;
		movf	RxReg, W
		movwf   ManchL
		GET_CHAR
		movf	RxReg, W
		movwf	ManchH
		call	FromManch
		movf	Temp1, W
		movwf	RxReg
;;;;;;
		movf	RxReg,W
		movwf	cp2		; range sample 2
		GET_CHAR
;;;;;;;;;;
		movf	RxReg, W
		movwf   ManchL
		GET_CHAR
		movf	RxReg, W
		movwf	ManchH
		call	FromManch
		movf	Temp1, W
		movwf	RxReg
;;;;;;
		movf	RxReg,W
		movwf	cp3		; range sample 3
		call	ChkFix		; correct bit errors using cp1,cp2,cp3
		movwf	RangeCount	; corrected GP2D02 output
;
		GET_CHAR
;;;;;;;;;;
		movf	RxReg, W
		movwf   ManchL
		GET_CHAR
		movf	RxReg, W
		movwf	ManchH
		call	FromManch
		movf	Temp1, W
		movwf	RxReg
;;;;;;
		movf	RxReg,W	
		movwf	cp1		; temp sample 1
		GET_CHAR
;;;;;;;;;;
		movf	RxReg, W
		movwf   ManchL
		GET_CHAR
		movf	RxReg, W
		movwf	ManchH
		call	FromManch
		movf	Temp1, W
		movwf	RxReg
;;;;;;
		movf	RxReg,W
		movwf	cp2		; temp sample 2
		GET_CHAR
;;;;;;;;;;
		movf	RxReg, W
		movwf   ManchL
		GET_CHAR
		movf	RxReg, W
		movwf	ManchH
		call	FromManch
		movf	Temp1, W
		movwf	RxReg
;;;;;;
		movf	RxReg,W
		movwf	cp3		; temp sample 3
		call	ChkFix		; correct bit errors using cp1,cp2,cp3
		movwf	SensorTemp	; corrected DS1820 output
;
		movf	RangeCount,W	; setup for range display
;		xorlw	0xFF		; skip 0xFF packet header
;		btfsc	STATUS, Z
;		goto  	WaitForNextSel  ; wait for next selection (command from Serial Port)
		movwf	LastRead	; save reading for calibration
		call	Calibrate	; apply linearization  should rename !
		call	Compare		; see if measurement >= set point

		call	BCD_DISPLAY	; LED display byte received		
		PAUSE_1SEC

		
; display temp
		movf	SensorTemp, W
		;xorlw	0xFF		; skip display if 0xFF header
		;btfsc	STATUS, Z	; so we don't try to calibrate the temp
		sublw	0xC8		; skip display if > 200 (in-exact match) 
		btfss	STATUS, C
		goto  	WaitForNextSel  ; wait for next selection (command from Serial Port)

		movf	SensorTemp,W
;;proto;;	
		call	BCD_DISPLAY	; LED display byte received
		PAUSE_1SEC

		goto	WaitForNextSel	; wait for next selection (command from Serial Port)

;*********************************************************************************************************
;                               RS-232 Routines
;;*********************************************************************************************************
;                               Interrupt Service Routine
;
; Only RTCC Inturrupt Is used. RTCC Inturrupt is used as timing for Serial Port Receive
;
;       Reception :
;                       When put in receive mode, RTCC is setup for external clock mode (FALLING EDGE)
;                       and preloaded with 0xFF. When a Falling Edge is detected on RTCC Pin, RTCC 
;                       rolls over and an Interrupt is generated (thus Start Bit Detect). Once the start
;                       bit is detected, RTCC is changed to INTERNAL CLOCK mode and RTCC is preloaded
;                       with a certain value for regular timing interrupts to Poll RTCC Pin (i.e RX pin). 
;
;*********************************************************************************************************

Interrupt:
		;btfss   INTCON,T0IF	; check if TMR0 interrupt flag set
		;retfie                 ; other interrupt, simply return & enable GIE
;
; Save Status On INT : WREG & STATUS Regs
;
		movwf   SaveWReg
		swapf   STATUS,W        ; affects no STATUS bits : Only way OUT to save STATUS Reg ?????
		movwf   SaveStatus
;
					; has a button press
		btfsc   INTCON,RBIF     ; changed the state of one of the RB7:RB4 pins?
		call    ButtonDown    	; yes then service
;
		btfsc   _rcvProgress	; set when RX start bit detected
		goto    _RcvNextBit     ; Receive Next Bit
		goto    _SBitDetected   ; Must be start Bit     
;
RestoreIntStatus:
		swapf   SaveStatus,W
		movwf   STATUS          ; restore STATUS Reg
		swapf   SaveWReg,F      ; save WREG
		swapf   SaveWReg,W	; restore WREG
		bcf     INTCON,T0IF	; clear TMR0 overflow interrupt flag
		retfie
;
;*********************************************************************************************************
;    
; Configure, RX_Pin (RTCC pin) as Input, which is used to poll data on reception.
; RX_Pin is defined in RS232.h
;
;*********************************************************************************************************
						
InitSerialPort:
		clrf    SerialStatus                      
		bsf     STATUS,RP0      ; Select Page 1 for TrisB access
		bsf     RX_Pin          ; set RX Pin As Input for reception
		bcf     STATUS,RP0	; select Page 0 for Port Access
		return


;
;*********************************************************************************************************
;
; Apply linearization
;

Calibrate:
		movlw	0x3E		; read zero cal
		READ_EEPROM		; from EEPROM
		movwf   ZeroCal

		movf	ZeroCal,F	; don't run this routine if
		btfsc	STATUS,Z	; calibrate button not pushed
		goto	EndCal		; (ZeroCal == 0)

		movf	ZeroCal,W	; set by button press
		subwf	RangeCount,F	; subtract initial reading to 0 display
		btfss	STATUS,C	; zero out if on edge of LSB changing and underflows
		clrf	RangeCount

		movf	RangeCount,W	; valid index to calibration data?
		sublw	0x3D		; only 62 (0-61) calibration data points  correct this!!
		btfss	STATUS,C
		goto	Crikees

		movf	RangeCount,W		; RangeCount now contains index to EEPROM addr
		READ_EEPROM		; EEPROM addr passed in W, data read returned in W

		movwf	RangeCount		; RnageCount now has linearized value for compare		
		goto	EndCal
	
Crikees:
		clrf	RangeCount		; too close to sensor, past calibration data, set display to 0
;		ALARM_ON		; make audable complaint

EndCal:
		RETURN


;
;*********************************************************************************************************
;
; Compare RangeCount with alarm level set point in BtnCount. If measured level >= set point, 
; turn on alarm.
;
Compare:	
		movf	BtnCount,W
		subwf	RangeCount,W		; if >= than
		btfsc	STATUS,C	; (C is Cnot for borrow)
		call	Alarm
		btfss	STATUS,C
		bcf	PORTB,1		; else turn off s/b MACRO?
		
		movf	RangeCount,W		; restore W
		return

Alarm:
		ALARM_ON
		return


;
;*********************************************************************************************************
;
; This routine checks which buttons are pressed.
; The result of user down/up button presses are stored
; in BtnCount. Waits till all keys have been
; released before returning form the service routine.
;

ButtonDown:
		bcf	INTCON,RBIE	; disable RB port change interrupt
		comf	PORTB,W		; read PORTB
		bcf	INTCON,RBIF	; clear RB port change interrupt flag
		movlw	.40		; do de-bounce for N mSecs
		call	delay
		comf	PORTB,W		; read port B again
		andlw	B'01110000'	; mask outputs
;
CountAgain:
		movwf	BtnTemp		; save in temp
		swapf	BtnTemp,F	; switch low and high nibble

		movf	LastRead,W 	; set calibration byte
		btfss	BtnTemp,2	; calibrate button?
		goto	IncDec
		movwf	ZeroCal
		;BCF 	STATUS, RP0	; bank 0
		MOVLW	0x3E		; zero cal address
		MOVWF	EEADR		; in EEPROM 
		WRITE_EEPROM ZeroCal	; save zero cal to EEPROM

IncDec:
		btfsc	BtnTemp,0
		incf	BtnCount,F	; increment count
		btfsc	BtnTemp,1
		decf	BtnCount,F	; decrement count 
		movf	BtnCount,W	; test for 00-FF
		xorlw	0xFF
		btfsc	STATUS,Z
		clrf	BtnCount	; if undeflow, set back to 0

		movf	BtnTemp,W	; is it calibrate? 
		andlw	B'00000100'		
		movwf	BtnTemp

		movf	BtnCount,W	; setup for display
		movf	BtnTemp,F	; don't display set point
		btfsc	STATUS,Z	; if calibrating
		;call	LCD_VAL		; show reading
		call	BCD_DISPLAY	; display set point count
;
		movlw	.255		; pause to see if button
		call	delay		; still down
		movlw	.255
		call	delay
		movlw	.100
		call	delay
		comf	PORTB,W		; read PORTB
		andlw	B'00110000'	; mask all but down/up
		btfss	STATUS,Z	; down/up key still pressed?
		goto	CountAgain	; inc or dec if button
					; still held down
;
		;BCF 	STATUS, RP0	; bank 0
		MOVLW	0x3F		; alarm level address
		MOVWF	EEADR		; in EEPROM 
		WRITE_EEPROM BtnCount	; save alarm level to EEPROM
;
		bcf   	INTCON,RBIF	; clear RB port change interrupt flag
		bsf   	INTCON,RBIE	; enable RB port change interrupt
		goto	RestoreIntStatus

;*********************************************************************************************************
;
; This sub-routine initializes PortB.
;
; pin assignments:	RB4	input, count down
;			RB5	input, count up
;			RB6	input, calibrate
;			RB1	output, alarm

InitPortB
		bsf	STATUS,RP0	; select bank 1
		movlw	B'01110000'	; RB4-RB6 inputs
		movwf 	TRISB	
		bcf   	OPTION_REG,NOT_RBPU  ; enable pull up
		bcf   	STATUS,RP0  	; select bank 0
		clrf  	PORTB          	; init port B
		bcf   	INTCON,RBIE     ; disable RB port change interrupt
		movf  	PORTB,W        	; read port B
		bcf   	INTCON,RBIF     ; clear RB port change interrupt flag
		bsf   	INTCON,RBIE     ; enable RB port change interrupt
		retfie                  ; enable global interrupts and return

;---------------------------------------------------------
; delay_1_ms subroutine
;
; This routine takes up 1ms of clock

delay_1_ms	movlw	0xC7		; Set the inner
		movwf	INNER		; count
inner_loop	nop			; 5 cycles loop
		nop			; until done
            	decfsz 	INNER,f
            	goto	inner_loop
            	return

;---------------------------------------------------------
; delay subroutine
;
; This routine stalls (W) ms

delay		movwf	OUTER		; loop (w) times
outer_loop  call	delay_1_ms	; at 1ms each
		decfsz  OUTER,f
		goto	outer_loop
            	return

;*********************************************************************************************************
;				Generate Parity for the Value in WREG
;
;  The parity bit is set in _parityBit (SerialStatus,7)
;  Common Routine For Both Transmission & Reception
;
;  Program Memory :	16 locations
;  Cycles	  :	72
;
;*********************************************************************************************************
  if _PARITY_ENABLE

GenParity:
	movwf	temp2				;save data
	movf	BitCount,W			;save bitcount
	movwf	temp1  
Parityloop
        rrf	temp2,F
	btfss	STATUS,C				;put data in carry bit
	goto	NotOne	
	xorlw	00h				;parity calculated by XORing all data bits
	goto	OneDone
NotOne
	xorlw	01h
OneDone
	decfsz	temp1,F
	goto	Parityloop			;decrement count
	movwf	temp1
; Parity bit is in Bit 0 of temp1
;
    if _ODD_PARITY
	bsf	_parityBit
	btfsc	temp1,0
        bcf	_parityBit
    else
	bcf	_parityBit
	btfsc	temp1,0
        bsf	_parityBit
    endif

	return
    endif    

;
;*********************************************************************************************************
;
; Error correction. Check 3 copies of the data word and
; return corrected data in W
;

ChkFix:
		movlw	.8
		movwf	lp
		movlw	.1
		movwf	mask
		clrf	finalcp

NextBit:
		clrf	vote
		movf 	mask,W
		andwf	cp1,W
		btfss	STATUS,Z
		incf	vote,F

		movf	mask,W
		andwf	cp2,W
		btfss	STATUS,Z
		incf	vote,F

		movf	mask,W
		andwf	cp3,W
		btfss	STATUS,Z
		incf	vote,F

		movlw	.2		; vote >= 2 ?
		subwf	vote,W
		btfsc	STATUS,C
		goto	IsSet
		goto	NotSet
IsSet:	
		movf	mask,W
		iorwf	finalcp,F
NotSet:
		bcf	STATUS,C	; ensure Carry clear for rotate
		rlf	mask,F
		decfsz lp,F
		goto	NextBit

		movf	finalcp,W
		return

;
; -------------------------------------------
; SUBROUTINE: CONVERT MANCHESTER WORD TO BYTE
; -------------------------------------------
; Data to convert is in ManchH and ManchL
; Data is returned in Temp1
; ManchH and ManchL remain same value
;
FromManch       movf ManchH,W
        movwf DataH
        movf ManchL,W
        movwf DataL

        movlw 8h                        ; 8 bit pairs to convert
        movwf BCount
;
; BIT PATTERN 10 - RESULT BIT = 1
; BIT PATTERN 01 - RESULT BIT = 0
;
DeManLoop       rlf DataL                       ; 1st bit determines result bit
        rlf DataH
        rlf Temp1

        rlf DataL                       ; skip unused 2nd bit
        rlf DataH

        decfsz BCount           ; do for 8 bit pairs
        goto DeManLoop
        return
	
;*********************************************************************************************************


	include "rcvr.asm"	; The Receiver Routines are in File "rcvr.asm"
	;include "lcd1ctrl.asm"
	include "leddis3.asm"	; home brew led display routines

;*********************************************************************************************************
;
; calibration data
;
	#include "sensor3p.asm"	; cooked calibration data for EEPROM

;*********************************************************************************************************

	END




