;---------------------------------------------------------
;  leddisplay3.asm
;
;  For MAX7219 home brew serial LED display.
;  This code makes the following assumptions about
;  the interface of the LED display to the 16C84:
;
;    - A<0> wired to MAX 7219 CLK
;    - A<1> wired to MAX 7219 DATA IN
;    - A<2> wired to MAX 7219 LOAD
;
;---------------------------------------------------------

CONSTANT	CLK=0
CONSTANT	DATAIN=1
CONSTANT	LOAD=2
CONSTANT	SHUTDOWN=0xC
CONSTANT	DECODE=0x9
CONSTANT	INTENSITY=0xA
CONSTANT	SCANLIMIT=0xB
CONSTANT	DISPLAYTEST=0xF
CONSTANT	DIG0=(0x1 << 4)
CONSTANT	DIG1=(0x2 << 4)
CONSTANT	DIG2=(0x3 << 4)

CONSTANT 	_MAX_VAR=2FH

_HBYTE		EQU	_MAX_VAR-0
_LBYTE		EQU	_MAX_VAR-1

_DISP		EQU	_MAX_VAR-2	; used by display
_TMP		EQU	_MAX_VAR-3	; routine

;_CLRCLK	EQU	_MAX_VAR-5
_LOOP		EQU	_MAX_VAR-6
_SAVE_W		EQU	_MAX_VAR-7
_SAVE_STAT	EQU	_MAX_VAR-8
_ADD		EQU	_MAX_VAR-9

;
; BCD constants and registers
;
LO_BYTE     	EQU 	_MAX_VAR-10
DEC_COUNT   	EQU 	_MAX_VAR-11
H_DIGIT		EQU	_MAX_VAR-12

;---------------------------------------------------------

            ORG	360H

;---------------------------------------------------------
; setup 16F84 output pins and MAX 7219 registers
;
; 

_INITDISP
		movwf	_SAVE_W		; save W and status
		swapf	STATUS, W
		movwf _SAVE_STAT


        	bcf 	STATUS, RP0 	; select bank 0
		clrf	PORTA		; port A with
					; clk and data
					; bit off 
		bsf	STATUS, RP0	; select bank 1
		movlw 	b'11111000'
		movwf	TRISA		; configure A<0:2>
					; as outputs 
		bcf	STATUS, RP0	; select bank 0


		movlw	SHUTDOWN
		movwf	_HBYTE
		movlw	0x1		; normal operation
		movwf	_LBYTE
		call	_WORDOUT

		movlw	DECODE
		movwf	_HBYTE
		movlw	0xF		; code B decode for digits 3-0
		movwf	_LBYTE
		call	_WORDOUT

		movlw	INTENSITY
		movwf	_HBYTE
		movlw	0x7		; 15/32 on time
		;movlw	0xF		; 31/32 on time
		movwf	_LBYTE
		call	_WORDOUT

		movlw	SCANLIMIT
		movwf	_HBYTE
		;movlw	0x2		; enable digits 0 1 2 
		movlw	0x7		; enable 8 digits to slow down scan rate
		movwf	_LBYTE
		movwf	_LBYTE
		call	_WORDOUT

		movlw	DISPLAYTEST
		movwf	_HBYTE
		clrf	_LBYTE		; normal operation
		call	_WORDOUT

		swapf	_SAVE_STAT, W	; restore status
		movwf	STATUS
		swapf	_SAVE_W, F
		swapf	_SAVE_W, W	; get back W without altering STATUS
_test
		;movlw	127
		;call	BCD_DISPLAY
		;nop
		;nop
		;goto    _test

		return

;---------------------------------------------------------
; display subroutine
; 
; This routine displays the W register in the LED display
; as a decimal number. 

_DISPLAY	
		movwf	_DISP		; store out number
		movwf  _ADD		; get digit address
		movlw	0xF0
		andwf	_ADD, f
		swapf	_ADD, f 
		movf	_ADD,W
		movwf	_HBYTE		; upper byte contains register address

		movlw	0x0F		; cleave off upper
		andwf	_DISP, f	; four bits
		movf	_DISP,W
		iorlw	b'10000000'	; OR in decimal point
		movwf	_LBYTE		; lower byte contains data


		call	_WORDOUT

		return

;---------------------------------------------------------
; _WORDOUT subroutine
; 
; This routine left shifts out a upper byte and lower byte.
; The upper byte is first, starting with bit 15.

_WORDOUT
		movf _HBYTE,W	; upper byte
		movwf	_TMP		
		bcf	STATUS, C	; set C to 0
		movlw 	8
		movwf 	_LOOP
_BITLOOP1
		rlf	_TMP, F
		btfsc	STATUS, C
		bsf	PORTA, DATAIN	; send a one
		btfss 	STATUS, C	;
		bcf 	PORTA, DATAIN  	; or a zero
		
		nop			; pause before
		nop			; clocking in

		bsf	PORTA, CLK	; toggle clk
		nop
		nop
		bcf	PORTA, CLK

		decfsz 	_LOOP, F
		goto	_BITLOOP1


		movf _LBYTE,W	; lower byte
		movwf	_TMP		
		bcf	STATUS, C	; set C to 0
		movlw 	8
		movwf 	_LOOP
_BITLOOP2
		rlf	_TMP, F
		btfsc	STATUS, C
		bsf	PORTA, DATAIN	; send a one
		btfss 	STATUS, C	;
		bcf 	PORTA, DATAIN  	; or a zero
		
		nop			; pause before
		nop			; clocking in

		bsf	PORTA, CLK	; toggle clk
		nop
		nop
		bcf	PORTA, CLK

		decfsz 	_LOOP, F
		goto	_BITLOOP2


		bsf	PORTA, LOAD ; load 16 bits
		nop
		nop
		bcf	PORTA, LOAD

		return

;---------------------------------------------------------



; BCD.ASM - adapted from BCD_1.ASM 
; 
; Illustrates how to convert a one byte natural binary quantity to BCD.
; This is limited to FF H or 255 decimal.  
;
; The result is output to a serial LCD.  If the hundreds character is 0
; it is suppressed.
;
; The natural binary quantity is passed to subroutine BCD_1 in variable
; LO_BYTE.  
;
; The algorithm is one of successively subtracting 100 from the one byte 
; natural binary while incrementing a variable DEC_COUNT in a BCD 
; fashion. This is then output to serial LCD.  If it is zero, a space is  
; ouput in place of the leading.
;
; The remainder is then decremented in a natural binary fashion while 
; incrementing DEC_COUNT in BCD.  DEC_COUNT is then output to the serial 
; LCD.
;
;
; Copyright, Peter H. Anderson, Morgan State University, July 3, '97

 
        
BCD_DISPLAY:
        MOVWF LO_BYTE   ; place it in LO_BYTE
BCD_1_HUNDS:
        CLRF DEC_COUNT
BCD_1_HUNDS1:
        MOVLW .100
        SUBWF LO_BYTE, F        ; LO_BYTE=LO_BYTE-100
        BTFSS STATUS, C         ; no borrow
        GOTO BCD_1_HUNDS2
        CALL INC_DEC_COUNT      ; if no borrow
        GOTO BCD_1_HUNDS1       ; keep going

BCD_1_HUNDS2:
        MOVF DEC_COUNT, W       ; get the count
	movwf H_DIGIT		; !0 = hundred digit flag
        BTFSC STATUS, Z         ; if not zero
        MOVLW 0xF               ; otherwise put in a blank
	IORLW	DIG0		; address 100's digit 0
	CALL _DISPLAY		; display 100's digit

        MOVLW .100
        ADDWF LO_BYTE, F        ; add back any falsely subtracted 100

BCD_1_TENS:
        CLRF DEC_COUNT
BCD_1_TENS1:
        MOVLW .10
        SUBWF LO_BYTE, F        ; LO_BYTE=LO_BYTE-10
        BTFSS STATUS, C         ; no borrow
        GOTO BCD_1_TENS2
        CALL INC_DEC_COUNT      ; if no borrow
        GOTO BCD_1_TENS1       	; keep going

BCD_1_TENS2:
        MOVF DEC_COUNT, W       ; get the count
        BTFSS STATUS, Z         ; if not zero
	goto SET_ADD
        MOVLW 0xF               ; put in a space

	movf H_DIGIT,F		; 
	btfsc STATUS,Z		; if a hundreds digit
	goto SET_ADD
	clrw			; put in a zero	digit
SET_ADD:
	IORLW	DIG1		; address 10's digit 1
	CALL _DISPLAY		; display tens digit

        MOVLW .10
        ADDWF LO_BYTE, F        ; add back any falsely subtracted 10

BCD_1_UNITS:
        CLRF DEC_COUNT          ; now for the units
BCD_1_UNITS1:
        MOVLW .1
        SUBWF LO_BYTE, F
        BTFSS STATUS, C         ; did not go from 00 to FF
        GOTO BCD_1_UNITS2       ; done with units
        
        CALL INC_DEC_COUNT      ; else increment and keep going
        GOTO BCD_1_UNITS1       

BCD_1_UNITS2:
        MOVF DEC_COUNT, W       ; get the number of units
	IORLW	DIG2		; address 1's digit 2
	CALL _DISPLAY		; display ones digit

        RETURN

INC_DEC_COUNT:                  ; increments counter in BCD fashion

        INCF DEC_COUNT, F
        MOVF DEC_COUNT, W
        ANDLW 0FH
        SUBLW .9                ; W = low nibble - 9
        BTFSC STATUS, C         ; low nibble is in range of A - F
        GOTO INC_DEC_COUNT_DONE

        MOVF DEC_COUNT, W
        ADDLW .6
        MOVWF DEC_COUNT

INC_DEC_COUNT_DONE:
        RETURN


 

