#include <EEPROM.h>

// IR DEFINITIONS
#define IRpin_PIN      PIND
#define IRpin          2

//#define MAXPULSE 5000
#define TIMEOUT 100000                  // the maximum pulse we'll listen for - 5 milliseconds and 100 ms for long held buttons
#define FUZZINESS 20                    // What percent we will allow in variation to match the same code     3 = 12.5%
#define MAX_PULSE_PAIRS 60              // maximum number of pulse pairs to store

// we will store up to 100 pulse pairs (this is -a lot-)
uint16_t pulses[MAX_PULSE_PAIRS][2]; // pair is high and low pulse
uint8_t currentpulse = 0; // index for pulses we're storing

// variables for storing times
unsigned long currMicros, lastMicros, diffMicros;
uint8_t hold = 0;    // how long has the button been held down

uint8_t buttonPos[2];       // row , col
uint8_t decodedVal = 0, prevDecodedVal = 0;
boolean readRemote = true;

// LED STRIP definitions
#define RED_PIN 10
#define GREEN_PIN 9
#define BLUE_PIN 11

#define MAX_BRIGHTNESS 255
#define MAX_EFFECT_DELAY 255

#define BLACK 225
#define WHITE 228
#define COLOR_START 0
#define FLASH_START 231
#define FADE_START 311
#define AUDIO_START 391
#define SETTINGS_START 396

uint8_t color[3], colorPos;        // red, green, blue
uint8_t brightness, effectDelay, sensitivity;
uint8_t effects = 1;                     // 00 0000  ed -cba  a: plain color  b: flash effect  c: smooth effect  d: music effect  e: settings in effect
uint8_t layer[5];                      // 0: color  1: flash  2: fade  3: music effect  4: settings

uint8_t state = 0;
boolean on = false;
boolean manualOverride = false;
boolean progMode = false;
boolean displayLED = false;

uint8_t effectCount, effectColorCount;
uint8_t effectLastColor[3], effectNextColor[3];
unsigned long effectLastChange = 0;

// Audio definitions
#define AUDIO_PIN A3
#define MAX_SENSITIVITY 255
#define AUDIO_ZERO 550                  // the analog value when no signal
#define AUDIO_SAMPLE_SIZE 30			// max 64
uint16_t audioMaxAmp = 0;
uint8_t audioCount = 0;
float audioBrightnessRatio = 0.0;
uint8_t audioZeroColor[3];

uint8_t clapCount = 0;
unsigned long audioLastPeak = 0;
boolean clap = true;

// Potentiometer definitions
#define POT1 A0
#define POT2 A1
#define POT3 A2


// Push Button definitions
#define PUSH_BUTT 5
#define DEBOUNCE_DELAY 50
#define PUSH_BUTT_HOLD_TIME 3000
int buttonState = HIGH, lastButtonState = HIGH;
unsigned long lastDebounceTime = 0;
boolean longPushButtHold = false;


// Indicator LED
#define INDIE_LED_PIN 6

uint8_t backupColorPos;
uint8_t backupBrightness, backupEffectDelay, backupSensitivity;
uint8_t backupEffects;
uint8_t backupState;
boolean backupOn;

uint8_t tempColorLayer, tempColorPos;
uint8_t tempEffectOrder[16];


void readIR()
{
  currMicros = micros();

  if(currentpulse < MAX_PULSE_PAIRS)            // stop recording if maximum buffer limit reached
  {
    diffMicros = currMicros - lastMicros;       // time since last pin change

    if( IRpin_PIN & (1 << IRpin) )              //high (no signal)
	{    
		pulses[currentpulse][0] = diffMicros;
		digitalWrite(INDIE_LED_PIN, LOW);
	}
    else
    {
      pulses[currentpulse++][1] = diffMicros;
	  
	  digitalWrite(INDIE_LED_PIN, HIGH);

      /*if(diffMicros > MAXPULSE)                                     // start recording again if there was no signal for longer than MAXPULSE milliseconds
        currentpulse = 0;*/

      if(currentpulse == 1)                                           //first signal pair completed
      {
        if(pulses[0][0] > 8000 && pulses[0][0] < 9600 && pulses[0][1] > 4200 && pulses[0][1] < 4530)                            // 8800, 4380 : start of signal
          hold = 0;                                                   // start hold time afresh
        else
          currentpulse = 0;                                           // invalid signal. start recording again
      }
      else if(currentpulse > 32)                                      // signal transmission finished
      {
        if(pulses[currentpulse-1][0] > 8000 && pulses[currentpulse-1][0] < 9600 && diffMicros > 2000 && diffMicros < 2400)      // 8800, 2200  : Button held down
        {
          hold++;                                                   // increase the time button is held down
          //currentpulse = 0;                                       // start recording afresh
        }
      }
    }

    lastMicros = currMicros;                                        // current time becomes the time of last signal value change
  }  
}

void setup(void)
{
	lastMicros = micros();
	attachInterrupt(0, readIR, CHANGE);
	
	pinMode(RED_PIN, OUTPUT);
	pinMode(GREEN_PIN, OUTPUT);
	pinMode(BLUE_PIN, OUTPUT);
	
	pinMode(INDIE_LED_PIN, OUTPUT);

	pinMode(PUSH_BUTT, INPUT);	
	
	displayRGB();
	
	Serial.begin(9600);
	//Serial.println("CLEARDATA");
	//Serial.println("LABEL,Time,x,amp");
	//  Serial.println("Ready to decode IR!!!");

	colorPos = 37;
	//int addr = colorPos*3 + COLOR_START;
	readColor(0, false);
	
	int addr = SETTINGS_START;
	brightness = EEPROM.read(addr++);
	effectDelay = EEPROM.read(addr++);
	sensitivity = EEPROM.read(addr);
}

void loop(void)
{
	int holdTime = decodeIR();

	if(holdTime!= -1)
		buttonActionsIR(holdTime);
	
	checkPushButton();

	switch(state)
	{
		case 0:	 	// Off
			if(clap)
				readAudio();
			break;
		case 1:		// On
			displayEffects();
			break;
		case 2:		// Set Color Override
		case 4:		// Save Color
			color[0] = analogRead(POT1) >> 2;
			color[1] = analogRead(POT2) >> 2;
			color[2] = analogRead(POT3) >> 2;
			break;
		case 3:		// Set Settings Override
			if(!displayLED)
				readAudio();
		case 5:		// Save Settings
			brightness = analogRead(POT1) >> 2;
			effectDelay = analogRead(POT2) >> 2;
			sensitivity = analogRead(POT3) >> 2;
			//Serial.println(brightness);
			if(on)
				displayEffects();
			break;
		case 6:		// Save FLASH Effect
		case 7:		// Save FADE Effect
			break;
		case 8:		// Save Music Effect
			displayEffects();
			break;
		case 9:		// Reset
			break;
	}

	displayRGB();
}

void checkPushButton()
{
	int currButtonState = digitalRead(PUSH_BUTT);
	unsigned long buttonHold = millis() - lastDebounceTime;
	
	if(currButtonState!= lastButtonState)
	{
		lastDebounceTime = millis();
		lastButtonState = currButtonState;
	}
	else if(buttonHold > DEBOUNCE_DELAY)
	{
		if(!longPushButtHold && buttonState == LOW && buttonHold > PUSH_BUTT_HOLD_TIME)
		{
			//Serial.println("Push Button (long)");
			switch(state)
			{
				case 0:	 	// Off
					gotoState(9);
					longPushButtHold = true;
					break;
				case 1:		// On
				case 2:		// Set Color Override
				case 3:		// Set Settings Override
					break;
				case 4:		// Save Color
					{
						int addr = colorPos*3 + COLOR_START;
						for(int i=0; i<3; i++)
							EEPROM.write(addr++, color[i]);
					}
					resultBlink(0);
					readColor(0, false);
					gotoState(backupState);
					longPushButtHold = true;
					break;
				case 5:		// Save Settings
					{
						int addr = layer[4]*3 + SETTINGS_START;
						EEPROM.write(addr++, brightness);
						EEPROM.write(addr++, effectDelay);
						EEPROM.write(addr++, sensitivity);
					}
					resultBlink(0);
					gotoState(backupState);
					longPushButtHold = true;
					break;
				case 6:		// Save FLASH Effect
					{
						int addr = layer[1]*16 + FLASH_START;
						for(int i=0; i<=tempEffectOrder[0]; i++)
							EEPROM.write(addr++, tempEffectOrder[i]);
					}
					resultBlink(0);
					readColor(0, false);
					gotoState(backupState);
					longPushButtHold = true;
					break;
				case 7:		// Save FADE Effect
					{
						int addr = layer[2]*16 + FADE_START;
						for(int i=0; i<=tempEffectOrder[0]; i++)
							EEPROM.write(addr++, tempEffectOrder[i]);
					}
					resultBlink(0);
					readColor(0, false);
					gotoState(backupState);
					longPushButtHold = true;
					break;
				case 8:		// Save Music Effect
					{
						int addr = layer[3] + AUDIO_START;
						EEPROM.write(addr++, tempColorPos);
					}
					resultBlink(0);
					readColor(1, false);
					gotoState(backupState);
					longPushButtHold = true;
					break;
				case 9:		// Reset
					break;
			}
		}
		
		if(currButtonState!= buttonState)
		{
			buttonState = currButtonState;
			if(!longPushButtHold)
			{
				if(buttonState == HIGH)
				{
					//Serial.println("Push Button (short)");
					switch(state)
					{
						case 0:	 	// Off
							//gotoState(2);
							digitalWrite(INDIE_LED_PIN, LOW);
							clapCount = 0;
							audioLastPeak = 0;
							clap = !clap;
							break;
						case 1:		// On
							gotoState(3);
							break;
						case 2:		// Set Color Override
							gotoState(backupState);
							break;
						case 3:		// Set Settings Override
							if(on)
								gotoState(1);
							else
								gotoState(2);
							break;
						case 4:		// Save Color
						case 5:		// Save Settings
							break;
						case 6:		// Save FLASH Effect
						case 7:		// Save FADE Effect
							if(tempEffectOrder[0] < 15)
							{
								tempEffectOrder[0]++;
								tempEffectOrder[ tempEffectOrder[0] ] = tempColorPos;
								resultBlink(1);
							}
							else
								resultBlink(2);
							break;
						case 8:		// Save Music Effect
						case 9:		// Reset
							break;
					}
				}
			}
			else
				longPushButtHold = false;
		}
	}
}

void buttonActionsIR(int holdTime)
{
//    Serial.print("\nDecoded Value: ");
//    Serial.println(decodedVal);
/*    Serial.print("Button row x col: ");
    Serial.print(buttonPos[0]);
    Serial.print(" x ");
    Serial.println(buttonPos[1]);
    Serial.print("Hold Time: ");
    Serial.println(hold);*/

	if(buttonPos[0] == 1 && buttonPos[1] == 3)	// WHITE
		if(state == 9 && holdTime == 13)
		{
			loadDefaults();
			gotoState(0);
			
			resultBlink(0);
		}
	
	if(buttonPos[0] == 0 && buttonPos[1] == 3)		//ON button pressed
	{	
		if(state!= 1)
		{
			if(state == 2)
				gotoState(backupState);
			
			if(!on)                // slowly fade to the color
			{
				displayLED = true;
		
				int tempBrightness = brightness;
				for(brightness=0; brightness<tempBrightness; brightness++)
				{
					displayRGB();
					delay(10);
				}
				
				on = true;  
			}

			if(state!= 3)
			{
				if(progMode)
					resultBlink(2);
					
				if(backupState == 3)
					gotoState(backupState);
				else
					gotoState(1);
			}
		}
	}
	else if(on)
	{
		if(buttonPos[0] == 0)        // topmost row
		{
			switch(buttonPos[1])
			{
				case 0:			// increase brightness button pressed
					if(!progMode)
					{
						if(state == 3)
							gotoState(1);
						else if(effects & 0x10)
						{
							sensitivity-= (sensitivity%17);
							if(sensitivity < MAX_SENSITIVITY)
								sensitivity+= 17;                          //15 levels
						}
						else if(effects & 0x06)
						{
							effectDelay-= (effectDelay%17);
							if(effectDelay > 0)
								effectDelay-= 17;                          //15 levels
							else
								effectDelay = 255;
						}
						else if(effects & 0x01)
						{
							brightness-= (brightness%17);
							if(brightness < MAX_BRIGHTNESS)
								brightness+= 17;                          //15 levels
							else
								brightness = 0;
						}
					}
					effects&= 0x0F;		// reset settings and disable audio
					break;
				case 1:         // decrease brightness button pressed
					if(!progMode)
					{
						if(state == 3)
							gotoState(1);
						else if(effects & 0x10)
						{
							sensitivity-= (sensitivity%17);
							if(sensitivity > 0)
								sensitivity-= 17;                          //15 levels
						}
						else if(effects & 0x06)
						{
							effectDelay-= (effectDelay%17);
							if(effectDelay < MAX_EFFECT_DELAY)
								effectDelay+= 17;                          //15 levels
							else
								effectDelay = 0;
						}
						else if(effects & 0x01)
						{
							brightness-= (brightness%17);
							if(brightness > 0)
								brightness-= 17;                          //15 levels
							else
								brightness = 255;
						}
					}
					effects&= 0x0F;				// reset settings and disable audio
					break;
				case 2:			// OFF button pressed
					if(state == 1)
					{
						gotoState(0);
						
						audioCount = 0;
						audioMaxAmp = 0;
					}
					else if(state == 3)
					{
						on = displayLED = false;
						
						audioCount = 0;
						audioMaxAmp = 0;
					}
					else if(state == 6 || state == 7)
					{
						// set color BLACK
						color[0] = color[1] = color[2] = 0;
						colorPos = BLACK;
					}
					else if(state == 8)
					{
						// set color BLACK
						audioZeroColor[0] = audioZeroColor[1] = audioZeroColor[2] = 0;
						tempColorPos = BLACK;
					}
					break;
				case 3:			// ON button pressed
					/*if(manualOverride == 1)
					cancelOverride();
					
					displayLED = true;
					if(!on)                // slowly fade to the color
					{
					int tempBrightness = brightness;
					for(brightness=0; brightness<tempBrightness; brightness++)
					{
					displayRGB();
					delay(10);
					}
					}
					on = true;*/
					break;
			}
		}
		else if(buttonPos[1] == 3)    // last column button pressed
		{
			switch(buttonPos[0])
			{
				case 1:				// White button pressed
					if(!progMode || !manualOverride)	//state == 1 || state == 3 || state>= 6)
					{
						if(state == 8)
						{
							audioZeroColor[0] = audioZeroColor[1] = audioZeroColor[2] = 255;
							tempColorPos = WHITE;
						}
						else
						{
							color[0] = color[1] = color[2] = 255;
							colorPos = WHITE;
							
							if(state == 1 || state == 3)
								effects = (effects & 0x20) | 1;
							//effectLastChange = 0;
						}
					}
					break;
				case 2: 			// FLASH button pressed
					if(state == 1 || state == 3)
					{
						if(effects & 0x02)		// if FLASH effect is already active
						{
							if( !(effects & 0x10) )	// if music effect inactive
							{
								if(holdTime == 13)
								{
									gotoState(6);
									resultBlink(1);
								}
								else
								{
									layer[1]++;
									if(layer[1] == 5)
										layer[1] = 0;
								}
							}
							else
								effects = (effects & 0x20) | 2;	// set music effect to off
						}
						else
						{
							layer[1] = 0;
							effects = (effects & 0x20) | 2;
						}
						effectLastChange = 0;
					}
					break;
				case 3: 		// STROBE button pressed (fade effect)
					if(state == 1 || state == 3)
					{
						if(effects & 0x04)		// if FLASH effect is already active
						{
							if( !(effects & 0x10) )	// if music effect inactive
							{
								if(holdTime == 13)
								{
									gotoState(7);
									resultBlink(1);
								}
								else
								{
									layer[2]++;
									if(layer[2] == 5)
										layer[2] = 0;
								}
							}
							else
								effects = (effects & 0x20) | 4;
						}
						else
						{
							layer[2] = 0;
							effects = (effects & 0x20) | 4;
						}
						effectLastChange = 0;
					}
					break;
				case 4:		// FADE button pressed (Audio effect)
					if(state == 1 || state == 3)
					{
						if(effects & 0x10) 	// if Audio effect is already active
						{
							if(holdTime == 13)
							{
								gotoState(8);
								resultBlink(1);
							}
							else
							{
								layer[3]++;
								if(layer[3] == 5)
									layer[3] = 0;
							}
						}
						else
						{						
							layer[3] = 0;
							effects|= 0x10;
						}
						
						if(state!= 8)
						{							
							audioCount = 0;
							audioMaxAmp = 0;
							
							//int addrPointer = AUDIO_START + layer[3];
							//int addr = EEPROM.read(addrPointer)*3 + COLOR_START;
							readColor(1, false);
						}
					}
					break;
				case 5:			// SMOOTH button pressed (Settings)
					if(state == 3)
						gotoState(1);
					if(state == 1)
					{
						if(effects & 0x20)
						{
							if(holdTime == 13)
							{
								gotoState(5);
								resultBlink(1);
							}
							else
							{
								layer[4]++;
								if(layer[4] == 5)
									layer[4] = 0;
							}
						}
						else
						{
							layer[4] = 0;
							effects|= 0x20;
						}
						
						if(state!= 5)
						{
							int addr = SETTINGS_START + layer[4]*3;
							brightness = EEPROM.read(addr++);
							effectDelay = EEPROM.read(addr++);
							sensitivity = EEPROM.read(addr);
						}
					}
					break;
			}
		}
		else                      // a colour button pressed
		{
			if(state>= 6 && state<= 8)
			{		
				if(tempColorPos == ( (buttonPos[0]-1)*3 + buttonPos[1] ) + tempColorLayer*15)
				{
					/*if(holdTime == 13)
					{
						tempEffectOrder[0]++;
						tempEffectOrder[ tempEffect[0] ] = colorPos;
					}*/
					tempColorLayer++;
					if(tempColorLayer == 5)
						tempColorLayer = 0;
				}
				else
					tempColorLayer = 0;
				
				tempColorPos = ( (buttonPos[0]-1)*3 + buttonPos[1] ) + tempColorLayer*15;
				/*int addr = tempColorPos*3 + COLOR_START;
				
				for(int i=0; i<3; i++)
					if(state == 8)
						audioZeroColor[i] = EEPROM.read(addr++);
					else
						color[i] = EEPROM.read(addr++);*/
				
					if(state == 8)
						readColor(1, true);
					else
						readColor(0, true);
			}
			if(state == 1 || state == 3)
			{
				if(effects & 0x01)				// if color was active
				{
					if( !(effects & 0x10) )		// music effect inactive
					{
						if(holdTime == 13 )
						{
							gotoState(4);
							resultBlink(1);
						}
						else
						{
							if(colorPos == ( (buttonPos[0]-1)*3 + buttonPos[1] ) + layer[0]*15)
							{
								layer[0]++;
								if(layer[0] == 5)
									layer[0] = 0;
							}
							else
								layer[0] = 0;
						}
					}
					else
						effects = (effects & 0x20) | 1;
				}
				else
				{
					layer[0] = 0;
					effects = (effects & 0x20) | 1;
				}
			
				if(state!= 4)
				{
					colorPos = ( (buttonPos[0]-1)*3 + buttonPos[1] ) + layer[0]*15;
					//int addr = colorPos*3 + COLOR_START;
					readColor(0, false);
				}
				//effects = 1;
				//effectLastChange = 0;
			}
		}
	}

  //prevDecodedVal = (buttonPos[0] << 2) + buttonPos[1];
}

void backup()
{
	backupColorPos = colorPos;
	backupBrightness = brightness;
	backupEffectDelay = effectDelay;
	backupSensitivity = sensitivity;
	backupEffects = effects;
	backupState = state;
	backupOn = on;
}

/*
	colorPos = backupColorPos;
	brightness = backupBrightness;
	effectDelay = backupEffectDelay;
	sensitivity = backupSensitivity;
	effects = backupEffects;
	state = backupState;
	on = backupOn;
*/

void gotoState(uint8_t toState)
{
	/*Serial.print("State ");
	Serial.print(state);
	Serial.print(" -> State ");
	Serial.println(toState);*/
	
	switch(state)
	{
		case 0:	 	// Off
		case 1:		// On
			break;
		case 4:		// Save Color
			progMode = false;
		case 2:		// Set Color Override
			on = backupOn;
			manualOverride = false;
			
			brightness = backupBrightness;
			readColor(0, false);
			break;
		case 5:		// Save Settings
			progMode = false;
		case 3:		// Set Settings Override
			manualOverride = false;
			break;
		case 6:		// Save FLASH Effect
		case 7:		// Save FADE Effect
			//effects = backupEffects;
			//colorPos = backupColorPos;
			readColor(0, false);
			progMode = false;
			break;
		case 8:		// Save Music Effect
			//effects = backupEffects;
			{
				//int addrPointer = AUDIO_START + layer[3];
				//int addr = EEPROM.read(addrPointer)*3 + COLOR_START;
				readColor(1, false);
			}
		case 9:		// Reset
			progMode = false;
			break;
	}
	
	backup();
	
	switch(toState)
	{
		case 0:	 	// Off
			on = displayLED = false;
			break;
		case 1:		// On
			on = displayLED = true;
			break;
		case 2:		// Set Color Override
			displayLED = true;
			brightness = 255;
		case 3:		// Set Settings Override
			manualOverride = true;
			break;
		case 4:		// Save Color
			brightness = 255;
		case 5:		// Save Settings
			manualOverride = true;
			progMode = true;
			break;
		case 6:		// Save FLASH Effect
		case 7:		// Save FADE Effect
			tempEffectOrder[0] = 0;
			tempColorLayer = 0;
			tempColorPos = 37;
			readColor(0, true);
		case 8:		// Save Music Effect
			tempColorLayer = 0;
			tempColorPos = 37;
			readColor(1, true);
		case 9:		// Reset
			progMode = true;
			break;
	}
	
	state = toState;
}

int decodeIR()
{
  unsigned long lastTime = lastMicros;
  unsigned long nowTime = micros();

  if(nowTime > lastTime)            // This check is so diffTime is positive. lastTime can sometimes be greater if interrupt is called inbetween
  {
    unsigned long diffTime = (nowTime - lastTime);

    if( currentpulse >= MAX_PULSE_PAIRS || (diffTime > TIMEOUT && currentpulse > 32) )
    {
      detachInterrupt(0);        // disable interrupt to prevent value changing

      int decodedVal1 = 0, decodedVal2 = 0;
      int offSignal, offCode;

      for(int i=0; i<5; i++)
      {
        offSignal = 1620;           //pulse length for HIGH
        offCode = pulses[i+17][1];
        if( abs(offCode - offSignal) < (offSignal * FUZZINESS / 100) )    //match (signal 1)
          decodedVal1 += 1<<i;

        offSignal = 520;            //pulse length for HIGH
        offCode = pulses[i+25][1];
        if( abs(offCode - offSignal) < (offSignal * FUZZINESS / 100) )    //match (signal 1)
          decodedVal2 += 1<<i;
      }

/*      Serial.println("ON, \tOFF");
      for(int i=0; i<currentpulse; i++)
      {
        Serial.print(pulses[i][0]/10);
        Serial.print(", \t");
        Serial.println(pulses[i][1]/10);
      }*/

      currentpulse = 0;
      int holdTime = hold;

      attachInterrupt(0, readIR, CHANGE);

      if(decodedVal1 == decodedVal2)
      {
        buttonPos[0] = decodedVal1 >> 2;
        buttonPos[1] = decodedVal1 & 0x03;
        prevDecodedVal = decodedVal;
        decodedVal = decodedVal1;
        return holdTime;
      }      
    }
  }
/*  else
    Serial.print("Heisenberg's bitch");*/

    return -1;
}

void readColor(int val, boolean temp)
{
	int addr;
	
	if(val == 0)
	{
		if(!temp)
			addr = colorPos*3 + COLOR_START;
		else
			addr = tempColorPos*3 + COLOR_START;
			
		for(int i=0; i<3; i++)
			color[i] = EEPROM.read(addr++);
	}
	else
	{
		int addrPointer;
		if(!temp)
		{
			addrPointer = AUDIO_START + layer[3];
			addr = EEPROM.read(addrPointer)*3 + COLOR_START;
		}
		else
			addr = tempColorPos*3 + COLOR_START;
			
		for(int i=0; i<3; i++)
			audioZeroColor[i] = EEPROM.read(addr++);
	}
}

void displayRGB()
{
	int tempColor[3];
	
	if(displayLED)
	{
		float brightnessRatio = float(brightness)/MAX_BRIGHTNESS;
		
		if(effects & 0x10)
		{
			//float percent = audioBrightness/MAX_BRIGHTNESS;
			
			for(int i=0; i<3; i++)
				tempColor[i] = min( int(audioZeroColor[i]*(1.0-audioBrightnessRatio)) + int(color[i]*audioBrightnessRatio), 255);
		}
		else
		{
			for(int i=0; i<3; i++)
				tempColor[i] = color[i];
		}

		for(int i=0; i<3; i++)
			tempColor[i] = tempColor[i] * brightnessRatio;
	}
	else
		tempColor[0] = tempColor[1] = tempColor[2] = 0;

	analogWrite(RED_PIN, tempColor[0]);
	analogWrite(GREEN_PIN, tempColor[1]);
	analogWrite(BLUE_PIN, tempColor[2]);	//255 - 
}

void displayEffects()
{
	if(effects & 0x06)
	{
		int addrPointer, addr;
		unsigned long currentTime = millis();

		if(effectLastChange == 0)		// restart effect
		{
			effectLastChange = currentTime;
			effectCount = 1;

			if(effects & 0x02)
				addrPointer = FLASH_START + layer[1]*16;
			else
				addrPointer = FADE_START + layer[2]*16;
			effectColorCount = EEPROM.read(addrPointer++);
			addr = EEPROM.read(addrPointer++)*3 + COLOR_START;

			for(int i=0; i<3; i++)
				effectLastColor[i] = color[i] = EEPROM.read(addr++);

			addr = EEPROM.read(addrPointer)*3 + COLOR_START;
			for(int i=0; i<3; i++)
				effectNextColor[i] = EEPROM.read(addr++);

			//displayRGB();
		}

		if(currentTime - effectLastChange >=  effectDelay*20)
		{
			for(int i=0; i<3;i++)
				color[i] = effectLastColor[i] = effectNextColor[i];

			effectCount++;
			if(effectCount == effectColorCount)
				effectCount = 0;

			if(effects & 0x02)
				addrPointer = FLASH_START  + layer[1]*16 + 1;
			else
				addrPointer = FADE_START  + layer[2]*16 + 1;

			addr = EEPROM.read(addrPointer + effectCount)*3 + COLOR_START;

			for(int i=0; i<3; i++)
				effectNextColor[i] = EEPROM.read(addr++);

			//displayRGB();

			effectLastChange = currentTime;
		}
		else if(effects & 0x04)
		{
			float percent = float(currentTime - effectLastChange) / float(effectDelay*20);
			//                Serial.println(percent);
			for(int i=0; i<3; i++)
			{
				int tempCol = int( float(effectNextColor[i])*percent + float(effectLastColor[i])*(1.0-percent) );
				color[i] = min(tempCol, 255);
			}
		}
	}

	if(effects & 0x10)
	{
		readAudio();
	}
}

void readAudio()
{
	int audioAmp = analogRead(AUDIO_PIN);
//Serial.println(audioAmp);
	
	if(audioAmp > audioMaxAmp)
	  audioMaxAmp = audioAmp;
	//if(audioAmp > AUDIO_ZERO)
	//	audioMaxAmp+= audioAmp;
	//else
	//	audioMaxAmp+= AUDIO_ZERO;

	audioCount++;

	if(audioCount == AUDIO_SAMPLE_SIZE)
	{
		//Serial.print(audioMaxAmp);
		//Serial.print("\t");

//		audioMaxAmp/= AUDIO_SAMPLE_SIZE;

		Serial.println(audioMaxAmp);
		//Serial.print("\t");

		if(audioMaxAmp > AUDIO_ZERO)
			audioMaxAmp = audioMaxAmp-AUDIO_ZERO;	//max(audioMaxAmp-AUDIO_ZERO, 0);
		else
			audioMaxAmp = 0;

		//if(displayLED)
//			Serial.println(audioMaxAmp);
	
		//Serial.print(audioMaxAmp);
		//Serial.print("\t");
		
		if(!displayLED)		// state == 0????
		{
			unsigned long currTime = millis();
			unsigned long tempTimeDiff = currTime - audioLastPeak;

			if(tempTimeDiff > 1000)
				clapCount = 0;
				
			//uint8_t LEDBrightness = map(audioMaxAmp, 0, 1023-AUDIO_ZERO, 0, 255);
			//analogWrite(INDIE_LED_PIN, LEDBrightness);

			if(audioMaxAmp > 200 && tempTimeDiff > 150)			// FIX this with variable AUDIO_ZERO
			{
				Serial.print(audioMaxAmp);
				Serial.print("\t");
				Serial.print(currTime);
				Serial.print("\t");
				Serial.print(tempTimeDiff);
				Serial.print("\t");
				
				digitalWrite(INDIE_LED_PIN, HIGH);

				switch(clapCount)
				{
					case 0:
						audioLastPeak = currTime;
						clapCount++;
						break;
					case 1:
						if(tempTimeDiff > 200 && tempTimeDiff < 350)
						{
							clapCount++;
							audioLastPeak = currTime;
						}
						else
							clapCount = 0;
						break;
					case 2:
						if(tempTimeDiff > 550 && tempTimeDiff < 750)
						{
							// Emulate ON button
							buttonPos[0] = 0;
							buttonPos[1] = 3;
							buttonActionsIR(1);
						}
						clapCount = 0;
						break;
				}
				// delay(10);
				Serial.println(clapCount);
			}
			else
				digitalWrite(INDIE_LED_PIN, LOW);
		}
		else
		{			
			float sensitivityRatio = (sensitivity / 63.75) + 1.0;				// 1.0 - 3.0

			audioMaxAmp = min( int(sensitivityRatio * audioMaxAmp), (1023-AUDIO_ZERO) );
			audioBrightnessRatio = float(audioMaxAmp) / float(1023-AUDIO_ZERO);	//	0.0 - 1.0
			//audioBrightness = sensitivityRatio * brightness;

			/*Serial.print("DATA,TIME,");
			Serial.print(millis());
			Serial.print(",");*/

			//Serial.print(sensitivityRatio);
			//Serial.print("\t");
			//Serial.print("\t");
			//Serial.println(audioBrightnessRatio);

			//      Serial.print("{");
			//      Serial.print("audio");
			//      Serial.print(",T,");
			//      Serial.println(audioBrightness);
			//      Serial.println("}");*/

			/*      for(int i=0; i<audioBrightness; i++)
			Serial.print("*");
			Serial.println("");*/

		}
		
		audioCount = 0;
		audioMaxAmp = 0;

	}
}

void resultBlink(uint8_t count)
{
	uint8_t tempCol[3];
	boolean tempDisplayLED = displayLED;
	for(int i=0; i<3; i++)
		tempCol[i] = color[i];
	displayLED = true;
	
	color[0] = color[1] = color[2] = 0;					// black
	displayRGB();
	delay(100);
	
	if(count == 0)
	{
		color[0] = color[1] = color[2] = 255;			// white
		displayRGB();
		delay(400);
		
		color[0] = color[1] = color[2] = 0;				// black
		displayRGB();
		delay(100);

	}
	else
	{
		for(int i=0; i<count; i++)
		{
			color[0] = color[1] = color[2] = 255;		// white
			displayRGB();
			delay(100);
			
			color[0] = color[1] = color[2] = 0;			// black
			displayRGB();
			delay(100);
		}
	}
	
	for(int i=0; i<3; i++)
		color[i] = tempCol[i];
	
	displayLED = tempDisplayLED;
}

void loadDefaults()
{
	resultBlink(true);

  int addr = 0;
  int pos;



//Serial.print("Colors Start: ");
//Serial.println(addr);
  for(int row=0; row<5; row++)	// save first layer of colors
  {
    int val = 64*row;
    if(val == 256)
      val = 255;

    pos = 0;

    for(int col=0; col<3; col++)
    {
      for(int rgb=0; rgb<3; rgb++)
      {
        switch(pos)
        {
          case 0: EEPROM.write(addr, 255);
                  break;
          case 1: EEPROM.write(addr, val);
                  break;
          case 2: EEPROM.write(addr, 0);
                  break;
        }
        addr++;
        pos++;
        if(pos == 3)
          pos = 0;        
      }  
      pos--;
      if(pos == -1)
        pos = 2;
    }
  }


  for(int row=0; row<5; row++)		// save second layer of colors
  {
    int val = 64*row;
    if(val == 256)
      val = 255;

    pos = 0;

    for(int col=0; col<3; col++)
    {
      for(int rgb=0; rgb<3; rgb++)
      {
        switch(pos)
        {
          case 0: EEPROM.write(addr, 255);
                  break;
          case 1: EEPROM.write(addr, 0);
                  break;
          case 2: EEPROM.write(addr, val);
                  break;
        }
        addr++;
        pos++;
        if(pos == 3)
          pos = 0;        
      }  
      pos--;
      if(pos == -1)
        pos = 2;
    }
  }
  
	for(int count=0; count<3; count++)	//Fill rest colors with white
		for(int row=0; row<5; row++)
			for(int col=0 ; col<3; col++)
				for(int rgb=0; rgb<3; rgb++)
					EEPROM.write(addr++, 255);

	// set default color to red
	EEPROM.write(112, 0);
	EEPROM.write(113, 0);


  for(int rgb=0; rgb<3; rgb++)			// save black color
    EEPROM.write(addr++, 0);

  for(int rgb=0; rgb<3; rgb++)			// save white color
    EEPROM.write(addr++, 255);

//Serial.print("Flash Start: ");
//Serial.println(addr);
	//for(int count=0; count<5; count++)	// save flash first layer
	{
		EEPROM.write(addr++, 6);
		
		EEPROM.write(addr++, 0);		// red
		EEPROM.write(addr++, 12);		// yellow
		EEPROM.write(addr++, 1);		// green
		EEPROM.write(addr++, 13);		// cyan
		EEPROM.write(addr++, 2);		// blue
		EEPROM.write(addr++, 14);		// magenta
		
		addr+= 9;
	}
	
	for(int count=0; count<4; count++)
	{
		EEPROM.write(addr++, 1);
		EEPROM.write(addr++, 76);	//white
		addr+= 14;
	}
	
//Serial.print("Fade Start: ");
//Serial.println(addr);
	//for(int count=0; count<5; count++)	// save fade first layer
	{
		EEPROM.write(addr++, 6);
		
		EEPROM.write(addr++, 0);		// red
		EEPROM.write(addr++, 12);		// yellow
		EEPROM.write(addr++, 1);		// green
		EEPROM.write(addr++, 13);		// cyan
		EEPROM.write(addr++, 2);		// blue
		EEPROM.write(addr++, 14);		// magenta
		
		addr+= 9;
	}
	
	for(int count=0; count<4; count++)
	{
		EEPROM.write(addr++, 1);
		EEPROM.write(addr++, 76);		// white
		addr+= 14;
	}

//Serial.print("Music Start: ");
//Serial.println(addr);	
	// save music colours
	EEPROM.write(addr++, 75);			// black
	EEPROM.write(addr++, 76);			// white
	EEPROM.write(addr++, 0);			// red
	EEPROM.write(addr++, 1);			// green
	EEPROM.write(addr++, 2);			// blue

//Serial.print("Settings Start: ");
//Serial.println(addr);
	//save settings
	for(int lay=0; lay<5; lay++)
	{
		//for(int count=0; count<3; count++)
		EEPROM.write(addr++, 127);
		EEPROM.write(addr++, 127);
		EEPROM.write(addr++, 0);
	}
	
	resultBlink(true);
}
