/*
RM69_transmitter2

D.R.Patterson
18/9/2018

MPL3115A2 Barometric Pressure Sensor
Altitude and Temperature
With Atmospheric Pressure Correction
Data acquired in sensor one shot mode

NB if sensor has a calibrated pressure offset, the sensor altitude fuction returns a bad reading
This version reads pressure and converts to altitude locally

 Hardware Connections (pressure breakout to mini 238):
 VCC = 3.3V
 SDA = A4
 SCL = A5
 Gnd = Gnd

use newbase = xx.x from serial port to change the base address
1 Decimal place!

Contains original code and code adapted from http://www.henrylahr.com/?p=99

*/

#include <Wire.h>
#include <EEPROM.h>
#define I2C_FREQ 400000L

boolean maxoversample = true; // greater accuracy, less speed
// bit 2 in CTRL_REG1 (26) allows over-sampling when set
// bit 4,5,6 contain the over sampling rate. 56 is max rate(512ms min), 48 next (258mS)

const int SENSORADDRESS = 0x60; // address specific to the MPL3115A2, value found in datasheet

// by polling the status register for the new pressure data flag we avoid using timers
const int PT_DATA_CFG = 0x13; // set flag register

byte IICdata[5] = {0,0,0,0,0}; //buffer for sensor data

boolean initSensor(){
Wire.begin();        // Join i2c bus
TWBR = ((F_CPU / I2C_FREQ) - 16) / 2;
  if(IIC_Read(0x0C) == 196) { //checks whether sensor is readable (who_am_i bit)
    if(useSerial) Serial.println(F("\nSensor OK."));
  } else {
    if(useSerial) Serial.println(F("\nSensor un-responsive."));
  return false;
  }

  // read the current value from the eeprom
  // if it exists the 1st 2 locations should read oK 
  if ( (EEPROM.read(0) == 79) && EEPROM.read(1) == 107) {
  unsigned long alt = readeeprom(2);
  basealt = float(alt) / 10;
  } else basealt = ALTBASIS;
newbasealt = basealt;

IIC_Write(PT_DATA_CFG, 0x07); // Enable all three pressure and temp event flags
calibrate();  
return true;
}

float Temp_Read(){
// return temperature in C
// correction applied by writing to reg 0x2C in calibrate 
// NB a call to this function assumes function Baro/Alt_Read has occured
// otherwise IIC_ReadData() will not have been called for current value
bool negSign = false;
byte msb = IICdata[3];
byte lsb = IICdata[4];
word foo = 0;
//Check for 2s compliment (negative temperature representation!)
  if(msb > 0x7F){
  foo = ~((msb << 8) + lsb) + 1; //2’s complement
  msb = foo >> 8;
  lsb = foo & 0x00F0;
  negSign = true;
  }
// The least significant bytes l_altitude and l_temp are 4-bit,
// fractional values, so you must cast the calulation in (float),
// shift the value over 4 spots to the right and divide by 16 (since
// there are 16 values in 4-bits).
float templsb = (lsb>>4)/16.0; //temp, fraction of a degree
float temperature = (float)(msb + templsb);
  if (negSign) temperature = 0 - temperature;
return temperature;
}

float Baro_Read(){
//this function takes values from the read buffer and converts them to pressure units
IIC_ReadData(); //reads registers from the sensor
unsigned long m_pressure = IICdata[0];
unsigned long c_pressure = IICdata[1];
float l_pressure = (float)(IICdata[2]>>4)/4; //dividing by 4, since two lowest bits are fractional value
return((float)(m_pressure<<10 | c_pressure<<2)+l_pressure); //shifting 2 to the left to make room for LSB
}
 
float Alt_Read(){
//Reads altitude in m (if CTRL_REG1 is set to altitude mode)
IIC_ReadData(); //reads registers from the sensor
int m_altitude = IICdata[0];
int c_altitude = IICdata[1];
float l_altitude = (float)(IICdata[2]>>4)/16;
return((float)((m_altitude << 8)|c_altitude) + l_altitude);
}

byte IIC_Read(byte regAddr){
// This function reads one byte over I2C
Wire.beginTransmission(SENSORADDRESS);
Wire.write(regAddr); // Address of CTRL_REG1
Wire.endTransmission(false); // Send data to I2C dev with option for a repeated start. Works in Arduino V1.0.1
Wire.requestFrom(SENSORADDRESS, 1);
return Wire.read();
}

void IIC_Write(byte regAddr, byte value){
// This function writes one byto over I2C
Wire.beginTransmission(SENSORADDRESS);
Wire.write(regAddr);
Wire.write(value);
Wire.endTransmission(true);
}
 
void IIC_ReadData(){  //Read Altitude/Barometer and Temperature data (5 bytes)
//This is faster than reading individual register, as the sensor automatically
//increments the register address, so we just keep reading...
byte i=0;
Wire.beginTransmission(SENSORADDRESS);
Wire.write(0x01); // Address of CTRL_REG1
Wire.endTransmission(false);
Wire.requestFrom(SENSORADDRESS,5); //read 5 bytes: 3 for altitude or pressure, 2 for temperature
while(Wire.available()) IICdata[i++] = Wire.read();
}

void tempcalib(){
// temp correction = 0.0625 per bit
// -8 to +7.9375  128 = -8  127 = 7.9375
byte twoc;
  if (tempcorrection <0) twoc  = ~byte(abs(tempcorrection) *16 - 1); else twoc = tempcorrection * 16;
IIC_Write(0x2C,twoc); //write temperature offset 
}

unsigned long readeeprom(int location) { 
// read stored altitude in eeprom
unsigned long temp = EEPROM.read(location);
temp = temp <<8;
temp = temp + EEPROM.read(location + 1);
temp = temp <<8;
temp = temp + EEPROM.read(location + 2);
temp = temp <<8;
temp = temp + EEPROM.read(location + 3);
return(temp);
}

void writeeeprom(int location, unsigned long memory) { 
// write to eeprom
cli();
int temp = int(memory  >> 24);
EEPROM.write(location, temp);
memory = memory << 8;
temp = int(memory  >> 24);
EEPROM.write(location + 1, temp);
memory = memory << 8;
temp = int(memory  >> 24);
EEPROM.write(location + 2, temp);
memory = memory << 8;
temp = int(memory  >> 24);
EEPROM.write(location + 3, temp);
EEPROM.write(location - 2, 79);  // o
EEPROM.write(location - 1, 107); // K flag the bytes are present
sei();
}

void calibrate(){
// Reference sea pressure is 101326, atmospheric conditions alters this!
// Calibration requires estimating sea pressure the measured pressure at a known altitude

String temp = "A " + String(basealt, 1) + "m";
unsigned int L = temp.length();
temp.toCharArray(pdata, L + 1); 
radio.sendWithRetry(GATEWAYID, pdata, L, 1);

float alt;
float myaltitude;
const byte samples = 15;
tempcalib(); // set temperaure offset

// set pressure correction
// Pa 4Pa per bit -512 to +508
// multiples of 4
float C = float(pcorrection) / 4.0;
byte twoc = byte(0.5 + abs(C));
  if (pcorrection < 0) twoc  = ~(twoc - 1); // 2's complement negative number
IIC_Write(0x2B, twoc); //write pressure offset twoc

  if (useSerial){
  Serial.println(F("\nPressure calibration...\n"));
  Serial.print(F("OverSample ")); if(maxoversample) Serial.println(F("On")); else Serial.println(F("Off"));
  Serial.print(F("Base altitude ")); Serial.print(basealt, 1);Serial.println(F(" m"));
  Serial.println(F("Use ""newbase = xy.z"" from Serial port to set a new altitude"));
  byte offset = IIC_Read(0x2B);
  int newval = offset;
    if ((offset & 128) == 128){
    offset = ~ offset;
    newval =  -1 - offset;
    }
  Serial.print(F("Pressure offset ")); Serial.print(newval);Serial.println(F(" (at 4 Pa per bit)"));

  unsigned long pHi = IIC_Read(0x14);
  unsigned long pLo = IIC_Read(0x15);
  Serial.print(F("Barometric Pressure Input, (High ")); Serial.print(pHi);
  Serial.print(F(" Low ")); Serial.print(pLo); Serial.print(") ");
  Serial.print(2 *(( pHi << 8) + pLo) ); Serial.println(F(" Pa"));
  byte T = IIC_Read(0x2C);
  Serial.print(F("Temperature offset (")); Serial.print(T,BIN);
  float Tcorrection = T;
    if((T & 128) == 128) {
    T = ~T;
    Tcorrection  = -1 - T;
    }
  Serial.print(F(") ")); Serial.print(0.0625 * Tcorrection, 4);
  Serial.println(F(" C (at 0.0625 C per bit)"));
  Serial.print(F("Altitude offset ")) ; Serial.print(IIC_Read(0x2D), BIN);Serial.println(F(" (at 1 m per bit)"));
  Serial.flush();
  }
// reset Barometric Pressure input to default (pg 29):
// 101326
// use 2Pa units
// 50663
IIC_Write(0x14, 197); // 50663 >> 8
IIC_Write(0x15,231); // 50663 & 0xFF

IIC_Write(0x2D,0); //write altitude offset=0 (because calculation below is based on offset = 0)

// read and discard 1st altitude value
IIC_Write(0x26, 0b10111001);
IIC_Write(0x26, 0b10111011);
  while ((IIC_Read(STATUS) & 2) == 0);
Alt_Read(); 
// read and discard 2nd altitude value
IIC_Write(0x26, 0b10111001);
IIC_Write(0x26, 0b10111011);
  while ((IIC_Read(STATUS) & 2) == 0);
Alt_Read();

IIC_Write(0x26, 0b10111001);
IIC_Write(0x26, 0b10111011);
  while ((IIC_Read(STATUS) & 2) == 0);
myaltitude = Alt_Read();

  if (useSerial){
  Serial.print(F("Uncalibrated "));
  Serial.print(myaltitude, 1);
  Serial.println(F(" m")); Serial.flush();
  }
IIC_Write(0x26, 0b00111001);
IIC_Write(0x26, 0b00111011);
while ((IIC_Read(STATUS) & 2) == 0);
Baro_Read(); // reject first
IIC_Write(0x26, 0b00111001);
IIC_Write(0x26, 0b00111011);
while ((IIC_Read(STATUS) & 2) == 0);
Baro_Read(); // reject

float currpress = 0;
basetemp = 0;
  for (byte i=0; i < samples; i++){
  IIC_Write(0x26, 0b00111001);
  IIC_Write(0x26, 0b00111011);
    while ((IIC_Read(STATUS) & 2) == 0);
  currpress += Baro_Read();
  basetemp += Temp_Read();
  }
currpress = currpress /samples;

basetemp = fixTemp(basetemp / samples);
  if (Tcalibrate){
  temp = "T " + String(basetemp, 1) + "C";
  L = temp.length();
  temp.toCharArray(pdata, L + 1); 
  radio.sendWithRetry(GATEWAYID, pdata, L, 1);
  }
mintemp = basetemp;
maxtemp = basetemp;
seapress = currpress/pow(1 - basealt * 0.0000225577,5.255877);

  if (useSerial){
  Serial.print(F("Average Pressure "));
  Serial.print(currpress,1);
  Serial.println(F(" Pa"));
  Serial.print(F("Sea Pressure "));
  Serial.print(seapress,1);
  Serial.println(F(" Pa"));
  Serial.print(F("(Allowing for temperature) "));
  Serial.print(sealevel(currpress, basetemp)); Serial.println(F(" Pa"));
  Serial.flush();
  }
  
// This configuration option calibrates the sensor according to
// the sea level pressure for the measurement location (2 Pa per LSB)
// default value = 101326 Pa
IIC_Write(0x14, (unsigned int)(0.5 + seapress / 2)>>8);
IIC_Write(0x15, (unsigned int)(0.5 + seapress / 2)&0xFF);

IIC_Write(0x26, 0b10111001);
IIC_Write(0x26, 0b10111011);
  while ((IIC_Read(STATUS) & 2) == 0);
Alt_Read(); // discard
myaltitude = 0;
  for(byte i=0; i < 2; i++) { // settle altitude reading
  IIC_Write(0x26, 0b10111001);
  IIC_Write(0x26, 0b10111011);
  while ((IIC_Read(STATUS) & 2) == 0);
  myaltitude += Alt_Read();
  }
myaltitude = myaltitude/2;

IIC_Write(0x26, 0b00111001);
IIC_Write(0x26, 0b00111011);
  while ((IIC_Read(STATUS) & 2) == 0);
Baro_Read(); // discard
IIC_Write(0x26, 0b00111001);
IIC_Write(0x26, 0b00111011);
  while ((IIC_Read(STATUS) & 2) == 0);
Baro_Read(); // discard

alt = 0;
  for(byte i=0; i < 6; i++) { // settle pressure reading
  delay(10);
  IIC_Write(0x26, 0b00111001);
  IIC_Write(0x26, 0b00111011);
  while ((IIC_Read(STATUS) & 2) == 0);
  alt += Baro_Read();
  }
alt = altitude(alt / 6, seapress);

  if (useSerial){
  Serial.print(F("Altitude by sensor function ")); Serial.print(myaltitude,1); Serial.println(F(" m"));
  Serial.print(F("Altitude by local function ")); Serial.print(alt, 2); Serial.println(F(" m"));
  Serial.print(F("Base temp ")); Serial.print(basetemp,1); Serial.println(F(" C\n"));
  Serial.flush();
  }
   
if (!useBase) newbasealt = alt;
}

void oneshotP(){ // Pressure
  if(maxoversample){
  IIC_Write(0x26, 0b00111001); // bits 3-5 control oversample
  IIC_Write(0x26, 0b00111011);
  }else{
  IIC_Write(0x26, 0b00110001);
  IIC_Write(0x26, 0b00110011);
  }
}

void oneshot(){ // Altitude
  if(maxoversample){
  IIC_Write(0x26, 0b10111001); // bits 3-5 control oversample
  IIC_Write(0x26, 0b10111011);
  }else{
  IIC_Write(0x26, 0b10110001);
  IIC_Write(0x26, 0b10110011);
  }  
}

float sealevel(float lpressure, float ltemperature){
float lseaP =  ALTBASIS /(29.2717593 * (ltemperature + 273.15) );
lseaP = lpressure * exp(lseaP);
return lseaP;
}

float altitude(float lpressure, float seapressure){
// h = 44330.77( 1 - (p/Po)^ 0.1902632 )
float alt = pow(lpressure / seapressure, 0.1902632);
alt = 44330.77 * (1 - alt);  
return alt;
}
