/* ========================================
 *
 * Copyright Andyk75, 2017
 * All Rights Reserved
 * UNPUBLISHED, LICENSED SOFTWARE.
 *
 * ========================================
*/

/* ========================================
 * Includes
 * ======================================== */
#include "project.h"
#include <device.h>
#include <BLE_Process.h>
#include <poveffect.h>
#include <povanimation.h>
#include <stdbool.h>
#include <persistentstorage.h>
#include <stdlib.h>
#include <watchdogtimer.h>

/* ========================================
 * Global defines, Typedefs, Structures
 * ======================================== */
uint32 Initialwaittime = 333;
e_Animation Active_Animation = TextAnimation;
uint32 speedcounter = 0;
uint32 SPEED = 1;
bool TriggerDisplayFromIRQ = false; 
bool TriggerDisplayFromButtonPressOrProgrammatically = true;
bool Debounceflag = false;
uint32 AnimationCounter = 0;
uint32 ActiveAnimationsPointer = 0;

uint8 MyAnimationMax = sizeof(e_Animation); 
uint32 WatchdogTimeout = 0;
bool BackupData = false;
bool SendNotification = false;
extern uint8 TextWriteCounter;


/* ========================================
 *  Local Functions-Definition
 * ======================================== */
void InitializeSystem(void);
static void LowPowerImplementation(void);
CY_ISR_PROTO(Timer_Interrupt);
void TriggerAnimation(uint8 anim);
void SwitchAnimation();
void main_WriteVariablesToPersistentStorage( void );
CY_ISR_PROTO(Main_Startbutton_ISR);
void main_SendBLENotification();


/* ========================================
 *  Public Functions
 * ======================================== */

int main(void)
{
    InitializeSystem();

    for(;;)
    {       
        CyBle_ProcessEvents();

        if( ( TriggerDisplayFromIRQ == true ) & (TriggerDisplayFromButtonPressOrProgrammatically == true) ) {
            speedcounter++;
            
            if(speedcounter >= SPEED ) {
                StripLights_Trigger(1);   
                CyDelay(2); // Wait for the hardware to update the strip. 
       
                // Calculate the next strip to display next time. 
                TriggerAnimation(Active_Animation);
                speedcounter = 0;
            }
        }
        main_SendBLENotification();
        
        if( BackupData == true ){
            main_WriteVariablesToPersistentStorage();
            BackupData = false;
        }
        
        LowPowerImplementation();
    }
}



/*******************************************************************************
* Function Name: InitializeSystem
********************************************************************************
* Summary:
* Start the components and initialize system.
*
* Parameters:
*  void
*
* Return:
*  void
*
*******************************************************************************/
void InitializeSystem(void)
{
    CyGlobalIntEnable; // enable global interrupts

    
    PSTOR_ReadUserFlashRow();
    PSTOR_ReadInitialParameters(e_SPEED, &SPEED );
    
    WatchdogTimeout = 512;    
    WDT_Start(WatchdogTimeout, &Timer_Interrupt );
    
    BackupData = false;
	/* Start BLE component and register the CustomEventHandler function. This 
	 * function exposes the events from BLE component for application use */
    CyBle_Start(CustomEventHandler);	
    CyDelay(10);
    
    START_IRQ_StartEx(Main_Startbutton_ISR);
    
    // striplights component init
    StripLights_Start();
        
    StripLights_Dim(StripLights_DimLevel_4);

    //cycle each of the primaries on all LED's, DisplayClear sets and pushes to LED's, MemoryClear just clears the buffer.
    StripLights_DisplayClear( StripLights_RED_MASK );	CyDelay( Initialwaittime ); 
	StripLights_DisplayClear( StripLights_GREEN_MASK );	CyDelay( Initialwaittime ); 
	StripLights_DisplayClear( StripLights_BLUE_MASK );	CyDelay( Initialwaittime ); 
	StripLights_DisplayClear( 0x00000000 );	 
    StripLights_Dim(StripLights_DimLevel_0);   
    
}


/*******************************************************************************
* Function Name: Timer_Interrupt
********************************************************************************
*
* Summary:
*  Handles the Interrupt Service Routine for the WDT timer.
*
* Parameters:
*   None
*
* Return:
*   None
*
*******************************************************************************/
CY_ISR(Timer_Interrupt)
{
    uint32 Interrupt_Source = CySysWdtGetInterruptSource();
    
    switch(Interrupt_Source) {        
        case WDT_BLE_INTERRUPT_SOURCE :
            // Just wake up to handle BLE
            // BLUE_Write(~BLUE_Read()); 
            TriggerDisplayFromIRQ = true;
            break;
                
        case WDT_INTERRUPT_SOURCE:
            // This is a 1s Interrupt source! 
            Debounceflag = false;
            // Just wake up for handling BLE
            break;
        
        default: 
            break;
    }
    
    /* Clear interrupt request */
    CySysWdtClearInterrupt(Interrupt_Source);
    
}


/*******************************************************************************
* Function Name: LowPowerImplementation()
********************************************************************************
* Summary:
* Implements low power in the project.
*
* Parameters:
* None
*
* Return:
* None
*
* Theory:
* The function tries to enter deep sleep as much as possible - whenever the 
* BLE is idle and the UART transmission/reception is not happening. At all other
* times, the function tries to enter CPU sleep.
*
*******************************************************************************/
static void LowPowerImplementation(void)
{
    CYBLE_LP_MODE_T bleMode;
    uint8 interruptStatus;
    
    /* For advertising and connected states, implement deep sleep 
     * functionality to achieve low power in the system. For more details
     * on the low power implementation, refer to the Low Power Application 
     * Note.
     */
    {
        /* Request BLE subsystem to enter into Deep-Sleep mode between connection and advertising intervals */
        bleMode = CyBle_EnterLPM(CYBLE_BLESS_DEEPSLEEP);
        /* Disable global interrupts */
        interruptStatus = CyEnterCriticalSection();
        /* When BLE subsystem has been put into Deep-Sleep mode */
        if(bleMode == CYBLE_BLESS_DEEPSLEEP)
        {
            /* And it is still there or ECO is on */
            if((CyBle_GetBleSsState() == CYBLE_BLESS_STATE_ECO_ON) || 
               (CyBle_GetBleSsState() == CYBLE_BLESS_STATE_DEEPSLEEP))
            {
                CySysPmDeepSleep();
            }
        }
        else /* When BLE subsystem has been put into Sleep mode or is active */
        {
            /* And hardware doesn't finish Tx/Rx opeation - put the CPU into Sleep mode */
            if(CyBle_GetBleSsState() != CYBLE_BLESS_STATE_EVENT_CLOSE)
            {
                CySysPmSleep();
            }
        }
        /* Enable global interrupt */
        CyExitCriticalSection(interruptStatus);
    }
    
}

void TriggerAnimation(uint8 anim) 
{
    switch(anim)
    {
        case Knightrider:   
            RedKitt();
        break;
        case Rainbowswirl:  
            ColorSwirl();
        break;
        case ColorKnightRider:  
            ColorKitt();   
        break;
        case Valueswirl: 
            ValueSwirl();
        break;
        case WS_CandyCane:  
            StripLights_Dim(StripLights_DimLevel_2);
            CandyCane(9, StripLights_BLUE, StripLights_RED );
        break;
        case WS_CandyCane_2: 
            StripLights_Dim(StripLights_DimLevel_2);
            CandyCane(9, StripLights_BLUE, StripLights_YELLOW );
        break;
        case WS_CandyCane_3: 
            StripLights_Dim(StripLights_DimLevel_2);
            CandyCane(9, StripLights_RED, StripLights_YELLOW );
        break;
        case WS_Twinkle:   
            Twinkle(16);
        break;
        case WS_Icicle:     
            Icicle(1,5 , 6);
        break;
        case WS_ColorWheel:     
            StripLights_Dim(StripLights_DimLevel_2);
            ColorWheel(1);
        break;
        case TextAnimation: 
            DisplayPOVAnimation();
        break;            
        case RGB_Control:   
            // Do nothing, LEDs are controlled by RGB-Data
            break;
        default:
            Active_Animation = 0;
        break;
    }    
}

// Set the new Animation from BLE 
void Main_SelectAnimation( e_Animation NewAnimation ) {
    Active_Animation = NewAnimation;
}

CY_ISR(Main_Startbutton_ISR) {
    Start_ClearInterrupt();
    START_IRQ_ClearPending();
    
    if(Debounceflag) return; 
    
    Debounceflag = true;
    CyDelay(5);
    
    if( TriggerDisplayFromButtonPressOrProgrammatically ) {
        TriggerDisplayFromButtonPressOrProgrammatically = false;    
    }else {
        TriggerDisplayFromButtonPressOrProgrammatically = true;   
    }
}

void Main_ActivateAnimation(bool NewStatus ) {
    Debounceflag = false;
    TriggerDisplayFromButtonPressOrProgrammatically = NewStatus;
}

void Main_ActivateNotification() {
    SendNotification = true;
}


/*
This function is called from the BLE_process.c when a special value should be stored. 
*/
uint32 Main_UpdatePersistentStorageVariable( S_PersistentStorageUpdate *UpdateValue ) {

    uint32 result = ERROR_VARIABLE_NOT_FOUND; 
    PersistenStorageDataNameEnum switchvalue = UpdateValue->Dataname;
    
    switch(switchvalue) {
        case e_SPEED: 
            // Check für Länge??
            SPEED = UpdateValue->DataValue;
            speedcounter = 0;
            result = 0;
        break;
        
        case e_WDTIMEOUT:   // Only for testing purposes, should not be used...
            WatchdogTimeout = UpdateValue->DataValue;
            WDT_Start(WatchdogTimeout, &Timer_Interrupt );
            result = 0;
        break;
        
        case e_POVANIMATION: 
           POVA_SetActivePOVAnimation( (uint8)UpdateValue->DataValue ) ;  
           result = 0; 
           break;
            
        default: 
            // unused parameter requested. 
            return ERROR_VARIABLE_NOT_FOUND;
        break;
    }
    BackupData = true;
    return result;     
}

void main_WriteVariablesToPersistentStorage( ) {
    PSTOR_WriteUserData( e_SPEED, &SPEED, true );
    PSTOR_WriteUserData( e_WDTIMEOUT, &WatchdogTimeout, true );    
    Main_ActivateAnimation(true);
}

void main_SendBLENotification() {
    if(SendNotification == true)
    {
        SendNotification = false;
        BLE_WriteMovementReceivedNotification(&TextWriteCounter);           
    }    
}

/* [] END OF FILE */
