/* Multifunction Digital thermometer
 * ---------------- 
 *
 * Mode 0: Thermometer
 * Reads the analog signal from a LM35DZ then outputs the
 * average of the past 10 temperature readings to a 7 segment display
 * by pushing button 2 it switches to fahrenheit
 *
 * Mode 1: Chronograph (count up timer)
 * Counts the seconds from when button 2 is pushed. Push it again
 * to stop, and once more to reset.
 *
 * Mode 2: Timer (countdown)
 * counts down to 0 from the set time.
 * can be set by changing the variable resistor to the tens of
 * seconds wanted, then push button 2 to set. Next, slide the
 * variable resistor to change the number to the ones of seconds
 * and puch button 2 again to start timing.
 *
 * Mode 3: Demo Mode
 * Displays various flasing light patterns
 * Push button 2 to cycle between the patterns
 * 
 * Created 2 April 2007
 * copyleft 2007 Jonathan Hodgins
 */

int tempPin=0;//the pin the thermistor is attached to
double temp=0;//a variable to store the input temperature
//a setup to pass either true or false, but so it is easier to read
const boolean celcius = true, fahrenheit = false;
const int numAvg = 10;//the number of temperatures readings to store to average
double pastRead[numAvg];//an array to store the temperature readings

int sliderPin=1;//the pin the variable resistor is attached to
int sliderValue=0;//a variable to store the value of the variable resistor
int demoCounter=0;//a variable to count through various states of the light flashing

long startTime=millis();
long currentTime=millis();
int waitA=0, waitB=0, waitTime=0;

//variables to change the mode(button 1) and sub mode(button 2)
int mode=0,modePin =12, modePinState=LOW;
const int numModes=4;
const int numSubModes[numModes]={
  2,3,4,4};
int subMode=0, subModePin =13, subModePinState=LOW;

//a setup to make reading easier later on
const bool on=true,off=false;
bool dimmer = on;//this controls if the slider can control the brightness of the screen
const int numOutputs = 2;//the number of numbers attached to the microcontroller
int segPins[8]={
  2,3,4,5,6,7,8,9};//the pins segments A - H are attached to
int seg[numOutputs][8];//a two dimemsional array to store which sigments are bright for each number
int cathodes[]={
  10,11};//the pins the common cathodes are attached to

int del = 100;//time between temperature readings
const int fps = 100;//the number of 
int delayFrame=(1000/fps)/numOutputs;//sets the ms to delay for each number

void setup() 
{
  for(int i=0;i<numAvg;i++)//sets the temp to zero to start
    pastRead[i]=0;

  for(int i=0;i<9;i++)//sets the display pins to outputs ans sets them so no light shows
  {
    pinMode(segPins[i],OUTPUT);
    digitalWrite(segPins[i],HIGH);
  }

  pinMode(modePin,INPUT);//sets the mode pin(button 1) to input
  pinMode(subModePin,INPUT);//sets the submode pin(button 2) to input

  for(int i=0;i<numOutputs;i++)//sets all of the segmants to low so no light shows
    for(int j=0;j<8;j++)
      seg[i][j]=LOW;
}

//displays the output on the segments
//@ms_delay the time to display the output
void display(int ms_delay)
{

  long clock = millis();//gets the milliseconds since the program started
  while (millis()-clock < ms_delay)//loop while the current time looping is less then the delay time
  {
    // it shoulld be noted that since I used a common cathode High in the segment pins means no light
    // that can be changed here if you used a common anode(more common), just change all highs to lows and vice versa
    for(int number=0;number<numOutputs;number++)
    {
      digitalWrite(cathodes[number],HIGH);
      for (int segNum=0;segNum<=8;segNum++)
      {
        if(seg[number][segNum]==HIGH)
          digitalWrite(segPins[segNum],LOW);
      }
      if(dimmer)
        delay(abs(delayFrame - ((0.0072*sliderValue)-2.3176)));
      else
        delay(delayFrame);
      digitalWrite(cathodes[number],LOW);
      if(dimmer)
        delay(abs((0.0072*sliderValue)-2.3176));
      for (int segNum=0;segNum<=8;segNum++)
      {
        digitalWrite(segPins[segNum],HIGH);
      }
    }
  }
}

/*converts double type to intergers
@param a the variable to change
@return the rounded number(automatically done)*/
int double_to_int(double a)
{
  return a;
}

/*takes the reading from the thermistor, converts it to the relative celcius and stores it
@return the average of the stored temperatures*/
double takeTemp()
{
  for(int i=numAvg-1;i>0;i--)
    pastRead[i] = pastRead[i-1];
    
  //This is the formula to change to your calibrated thermistor
  pastRead[0] = (0.4837*analogRead(tempPin))-1.1163;
  double avg=0;
  for(int i=0;i<numAvg;i++)
    avg += pastRead[i];
  avg = avg/numAvg;

  return avg;
}

/* a general function to call for temperatures
@param T the average temperature
@inCelcius a ture or false to tell the function if it is to display celcius or fahrenheit*/
void displayTemp(double T, boolean inCelcius)
{
  if (!inCelcius)
  {
    T=(T*(1.8)) + 32;//the formula to convert from celcius to fahrenheit
  }
  if (T>=10 && T <100)
  {
    changeDisplay(T/10,0);
    seg[0][7]=LOW;
    changeDisplay( double_to_int(T)%10,1);
  }
  else if (T>=0 && T < 10)
  {
    changeDisplay(T,0);
    seg[0][7]=HIGH;
    changeDisplay(double_to_int(T*10)%10,1);
  }
  else if (T<0 && T > -10)
  {
    changeDisplay(-1,0);
    changeDisplay(double_to_int(T),1);
  }
  else//display two lines if there is an error
  {
    changeDisplay(-1,0);
    changeDisplay(-1,1);
  }

}

//the demo light mode
void demo()
{
  for(int i=0;i<numOutputs;i++)//automatically sets all segment to no light
  {
    seg[i][0]=LOW;
    seg[i][1]=LOW;
    seg[i][2]=LOW;
    seg[i][3]=LOW;
    seg[i][4]=LOW;
    seg[i][5]=LOW;
    seg[i][6]=LOW;
    seg[i][7]=LOW;
  }
  if (subMode==0)//displays single segments going in circles
  {
    seg[0][5-demoCounter]=HIGH;
    demoCounter = (demoCounter+1)%6;
    seg[1][demoCounter]=HIGH;
  }
  else if (subMode==1)//a light shimmers across two digits
  {
    demoCounter = (demoCounter+1)%10;
    if(demoCounter==0)
    {
      seg[0][4]=HIGH;
      seg[0][5]=HIGH;
    }
    else if(demoCounter==1 || demoCounter==2)
    {
      seg[demoCounter-1][0]=HIGH;
      seg[demoCounter-1][3]=HIGH;
      seg[demoCounter-1][6]=HIGH;
    }
    else if(demoCounter==3)
    {
      seg[1][1]=HIGH;
      seg[1][2]=HIGH;
    }
  }
  else if (subMode==2)//a bar of light climbs up
  {
    demoCounter = (demoCounter+1)%3;
    seg[0][demoCounter*3]=HIGH;
    seg[1][demoCounter*3]=HIGH;
  }
  else if (subMode==3)//light appears in a circle until all on then cycles off
  {
    demoCounter = (demoCounter+1)%12;

    if(demoCounter<6)
      for(int i=0;i<=demoCounter;i++)
      {
        seg[0][(i+1)%6]=HIGH;
        seg[1][5-i]=HIGH;
      }
    else
      for(int i=5;i>=demoCounter%6;i--)
      {
        seg[0][(i+1)%6]=HIGH;
        seg[1][5-i]=HIGH;
      }

  }
}

//countup timer
void chrono()
{
  if(subMode==0)//Reset(or start)
  {
    startTime=millis();
    changeDisplay(0,0);
    changeDisplay(0,1);
  }
  else if(subMode==1)//Time
  {
    currentTime=millis();
    changeDisplay(((currentTime-startTime)/1000)/10,0);
    changeDisplay(((currentTime-startTime)/1000)%10,1);
    if((currentTime-startTime)/1000>=99)
      subMode++;
  }
  else if(subMode==2)//Stop
  {
    changeDisplay(((currentTime-startTime)/1000)/10,0);
    changeDisplay(((currentTime-startTime)/1000)%10,1);
  }
}

//The timer function
void timer()
{
  if(subMode==0)//set digit 0
  {
    waitA=abs(-(0.013*sliderValue)+13.5);
    waitB=0;
    changeDisplay(waitA,0);
    changeDisplay(waitB,1);
  }
  else if(subMode==1)//set digit 1
  {
    waitB=abs(-(0.013*sliderValue)+13.5);
    changeDisplay(waitA,0);
    changeDisplay(waitB,1);
    waitTime=(waitA*10)+waitB;
    startTime=millis();
  }
  else if(subMode==2)//start
  {
    currentTime=millis();
    changeDisplay((waitTime-((currentTime-startTime)/1000))/10,0);
    changeDisplay((waitTime-((currentTime-startTime)/1000))%10,1);
    if((waitTime-((currentTime-startTime)/1000))<1)
      subMode++;
  }
  else if(subMode==3)//alarm(well, not really :)
  {
  }
}

//The main program loop
void loop()
{
  //reads the buttons
  modePinState = digitalRead(modePin);
  subModePinState = digitalRead(subModePin);
  sliderValue=analogRead(sliderPin);

  if(modePinState==HIGH)//changes the mode if button 1 is pushed
  {
    mode = (mode + 1)%numModes;
    subMode=0;
    while(modePinState==HIGH || subModePinState==HIGH)//don't leave until released
    {
      modePinState = digitalRead(modePin);
      subModePinState = digitalRead(subModePin);
      display(del/2);
    }
  }
  else if(subModePinState==HIGH)//changes the submode if button 2 is pushed
  {
    subMode = (subMode + 1)%numSubModes[mode];
    while(modePinState==HIGH || subModePinState==HIGH)//don't leave until buttons are released
    {
      modePinState = digitalRead(modePin);
      subModePinState = digitalRead(subModePin);
      display(del/2);
    }
  }

  if(mode==0)//temperature reading
  {
    temp = takeTemp();
    if (subMode==0)
      displayTemp(temp,celcius);
    else if (subMode=1)
      displayTemp(temp,fahrenheit);
    dimmer = on;
    display(del);
  }
  else if(mode==1)//Count Up Timer(chronograph)
  {
    chrono();
    dimmer = on;
    display(del);
  }
  else if(mode==2)//Count Down Timer
  {
    timer();
    dimmer = off;
    display(del);
  }
  else if(mode==3)//Demo Mode
  {
    demo();
    dimmer = off;
    display(sliderValue/8);
  }
}

/*updates the display when called
@param changeTo, the number from 0-9 to change to, otherwise displays a line
@param num, the segment to change*/
void changeDisplay(int changeTo,int num)
{
  switch(changeTo)
  {
  case 0:
    seg[num][0]=HIGH;
    seg[num][1]=HIGH;
    seg[num][2]=HIGH;
    seg[num][3]=HIGH;
    seg[num][4]=HIGH;
    seg[num][5]=HIGH;
    seg[num][6]=LOW;
    break;
  case 1:
    seg[num][0]=LOW;
    seg[num][1]=HIGH;
    seg[num][2]=HIGH;
    seg[num][3]=LOW;
    seg[num][4]=LOW;
    seg[num][5]=LOW;
    seg[num][6]=LOW;
    break;
  case 2:
    seg[num][0]=HIGH;
    seg[num][1]=HIGH;
    seg[num][2]=LOW;
    seg[num][3]=HIGH;
    seg[num][4]=HIGH;
    seg[num][5]=LOW;
    seg[num][6]=HIGH;
    break;
  case 3:
    seg[num][0]=HIGH;
    seg[num][1]=HIGH;
    seg[num][2]=HIGH;
    seg[num][3]=HIGH;
    seg[num][4]=LOW;
    seg[num][5]=LOW;
    seg[num][6]=HIGH;
    break;
  case 4:
    seg[num][0]=LOW;
    seg[num][1]=HIGH;
    seg[num][2]=HIGH;
    seg[num][3]=LOW;
    seg[num][4]=LOW;
    seg[num][5]=HIGH;
    seg[num][6]=HIGH;
    break;
  case 5:
    seg[num][0]=HIGH;
    seg[num][1]=LOW;
    seg[num][2]=HIGH;
    seg[num][3]=HIGH;
    seg[num][4]=LOW;
    seg[num][5]=HIGH;
    seg[num][6]=HIGH;
    break;
  case 6:
    seg[num][0]=HIGH;
    seg[num][1]=LOW;
    seg[num][2]=HIGH;
    seg[num][3]=HIGH;
    seg[num][4]=HIGH;
    seg[num][5]=HIGH;
    seg[num][6]=HIGH;
    break;
  case 7:
    seg[num][0]=HIGH;
    seg[num][1]=HIGH;
    seg[num][2]=HIGH;
    seg[num][3]=LOW;
    seg[num][4]=LOW;
    seg[num][5]=LOW;
    seg[num][6]=LOW;
    break;
  case 8:
    seg[num][0]=HIGH;
    seg[num][1]=HIGH;
    seg[num][2]=HIGH;
    seg[num][3]=HIGH;
    seg[num][4]=HIGH;
    seg[num][5]=HIGH;
    seg[num][6]=HIGH;
    break;
  case 9:
    seg[num][0]=HIGH;
    seg[num][1]=HIGH;
    seg[num][2]=HIGH;
    seg[num][3]=HIGH;
    seg[num][4]=LOW;
    seg[num][5]=HIGH;
    seg[num][6]=HIGH;
    break;
  default:
    seg[num][0]=LOW;
    seg[num][1]=LOW;
    seg[num][2]=LOW;
    seg[num][3]=LOW;
    seg[num][4]=LOW;
    seg[num][5]=LOW;
    seg[num][6]=HIGH;
    seg[num][7]=LOW;
    break;
  }
}
