/******************************************************
 MCU Test - 4x4	NoISR	does Not use an Interrupt Service Routine for anything
    Basic Testing of any MCU, with attached 4 LEDs  + 4 buttons using only 5 I/O lines (Audio separately optional)
    Test Operations execise Buttons, Leds, timing, interrupts, EEPROM storage   and optionally buzzer &/or speaker
    from: https://www.instructables.com/member/RonM9
    
 Ron Miller Feb 2018
*******************************************************/

#include <avr/interrupt.h>
#include <avr/sleep.h>

#include <EEPROM.h>
// the address in the EEPROM we're going to write to
int eeCntAddr = 77;  // totally arbitary
int eePerfAddr = 80;
unsigned long perfCnt;

// constants won't change. They're used here to set pin numbers:
// =================  hardware dependant section  ===================
// Changes are done here when components or their configuration chages
// these pin assignments are for AVR MSUs
#define ONBOARD_LED 13     // the number of the LED 'L' on the board
#define UNUSED_A0 A0
#define PROBE A1

bool volHigh = false;   // only useful with speakers, and some passive piezo
// speaker with a sm inline cap. are generally a little louder. (+ provides level isolation)
// but sm inline cap. significantly reduces effect of 'volHigh'

/*************** Standard Fixture Active HIGH configuration for UNO/Nano/miniPro **********/
    // if using an Active Buzzer set to 'true' else when using a passive 'buzzer' or Speaker set to 'false'
    #define ACTIVE_BUZZER false
    const byte BEEPPIN = 12;
    const int SPEAKERPIN = 12;
    const int SPEAKERNEG = 13;
    const int button[] = {2, 3, 4, 5}; //The four button input pins
    #define PRESSED_STATE 1
    #define DIO_SHARING
    #define BTN_ENB 6
    const int nleds=4;
    const int lites[]  = {2, 3, 4, 5};     //the LED pins
    #define ON_STATE 1
    const byte colorSet[] = { 0, 0x1, 0x2, 0x4, 0x8, 0x0F };

/*************** Active LOW configuration Might look like this ***********************
    // if using an Active Buzzer set to 'true' else when using a passive 'buzzer' or Speaker set to 'false'
    #define ACTIVE_BUZZER true
    const int button[] = {16, 17, 18, 19}; //The four button input pins
    #define PRESSED_STATE 1
    #define DIO_SHARING
    #define BTN_ENB 15
    const int nleds=4;
    const int lites[]  = {16, 17, 18, 19};     //the LED pins
    #define ON_STATE 1
    const byte colorSet[] = { 0, 0x1, 0x2, 0x4, 0x8, 0x0F };
/**************** for testing with My old little Black-Box Project  ******************
    #define ACTIVE_BUZZER false
    const byte BEEPPIN = 13;
    const int SPEAKERPIN = 13;
    const int SPEAKERNEG = 12;
    const int button[] = {19, 18, 17, 16}; //The four button input pins
    #define PRESSED_STATE 0
    // NO DIO_SHARING
    #define BTN_ENB 0
    const int nleds=8;
    #define ON_STATE 0
    const int lites[] = {2, 3, 4, 5, 6, 7, 8, 9};     //the LED pins
    const unsigned int colorSet[] = { 0, 0x03, 0x0C, 0x30, 0xC0, 0xFF };
/********** My 'Home Station' larger project box ************************************
    const int SPEAKERPIN = 14;
    const byte BEEPPIN = 14;
    const int SPEAKERNEG = 4; // acually for the btns
    #define ACTIVE_BUZZER true
    const int button[] = {5, 6, 7, 8}; //The four button input pins
    #define PRESSED_STATE 0
    // NO DIO_SHARING
    #define BTN_ENB 0
    const int nleds=4;
    #define ON_STATE 1
    const int lites[] = {2, 3, 17, 16};     //the LED pins
    const byte colorSet[] = { 0, 0x1, 0x2, 0x4, 0x8, 0x0F };
/*****/

const bool debugPrt=false;
// =====================================================
// Button stuff

bool btnChanged = false;
bool btn1Changed = false;
bool btn2Changed = false;
bool btn3Changed = false;
bool btn4Changed = false;

// made into a function: #define btnPressed (btnChanged && Btn)
#define btn1Pressed (btn1Changed && btn1)
#define btn2Pressed (btn2Changed && btn2)
#define btn3Pressed (btn3Changed && btn3)
#define btn4Pressed (btn4Changed && btn4)

#define  DEBOUNCE_CNT 2
int debounceCnt;
byte btnState = 0, btnNum, priorBtn;
bool btn1 = false;
bool btn2 = false;
bool btn3 = false;
bool btn4 = false;
bool Btn=false; // Buttons collectively
bool ESC = false;
bool time2Escape = false;
int x;

volatile unsigned long myMillis=0; 
unsigned long msec, usecs, lastMsec; 
unsigned int mySecs=0, secsIdle=0;

unsigned int msCnt;
unsigned long t0,t1,t2;
int litPeriod, period, cnt, maxCnt, lastMaxCnt, savedCnt;
bool isLit, incDone = false, decDue = false;
byte level, goal;
#define STARTPERIOD 330;

int escCnt=0;
unsigned int idleCnt=0, loopCnt;
unsigned long tt, t1000, loopTime;
bool initDone=false;

// =====================================================
//   LED display stuff

bool lit[14], dim[14], brt[14];  // 12 (1-12) plus [0] &[13] for over flow calculations
byte levelOn[14]; // for lighting 0-10 levels
byte cell[14];  // working buffer
bool dimOn; // true if any led is 'dim' on
int spriteOne, spriteTwo; // mobile hightlights
int heartBeat=0;
int darkOne;      // a mobile dark spot, normally momentary
int spritePtr;    // sprite that is not seen
int cursorPtr;
int litLED=0; // the number of The 'lit' Led
int dimIt=0; // the number of the led to 'dim' off 
int flashLed, blinkLed;
byte allColor=0;  //  0: none, bits: 1:RED, 2:YEL, 3:GRN, 4:BLUE, 5:ALL
int nOff=0;

byte func, nextFunc;
int algo;
int Vcc, v3;

/***************
//======================================================
//Timer2 Overflow Interrupt Vector, called every 1ms
ISR(TIMER2_OVF_vect) {
  TCNT2 = 130;           //Reset Timer to 130 out of 255
  TIFR2 = 0x00;          //Timer2 INT Flag Reg: Clear Timer Overflow Flag
  myMillis++;
  if (myMillis%1000 == 0) { // every 1 second
    if (heartBeat) spriteOne=(spriteOne==0)? heartBeat : 0;
    mySecs++;
    secsIdle++;
  }
  oneMilliUpdate();
};
************************/

//======================================================
long readVcc() {
  ADMUX = _BV(REFS0) | _BV(MUX3) | _BV(MUX2) | _BV(MUX1);
  delay(2); // Wait for Vref to settle typical: +/-.03 (1) is good @ +/-.06
  ADCSRA |= _BV(ADSC); // Start conversion
  while (bit_is_set(ADCSRA,ADSC)); // measuring

  delay(2); // settling
  uint8_t low  = ADCL; // must read ADCL first - it then locks ADCH  
  uint8_t high = ADCH; // unlocks both
 
  long result = (high<<8) | low;
 
  result = 1125300L / result; // Calculate Vcc (in mV); 1125300 = 1.1*1023*1000
//  result = 1126400L / result; // Calculate Vcc (in mV); 1126400 = 1.1*1024*1000
  return result; // Vcc in millivolts
}

//======================================================
int checkVoltage() {
  Vcc=readVcc();  // print out the voltage (Vcc) value
  Serial.print("Vcc = "); Serial.print(Vcc/1000); Serial.print("."); Serial.println(Vcc%1000); 

  if (Vcc<2700) {
    clearDisp();
    spriteOne=1;
    beep(50); refreshWait(200); beep(100);
    spriteOne=0; // save the 1ma of this led
    refreshWait(2);  // get the disp cleared
    //  !!!!!!!!!!!!!!!!!!!  full Stop  !!!!!!!!!!!!!!!!!!!!!!!!!!
    set_sleep_mode(SLEEP_MODE_PWR_DOWN);  // for Arduino w/USB SIO chip <5ma   else <1ma maybe
    cli();  // disable interrupt processing
    sleep_enable();
    sleep_cpu();
    // goes into a comma //
  }
  return (Vcc);
}

// the setup routine runs once on power up or when you press reset:
//===============================================================================
void setup() {
  pinMode(BEEPPIN, OUTPUT);  // declare BEEPPIN to be an output:
  pinMode(SPEAKERNEG, OUTPUT); 
  digitalWrite(SPEAKERNEG, LOW);
  
  // initialize serial communication and its Baud rate
  Serial.begin(115200);
  Serial.println("        Test - 4x4 without Interupts ");

  pinMode(ONBOARD_LED, OUTPUT);
//  digitalWrite(ONBOARD_LED, HIGH);  also goes to buzz in  this project so don't do this
  pinMode(A0, INPUT);   // used as floating open input for real world generated Randomness
  pinMode(A7, INPUT);
  pinMode(PROBE, INPUT);

  pinMode(SPEAKERPIN, OUTPUT);
  // remove these when not needed, given wired grounds
  pinMode(SPEAKERNEG, OUTPUT);
  digitalWrite(SPEAKERNEG, LOW);

  if (BTN_ENB!=0) {
    pinMode(BTN_ENB,OUTPUT); 
    digitalWrite(BTN_ENB, !PRESSED_STATE);
  }

  for (int x = 0; x < 4; x++) // Button pins are inputs
  {
    pinMode(button[x], INPUT);  // button pins are inputs
    digitalWrite(button[x], HIGH);  // enable internal pull-up; buttons start in high position; negative logic
  }

//  nleds = 6;
  for (int x = 0; x < nleds; x++) // LED pins are outputs
  {                               // if buttons & leds
    pinMode(lites[x], OUTPUT);
    digitalWrite(lites[x], (ON_STATE==LOW));  // start w/ LEDs off
  }

  randomSeed(analogRead(0)); //Added HW generated "more randomness" to the random() function

  func=1;
  nextFunc=0;
  //nextFunc=4; // ********* if you want to automatically start with a given function set its # here
  
  pinMode(BEEPPIN, OUTPUT);  // declare BEEPPIN to be an output:
  digitalWrite(lites[0], ON_STATE);   beep(60); delay(440);  
  digitalWrite(lites[1], ON_STATE);   Beep(60); delay(440);  
  digitalWrite(lites[2], ON_STATE);   buZZ(40); delay(460); // give a little less as it is more potent
  digitalWrite(lites[3], ON_STATE);   genTone(1600,60);
  delay(500);
  //  display update should later turn the LEDs back off

/************  
  cli();    // Disable interrupts
  //Setup Timer2 to fire every 1ms
  //------------------------------
  TCCR2B = 0x00;        //Disable Timer2 while we set it up
  TCNT2  = 130;         //Reset Timer Count to 130 out of 255
  TIFR2  = 0x00;        //Timer2 INT Flag Reg: Clear Timer Overflow Flag
  TIMSK2 = 0x01;        //Timer2 INT Reg: Timer2 Overflow Interrupt Enable
  TCCR2A = 0x00;        //Timer2 Control Reg A: Wave Gen Mode normal
  TCCR2B = 0x05;        //Timer2 Control Reg B: Timer Prescaler set to 128
  sei();    // Enable interrupts
****************/

  clearDisp();
  refreshWait (500);
  Vcc = checkVoltage();
  v3 = (Vcc-3000)/100;
  if (v3<0) v3=0;
  showLevel(v3);   // display # of 0.1v above 3.0v (note: the ISR needes to be setup prior)

  // read previous incarnation of 'max count' from EEPROM for the 'Push-It' game
  EEPROM.get(eePerfAddr, perfCnt);
  Serial.print(" eePROM recovered Perfornace Rating:  "); Serial.print(perfCnt); Serial.println(" MCU Marks,  A+D updates/sec");
  
  t0=millis();
  delay(1000);
  Serial.print(" #ms delay(1000):  "); Serial.println(millis() - t0);
  
  refreshWait(1000);
  heartBeat=1;
  initDone=true;
}


// the loop routine runs over and over again forever:
// ===============================================================================
void loop() {
  unsigned int x;
  int pos, lastPos;
  int n=0, cnt=0, cntMax=0;
  bool toggle;
  int dir[4];
  int maxDir, maxSum;
  unsigned int sum;
  int tst[4];
  byte ans, keyCode;
  bool ackd, pause, equal;
  unsigned int mCnt;
  int minCnt;
  byte hits1,hits4;
  bool doit;
  bool btnOccured;
  long pattern;
  byte patcell[14];
  int divr, state;
  int deltaOne, deltaPtr, beet;
  bool gMute;
  byte speed;
  int spdCnt[] = {1000,500,250,100};
  byte maxFunc;

  digitalWrite(ONBOARD_LED, LOW);
  clearDisp();  refreshWait(1000);
  cursorPtr = 0;

  while(heartBeat && !Btn) {UI_update();}  // wait for first button press
  heartBeat=0;

  // Indicate posible selects, function 1-5
  dim[1]=dim[2]=dim[3]=dim[4]=dim[5]=true;
  spritePtr=1;

  maxFunc=4; // The number of functions to selected from

  UI_update();
  while(Btn) {UI_update();} // wait for no button

  if (nextFunc) {
    spritePtr = nextFunc;   // use 'spriteOne' as the working selection
    nextFunc = 0;
  } else {
    while (! btn1) {
      UI_update();
      if (btn2Pressed) spritePtr = (spritePtr==1)?  nleds : spritePtr-1;  // -- move left
      if (btn3Pressed) spritePtr = (spritePtr%maxFunc) + 1; // ++ moves right
      if (btn4Pressed) spritePtr = (spritePtr%maxFunc) + 1; // allow most play with only 2 buttons (1&4)
      darkOne = ((myMillis%750)<300 && spritePtr<5)? spritePtr : 0;
      dimIt = (spritePtr>4)? spritePtr : 0; 
    }
  }

  darkOne=dimIt=0;
  func = spritePtr;
  clearDisp();
  level=0; 
  while(Btn) UI_update(); // wait for button release

  cnt = 0;
  pause=ackd=false;
  nextFunc = 0;
  Serial.print(" FUNC: "); Serial.println(func);
  switch (func) {
    case 1:
      //  =====================================
      btnReporting();
      break;

    case 2:
      //  =====================================
      Scales();
      //pingPong();
      break;

    case 3:
      Timer();
      break;

    case 4:
      PerfTest();
      EEPROM.put(eePerfAddr, perfCnt);
      break;

    case 5:
      //PushIt();
      //tuneColorOrgan();
      break;

    case 6:
      break;

  }  // end of switch structure
  allColor=5;
  Beep(50);

}

// ************************  broke out Functions  *******************************

// ************************   Loops   *******************************

void btnReporting() {
  short int cnt, aVal, avTemp;
  bool invert=false;

  Beep(100);
  clearDisp();

  priorBtn = -1;  // prevent false double-ESC detection 
  wait_BtnRelease(); // wait for no button
  
  while (1) {
    // highlight any change with the onBrd Led
    digitalWrite(ONBOARD_LED, btnChanged);
    if (invert) {
        brt[1] = !btn1;   // button causes associated LED to go OFF
        brt[2] = !btn2;
        brt[3] = !btn3;
        brt[4] = !btn4;
    } else {
        lit[1] = btn1;   // button causes associated LED to turn ON
        lit[2] = btn2;
        lit[3] = btn3;
        lit[4] = btn4;
    }
    
    //delay(1);  // extra time extends ECS timeout peroid (1.5 -> 3 secs)
	//    update_ledDisplay();  // following btnPressed() will keep display refreshed
    // ------------------- button reporting
    if (btnPressed()) {
      Serial.println(" ");
      if (btn1) Serial.print("[Btn1] "); else Serial.print("       ");
      if (btn2) Serial.print("[Btn2] "); else Serial.print("       ");
      if (btn3) Serial.print("[Btn3] "); else Serial.print("       ");
      if (btn4) Serial.print("[Btn4] "); else Serial.print("       ");
      if (time2Escape) {    // new btn pushed while in Esc timeout state
        if (btn1) {Beep(50); return;}
        if (btn2) invert = !invert;
        clearDisp();
      }
    }
    if (btnChanged) beep(10); // make a key press click, for up&down
    if (ESC && spriteOne!=4) Serial.print("[ESC] ");    
    spriteOne = (ESC)? 4 : 0;
  }
}

void  Scales()  { //  =====================================   Scales thru Leds and audio levels
  // delays & Display Refreshe timing is critical to proper display, as a changing myMillis is used to fill the display 
  // when using interupt driven display updates this is not an issue.
  int effect, cnt, dir, lvl;
  bool tomove, waiting;
  byte n, c, lastN, lastC;
  unsigned int tCnt;

    escCnt=ESC=0;
      while (!ESC) {

        clearDisp();
        lvl=0;
        wait_BtnRelease();
        waiting=true;
        while (!btn4) {  // -------------------------------- step thru standard set of lights and sound
          // delay(1);
          UI_update();
          if (waiting && !Btn) {
            // ------------- create a 'Waiting' display banner
            n=(myMillis/1000)%nleds;
            //c = (n&1)? 1 : nleds;                 // toogle between 1st and last Led
            //c = (n/(nleds/2)) + 2*(n%4);            // Step thru odd then even Leds
            c =(nleds/2)*(n&1) + (n>>1);             // pop back&fore lower-upper section
            //c = (n&0xFC) + ((n<<1)&2) + ((n>>1)&1);  // reverse the lower two bits, ~ skippy travel
            dimIt=c+1;
          } else {
            dimIt=0;
            waiting=false;
            // lvls: 0=off,  1=dim,  2=lit,  3=brt,  4=running flash
            if (btn2Pressed) lvl = (lvl==0)?  4 : lvl-1;  // -- down
            if (btn3Pressed) lvl = (lvl+1) % 5;           // ++ up
            dim[myMillis%nleds+1] = (lvl==1);  // note this is dependant on ISR, to fill over time
            lit[myMillis%nleds+1] = (lvl==2);
            brt[myMillis%nleds+1] = (lvl>=3);
            if (lvl==4) {
              for (int i=1; i<=nleds; i++) {  // flash odd & even LEDs in opposition
                lit[i] = (((myMillis/333)%2) == (i&1));
              }
              darkOne = (myMillis/100)%(nleds+1);
//              delay(1);
            } else darkOne=0;

            if (Btn && !btn4) {
              while (Btn && lvl==1) {genTone(150,20); UI_update();}
              while (Btn && lvl==2) {beep(20); UI_update();}
              while (Btn && lvl==3) {Beep(20); UI_update();}
              while (Btn && lvl==4) {buZZ(30); UI_update();}
            }
//            refreshWait(1);
          }
        }
        
        clearDisp();
        lvl=0;
        if (!ESC) wait_BtnRelease();
        waiting=true;
        while (!btn4) {  // ---------------------- step thru the scale of 10 tones and levels of lit
          //delay(3);        
          UI_update();
          if (waiting && !Btn) {
            n=(myMillis/1000)%nleds;
            //c = (n&1)? 1 : nleds;                 // toogle between 1st and last Led
            //c = (n/(nleds/2)) + 2*(n%4);            // Step thru odd then even Leds
            //c =(nleds/2)*(n&1) + (n>>1);             // pop back&fore lower-upper section
            if (n!=lastN) c += (n&0x01)*3 -1;        // 2 forward, 1 back, ~ skip travel
            //c = (n&0xFC) + ((n<<1)&2) + ((n>>1)&1);  // reverse the lower two bits, ~ skippy travel
            dimIt=(c%nleds)+1;
            lastN = n;

          } else {
            dimIt=0;
            allColor=0;
            waiting=false;
            if (btn2Pressed) lvl = (lvl==0)?  10 : lvl-1;  // -- down
            if (btn3Pressed) lvl = (lvl+1) % 11; // ++ up
            levelOn[myMillis%nleds+1]=lvl;

            if (btn2Pressed || btn3Pressed) {
              flashLed =  (lvl>0)? (lvl-1)%nleds+1 : 0;
            }

            if ((btnChanged && Btn) || btn1) playNote(5+lvl,150);
            //delay(1);
          }
//          refreshWait(1);
        }

        cnt=1;
        dir=1;
        tomove=true;
        clearDisp();
        effect=1;
        wait_BtnRelease();
        while (!btn4) {
            switch (effect) {
            case 1:
              // ---------------------- a Cylon Centurion helmet scan (also interesting would be Kit 2000 grill display)
              //sweeps a group of LEDs back&forth with a Wiz-a-Wom ...     ~1sec to cross
              // every 1/6 sec move lite, 1st right then left accross display, & appears to overrun
              // hold for 1 counts on the ends
              // with sweeping noise
//			  msec= myMillis;
              msec=millis(); // use instead of myMillis so as not to be interupt ISR dependant
              if (tomove && (msec%(1000/(nleds))<100))  {  // cadence to result in ~1sec across displ
                tomove=false;
                //if (!(cnt==0 || cnt==nleds+1)) {  // 1 cnt pause on ends
                if (!(cnt<1 || cnt>nleds)) {  // setup 2 cnt pause on ends, special to work with only 4 leds  !!!
                  lit[cnt]=lit[cnt+dir]=true;
                  //delay(50);          // allow for dim tail
                  lit[cnt-dir]=false; //
                }
                cnt=cnt+dir;
                //if (cnt<0) {cnt=1; dir=1;}
                //if (cnt>(nleds+1)) {cnt=nleds; dir=-1;}
                if (cnt<-1) {cnt=1; dir=1;}               // for use to work with 4 leds  !!!
                if (cnt>(nleds+2)) {cnt=nleds; dir=-1;}   //  "
//                refreshWait(1);
              }
              if (msec%(1000/(nleds))>100) tomove=true;
              // produce sound of WhiZ-a-woom
              if ((dir==1 && cnt==1)         || (dir<0 && cnt==nleds))      genTone(150,20);  // Wiz
              if ((dir==1 && cnt==2)         || (dir<0 && cnt==(nleds-1)))  genTone(200,20);  // -a-
              if ((dir==1 && cnt>2 && cnt<5) || (dir<0 && cnt>(nleds-4) && cnt<(nleds-1)))  genTone(100,20);  // woom
              break;
          case 2:   // Simon like lights and sound
            n=(myMillis/750)%4;
            c=((n&1)<<1) + (n>>1);  // reverse the two bits
            allColor = c+1;
            if (c!=lastC) playNote(8+2*c, 200);
            lastC=c;
            break;
          case 3:   // futuristic effect, simplied for space
              cnt=(cnt+1)%32;
              n = (cnt<16)? cnt : 31-cnt;
              litLED=n%nleds+1;
              genTone(400+100*n, 100);
              break;
          }
          //delay(1);
          UI_update();
          if (btnNum) effect=btnNum;
          if (btnChanged && !Btn) clearDisp();
        }

        if (!ESC) wait_BtnRelease();
      }
}


void  Timer()  { //  =====================================   Timer  (+ Stop-Watch functions)
                 // modified to track sets of 15 seconds, for use with a display of only 4 LEDs
  int  pos, minCnt, secs, n;
  bool toggle, timing, paused;
  unsigned int mCnt;
  unsigned long mStart, msecs;

  int ver=1;    // ver 1 uses delay()     ver 2 uses a 1 millisecond ISR interurpt routine

  while (1) {
      clearDisp();
//      xdimEnabled = true;
      mStart=myMillis;
      msecs=millis();
      mCnt=0;
      minCnt=0;
      pos=1;
      toggle=false;
      paused=false;
      timing=true;
      while(timing) {
        secsIdle=0; // block Idle timeout
        if (ver==2) {
          while (mCnt == (myMillis-mStart)) {}  // wait for ISR to change 'myMillis'
          mCnt=myMillis-mStart;
        } else {
          //delay(1);   // ??? this will result in 1-3 secs/min error due to execution time of the rest of the code
          while (msecs==millis()) {;}
          msecs++;
          mCnt++;
        }
        //update_ledDisplay();
        //UI_update();	??? dont want to tick mymillis ???
        oneMilliUpdate();
        scanBtns();
        if (ESC) return;
        if (btn1Pressed) timing=false;    // Restart
        // not herein supported:  if (btn2Pressed) ver=2;     // switch to ver=2 using the ISR
        if (btn4Pressed) paused= !paused;   // toggle in/out of a paused state
        if (paused) {
          wait_BtnRelease();
          while (paused) {
            secs = mCnt%15000 / 1000; // seconds into this 15 sec period
            if (0==secs) {
              spriteOne = pos;
              noBtnDelay(1000);
            } else {
              for (byte s=1; s<=secs; s++) {
                spriteOne = pos;  noBtnDelay(50);
                spriteOne = 0;  noBtnDelay(200);    
              }
            }
            spriteOne = 0;
            noBtnDelay(1000);   // wait upto  1 sec, given no buttons pressed
            if (btnState>0) {
              paused=false;
              wait_BtnRelease();
            }
          }
          if (time2Escape) return;
        }
        if (time2Escape) return;
        
        if (mCnt%500 == 0) { // once a second lite then noLite then adv the sprite
          toggle = !toggle;
          if (toggle) {
            spriteOne =0;
          } else {
//            pos = pos%nleds + 1;
            spriteOne = pos;
          }
        }
          
//        if (ver==1 && mCnt%1000 == 500)  mCnt+=75; // imperical rate adjustment, re given unit, for timing accuracy sake (MCU,overhead,V,temp. ?dependant)
                                                   // mCnt+=16.6 shortens interval by 1/60th (making 1sec difference in a minute)
        if (mCnt%15000 == 0) {    // little beep every 15 seconds
          spriteOne = pos;  // be sure it is lit
          beepNdisp(20);
//          mCnt += 40;     // adj for time spent
          pos = pos%4 + 1;  // 4*15 for 60secs / Min.
          idleCnt=0;  // keep from timing out
        }
        if (mCnt>=60000) {  // every Minute
          mStart=myMillis;
          clearDisp();
          mCnt=1;
          minCnt++;   // inc Minute count
          for (n=1; n<=minCnt; n++) {
            dim[n]=true;
            spriteOne = (n-1)%nleds + 1;
            beepNdisp(50); refreshWait(300);
//            mCnt += 400;  // account for time spent
          }
          litLED=spriteOne; // add extra brightness on the Led of the current Minute
          pos=1 + mCnt/15000;
        }
      }
  }
}

// ==================================================================================================================
// ==============  BENCHMARK Testing             Rons "MCU Marks"
void PerfTest() {  // Preformance Test, does Analog & Digital I/O with Floating Pt. & Integer math with general logic
  // This code is intensionally not optimized, due to benchmarking purposes.
  // AtMega328 16MHz:     MPU Marks = 5100
  // ATtiny-85 16.5MHz:    "    "     4200  apparently it has some compromizes in performance      
  
  bool audLvl=LOW, audio=false, prtEnb=true;
  // without =assignment auLvl may have been true||false But may have been 0x10 (or Undefined?) and not worked for digitalWrite
  unsigned long readings, msecs, mStart, ms, lastms;
  int tenths, rawA, oneCnt, secs=0, u;
  long avgA=0, audCnt=0;
  float v;
  double floatA=0.0;

  clearDisp();
  wait_BtnRelease();
  while (!ESC) {
    digitalWrite(BEEPPIN,LOW);
    digitalWrite(ONBOARD_LED,LOW);
    mStart = 0;
    while (mStart<100)  {  // One second loop
// NOT HEREIN USED:     TIMSK2 = 0x00;        //Disable Timer2 INT
      lastms=msecs = millis();
      while (msecs == millis()) ; // scyn to a starting tick
      while ((ms=millis())-msecs <= 10) {      // Ten milliSec testing loop
        readings++;
        // continuous Analog Reading
        rawA = analogRead(PROBE);
        avgA = (15 * avgA + rawA) >> 4;  // running average math
        floatA = sqrt(((floatA*floatA)  + (float) avgA*avgA) / 2.0);
        // digial write to our Analog In, switching in/out the pull-up to keep line active, for consistent calculation load
        digitalWrite(PROBE, !digitalRead(PROBE)); // do both a digital Read & Write

        if (ms!=lastms) {   // every One millisec
          lastms=ms;
//          digitalWrite(BEEPPIN, ms&1);  // always output even when you get no audio
           digitalWrite(ONBOARD_LED, ms&1);  // always output even when you get no audio
        }
      }
//      TIMSK2 = 0x01;        //Enable Timer2 INT
      
      // ------------------------------  once every Ten milliSec ...
      UI_update();
//    ~  delayMicroseconds(76); // UI_update takes ~76 msecs (or 7.6% of the time on a 8MHz Nano)
//     if (ONBOARD_LED!=SPEAKERPIN && ONBOARD_LED!=SPEAKERNEG) digitalWrite(ONBOARD_LED, audLvl);  // flash the on board led
//      digitalWrite(ONBOARD_LED, rawA<floatA);  // flash the on board led
      if (btn1Pressed) audio = !audio;  // note: audio On brings down reads/sec ~20+%
      if (btn3Pressed) prtEnb = !prtEnb;
      if (btn4) {digitalWrite(BEEPPIN,LOW); return;}

      if (avgA==0) avgA=2;    // defeats execution optimization for operands=0
      v = ((float)Vcc * avgA / 1024.0) / 1000.0;
      // reflect reading on Led display 
      for (int i=1; i<=nleds; i++) {
        brt[i] = (avgA>(i*1024/nleds));
        levelOn[i]=0;
      }
      oneCnt = (1024/nleds); // calc & display fractional portion
      tenths = (avgA % oneCnt) / (oneCnt/10);
      levelOn[avgA/oneCnt+1] = tenths;            // as 1 of 10 lit level
      
      // Refresh Display.  now done in parallel once/msec via ISR
//      delayMicroseconds(20);  // fine adjustment, so that displUpdating accounts for 121 microsecs (same as direct calling)
      mStart++;
    }
    
    // --------------------------------  once a Second ...
    // the following is outside the test loop, & dont effect results 
    clearDisp();
    refreshWait(333);
    showLevel(readings/1000);
    perfCnt = readings;
    if (4==secs++) {
      //EEPROM.put(eePerfAddr, perfCnt);
//      EEPROM.write(eePerfAddr, perfCnt & 0xFF); 
//      EEPROM.write(eePerfAddr+1, perfCnt >> 8);
      EEPROM.put(eePerfAddr, perfCnt);
      delay(10); 
    }

    // output updated readings and stats
    if (prtEnb) {
      // having the print of v & floatA are required, so their calculations are not optimized out
      Serial.print("Average Ain = "); Serial.print(floatA);
      Serial.print("   Vin = "); Serial.print(v); Serial.print("v  ");
      Serial.print("       Reads/Seconds = "); Serial.println(readings);
      //if (audio) {Serial.print("  Average Audio Freq. = "); Serial.println(audCnt);}
      audCnt=0;
    }
    refreshWait(1000);  // allow time for SIO to complete
    readings=0;
  }
}


// ===================================================================================
//        Display Output Processing

// ----------------------
void clearDisp() {
    allColor=0;
    dimIt=0;
    litLED=0;
    spriteOne = 0;
    spriteTwo = 0;
    darkOne = 0;
    flashLed=0;
    blinkLed=0;
    for (int i=0; i<=nleds; i++) {
      dim[i] =false;
      lit[i]=false;
      brt[i]=false;
      levelOn[i]=0;
    }
}

// -----------------------
void showLevel(byte level) {
  int n;
  clearDisp();
  for (int i=0; i<level; i++) {
    n = (i%nleds)+1;
    dim[n]=true;
    spriteOne=n;
    refreshWait(100);
  }
  refreshWait(100);
}

// -----------------------
void refreshDisp() {
  if (initDone) {
    oneMilliUpdate();   // NOT updated via a call in a One MilliSecond interrupt service routine
  } 
}

// -----------------------
void refreshWait(int msec) {
  while (msec>0) {
    refreshDisp();   // NOT updated via a call in a One MilliSecond interrupt service routine
    while (lastMsec==millis()) {;}  // if a millisecond has been since last UI update wait 
    msec--;
  }
}

void delay_dispUpd(int msec) {
  do {
    refreshDisp();
    //delay(1);
    while (lastMsec==millis()) {;}  // if a millisecond has been since last UI update wait 
    msec--;
  } while (msec>0);
}

//  ---------------------  Display Update  - Needs to be called ~ every One Milli-Second
void oneMilliUpdate() {
  // takes 83-96 usecs  (striped down: 50-60 usecs)
  bool on[13];
  int ledSet, ledn, Lite;
  unsigned int ledTime, colorBits;
  int i, n, set;
  bool flashTime, blinkTime;

  lastMsec=millis();
  /************  was otherwise done in ISR   ******/
  // ------------ do myMilles Second bookkeeping
  myMillis++;
  if (myMillis%1000 == 0) { // every 1 second
    if (heartBeat) spriteOne=(spriteOne==0)? heartBeat : 0;
    mySecs++;
    secsIdle++;
  }

  msCnt++;
  
  for (i=1; i<=nleds; i++) on[i]=0; // init working array

  // ------------- as called for, Light all Leds prepesented by bits
  if (allColor != 0) {
      colorBits = colorSet[allColor];
      for (i=0; i<nleds; i++) {
        on[i+1] = (colorBits>>i) & 1;
      }
  } else {

    // -------------- determine which Leds its time to illuminate
    ledTime = msCnt;
    if (ledTime%9 == 0) {                 // 1/9 cadence time to show 'dim'
      dimOn = false;
      for (i=1; i<=nleds; i++) {on[i]=dim[i];  dimOn|=dim[i];}
      on[dimIt]= !on[dimIt];  // if off make it dim, if dim make it off
    } else if (ledTime%3 == 1) {          // 1/3 of the time process 'lit'
      for (i=1; i<=nleds; i++) on[i]=lit[i];
      on[litLED] = true;
    }
    // support for 10% PWM type gradations of lighting LEDs
    for (i=1; i<=nleds; i++) {  // provide for lighting of 0-10 levels 
      if (levelOn[i]>(ledTime%10)) on[i]=true;
    }
    
    for (i=1; i<=nleds; i++) if (brt[i]) on[i]=true;
    on[spriteOne] = true;
    on[spriteTwo] = true;
    on[darkOne] = false;
    flashTime = ((msCnt%500) < 200);  // 150ms out of 500ms turn it on
    if (flashTime && flashLed) on[flashLed] = !on[flashLed];

    blinkTime = (mySecs%2);  // every other second flash the led, otherwise leave it be
    if (blinkTime && blinkLed) on[blinkLed] = flashTime;
  }

  // -------------- drive the resulting Leds of interest
  for (i=0; i<nleds; i++) {
    ledn = i+1;
    Lite = lites[i];
    digitalWrite(Lite, (on[ledn]==ON_STATE));
  }
}



// ===================================================================================
//        Button Input Processing

void refreshBtns() {
  //delay(1);
  UI_update();
}

// ?????????????????????????????????????????????????????????????????/
void escDelay(int msec) {
  bool b4;
  while (msec-- >0) {
    delay(1);
    //b4 = ! digitalRead(button[3]);
    pinMode(button[3], INPUT);b4 = PRESSED_STATE == digitalRead(button[3]);pinMode(button[3], OUTPUT);
    if (b4) escCnt++;
    else escCnt=0;
    ESC = (escCnt>1500);  // takes <2secs given Btns check ~ every msec
    if (ESC) return;
  }
}

void noBtnDelay(int msec) {
  while (!Btn && msec-->0) {
    //delay(1);
    UI_update();
  }
}

void wait_BtnRelease() {
  escCnt=0;
  while (btnState!=0 || btnChanged) {  // cares about keys except FuncKey
    //delay(1);
    UI_update();
    if (ESC) return;
  }
  escCnt=0;
  ESC=time2Escape=false;
}

// -----------------------
void scanBtnWait(int msec) {
  while (msec>0) {
    UI_update();
    //delay(1);
    msec--;
  }
}

bool btnPressed() {
  UI_update();
  return(btnChanged && Btn);
}

void UI_update() { // -----------------  UI (button Input & display output) processing
  while (lastMsec==millis()) {;}  // if a millisecond has been since last UI update wait 
  oneMilliUpdate();  //updates lastMsec
//  oneMilli_DispUpdate();
  
  //digitalWrite(ONBOARD_LED,mySecs&1);  // DEBUG flash
  if (secsIdle>5*60) { // every 5 mins.
    secsIdle=0;
    checkVoltage();       // Check battery health
    beep(100); beep(100); // say HEY! Don't forget me
  }

  if (btn4) escCnt++;
  else if (!btnChanged && !Btn) escCnt=0; 
  ESC = (escCnt>1500);  // takes <2secs given Btns check ~ every msec
//  if (1==escCnt%100)  {Serial.print("escCnt:  ");  Serial.println(escCnt);}
  time2Escape = ESC;

// must have ESC handling code, before:  
//if (myMillis%5!=0) return; // only need to re-scan buttons every 5 msecs

  scanBtns();
}

void scanBtns() { // -----------------  UI (button Input & display output) processing
  // during no button activity, & without multiplexing       ~ 25 usecs to execute
  // during no button activity, & with multiplexed Btns/Leds ~ 130 usecs to execute
  bool b1,b2,b3,b4;
  bool s0,s1,s2,s3;
  byte i,k;
  byte currState;
int l2;

  btnChanged = btn1Changed = btn2Changed = btn3Changed = btn4Changed = false;

  currState=-1;
  debounceCnt = 0;
  while (currState!=btnState && (++debounceCnt<=DEBOUNCE_CNT)) {
    if (debounceCnt>1)   delayMicroseconds(500);
    #ifdef DIO_SHARING
    // -------------------------- support for buttons parrelleing LEDs
//      TIMSK2 = 0x00;        //Disable Timer2 INT
      //cli();
      //noInterrupts();
/******  use this block for minimum flickering of low level leds   ***********
      if (BTN_ENB!=0) digitalWrite(BTN_ENB, PRESSED_STATE);
      pinMode(button[0], INPUT);b1 = PRESSED_STATE == digitalRead(button[0]); pinMode(button[0], OUTPUT); // read the button inputs
      pinMode(button[1], INPUT); b2 = PRESSED_STATE == digitalRead(button[1]);pinMode(button[1], OUTPUT);
      pinMode(button[2], INPUT);b3 = PRESSED_STATE == digitalRead(button[2]);pinMode(button[2], OUTPUT);
      pinMode(button[3], INPUT);b4 = PRESSED_STATE == digitalRead(button[3]);pinMode(button[3], OUTPUT);
      if (BTN_ENB!=0) digitalWrite(BTN_ENB, 0==PRESSED_STATE);  // ----- Undo setup for button reading
 
/****** use this block if detection of multi button presses are needed for your UI ***********/      
// note the frequency of calls to scanBtns() effects degree of low level flicker if any 
      s0 = digitalRead(button[0]); s1 = digitalRead(button[1]); s2 = digitalRead(button[2]); s3 = digitalRead(button[3]);
      pinMode(button[0], INPUT); pinMode(button[1], INPUT); pinMode(button[2], INPUT); pinMode(button[3], INPUT); 
      if (BTN_ENB!=0) digitalWrite(BTN_ENB, PRESSED_STATE);
      b1 = PRESSED_STATE == digitalRead(button[0]);  // read the button inputs
      b2 = PRESSED_STATE == digitalRead(button[1]);
      b3 = PRESSED_STATE == digitalRead(button[2]);
      b4 = PRESSED_STATE == digitalRead(button[3]);
      if (BTN_ENB!=0) digitalWrite(BTN_ENB, 0==PRESSED_STATE);  // ----- Undo setup for button reading
      pinMode(button[0], OUTPUT); pinMode(button[1], OUTPUT); pinMode(button[2], OUTPUT); pinMode(button[3], OUTPUT);
      digitalWrite(button[0], s0); digitalWrite(button[1], s1); digitalWrite(button[2], s2); digitalWrite(button[3], s3); 

/******/
//      TIMSK2 = 0x01;        //Enable Timer2 INT
#else
      // -------------------------- handle straight forward buttons
      b1 = PRESSED_STATE == digitalRead(button[0]);  // read the button inputs
      b2 = PRESSED_STATE == digitalRead(button[1]);
      b3 = PRESSED_STATE == digitalRead(button[2]);
      b4 = PRESSED_STATE == digitalRead(button[3]);
#endif

//    currState = (((b4<1 + b3)<1) + b2)<1 + b1;  //  does not work with type 'bool'
    currState = (b1)? 1:0;
    if (b2) currState += 2;   // encode btn #s, so any combination is unique
    if (b3) currState += 4;
    if (b4) currState += 8;
  }
  
  if (currState != btnState) {  // the 'button state' has changed
      secsIdle=0;
      debounceCnt = 0;
      if (debugPrt)  {Serial.print("CurrState Btn State:  ");  Serial.println(currState);}
      priorBtn = btnState;
      btnState = currState;
      btnChanged=true;
  
      btnNum = (b1)? 1 : ((b2)? 2 : ((b3)? 3 : ((b4)? 4 : 0)));  // logical button # (1-4)  // assumes only 1 btn pressed
      Btn = b1 || b2 || b3 || b4;
      if(b1 != btn1) {btn1 = b1; btn1Changed = true;}
      if(b2 != btn2) {btn2 = b2; btn2Changed = true;}
      if(b3 != btn3) {btn3 = b3; btn3Changed = true;}
      if(b4 != btn4) {btn4 = b4; btn4Changed = true;}
//    }
  }
}

// ===================================================================================
//        Audio Output Processing
// there is no display updates while outputing 'beeps'

void beep(int delayms){  // works for both Buzzers & Speakers
//  pinMode(BEEPPIN, OUTPUT);
  for (int i=0; i<delayms; i++) {         // wait for a delayms ms
    digitalWrite ( BEEPPIN, HIGH); 
    delayMicroseconds(500);    
    digitalWrite ( BEEPPIN, LOW);
    delayMicroseconds(500);
  }
//  delay(delayms);          // wait for a delayms ms   
}  

void xBeep(unsigned short int delayms){  // higher freq & longer dury cycle
  for (int i=0; i<delayms; i++) {         // wait for a delayms ms
    digitalWrite ( BEEPPIN, HIGH); 
    delayMicroseconds(666);    
    digitalWrite ( BEEPPIN, LOW);
    delayMicroseconds(333);
  }
  digitalWrite ( BEEPPIN, LOW);;
//  delay(delayms/2);          // wait for a delayms ms   
}
void Beep(unsigned short int delayms){  // higher freq & longer dury cycle
  delayms = 2 * delayms / 3;
  for (int i=0; i<delayms; i++) {         // wait for a delayms ms
    digitalWrite ( BEEPPIN, HIGH); 
    delayMicroseconds(666);    
    digitalWrite ( BEEPPIN, LOW);
    delayMicroseconds(333);
    digitalWrite ( BEEPPIN, HIGH); 
    delayMicroseconds(333);    
    digitalWrite ( BEEPPIN, LOW);
    delayMicroseconds(166);
  }
  digitalWrite ( BEEPPIN, LOW);;
//  delay(delayms/2);          // wait for a delayms ms   
}


void buZZ(int msec){  
  if (ACTIVE_BUZZER) {    // sound the Buzzer, works with true 'active' Buzzers only
    digitalWrite ( BEEPPIN, HIGH);
    delay(msec);
    digitalWrite ( BEEPPIN, LOW);
  } else {      // fake buzzer, uses more time to replace the punch of a real BuZZer
    //genTone(1200,msec/2);  genTone(1250,msec);
    //genTone(1200,msec/2);   genTone(1150,msec/2);  genTone(1250,msec/2);
    speakerBuzz(2*msec);  // more advanced buZZer simulation
  }
}

void speakerBuzz(int msec){  // sound the Buzzer, works with Speaker (as well as buzzer)
                            // produces a sawtooth, 2500..--2000
  unsigned long startMs;
  int p, pmin, pmax;
  startMs=millis();
  pmin=200;
  pmax=250;
  p=pmin;
  while (millis()-startMs < msec) {
    digitalWrite ( BEEPPIN, HIGH); 
    delayMicroseconds(p);    
    digitalWrite ( BEEPPIN, LOW);
    delayMicroseconds(p);
    p+=5;
    if (p>pmax) p=pmin;
  }
  digitalWrite ( BEEPPIN, LOW);
}


void beepNdisp(int delayms){  // works for both Buzzers & Speakers
//  pinMode(BEEPPIN, OUTPUT);
  for (int i=0; i<delayms; i++) {         // wait for a delayms ms
    digitalWrite ( BEEPPIN, HIGH); 
    delayMicroseconds(500);    
    digitalWrite ( BEEPPIN, LOW);
    delayMicroseconds(500);
    refreshDisp();
  }
}

void taDah() {
  allColor=0;
  allColor=5; beepNdisp(50);                    // Tah
  allColor=0; refreshDisp(); delay(150);  // ...
  allColor=5; beepNdisp(200);                   // Dah
  if (level>0) {  // for games using 'level' show it
    showLevel(level);
    while (!btnPressed()) {refreshDisp();}
  }
  allColor=0; refreshDisp();
}

// ---------------------  tone() type Audio Output
//                        true tone audio with a Speaker;  fine with Passive buzzer
//                        only almost, sort of, ok with Active Buzzer, due to Moiré interference patterns

//   Generate Tone frequencies with out using "tone()" (saving 1500 program bytes & 21 ram bytes) 
//     sounding almost as good
void genTone(int toneVal, int delayms){       // Generate a specified TONE
  byte pinState = 0;       // 0 turns it off
  unsigned long t0, phase;

  if (toneVal==0) {refreshWait(delayms); return;}
  pinMode(SPEAKERPIN, OUTPUT);

  phase = (1000000L/toneVal)/2;
  int msCnt = ( (long) delayms * toneVal) / 1000;
  for (int i=0; i<(2*msCnt); i++) {         // wait for a delayms ms
    if (volHigh) digitalWrite ( SPEAKERNEG, pinState); // set NEG lead to opposite of + lead
    pinState = 1 - pinState;
    t0=micros();
    digitalWrite ( SPEAKERPIN, pinState); 
    //delayMicroseconds((1000000L/toneVal)/2);    // replaced by next line
    while ((micros()-t0)<phase) {if (lastMsec!=millis()) refreshDisp();}  // needed when not updating display in the background 
  }
  digitalWrite ( SPEAKERPIN, LOW);
  digitalWrite ( SPEAKERNEG, LOW);
}

const int basicKeyNotes[] =
   {0,131,147,165,175,196,220,247,262,294, 330,349,392,440,494,523,587,659,698,784,880,988,1047};
  //    1   2   3   4   5   6   7   8   9   10  11  12  13  14  15  16  17  18  19  20  21  22
  //                                do  re  mi  fa  so  la  ti  do
  //    C3  D3  E3  F3  G3  A3  B3  C4  D4  E4  F4  G4  A4  B4  C5  D5  E5  F5  G5  A5  B5  C6

void playNote(int note, int delayms){
   genTone(basicKeyNotes[note], delayms);
}

void playNote(int note){
   playNote(note, 500);
}

