package com.example.germt.autojoystick;

import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.widget.Toolbar;
import android.view.MotionEvent;
import android.view.View;
import android.view.Menu;
import android.view.MenuItem;
import android.widget.ImageView;
import android.widget.TextView;
import android.widget.Toast;

public class MainActivity extends AppCompatActivity {

    //the name of the configuration file and settings to save the settings
    public static final String APPSETTINGS = "ajsettings";
    public static final String SRVRIP = "server_ip";
    public static final String SRVPRT = "server_port";
    public static final String AUTOCONNECT = "autoconnect";
    public static final String MYWIFISSID = "wifi_ssid";

    public static final String HNDLKEY = "SRV_MSG";//Key, which identifies the message from toy car
    public static final String SKT_ERR = "skt_error";//error creating a socket
    public static final String SKT_RD_ERR = "skt_read_error";//error reading data from the socket
    public static final String NOCONNTOMYWIFI = "no_conn_mywifi";//no connection to the required WiFi network
    public static final String READY = "ready";//toy car is ready to take command

    private TextView mTxtView;//for debugging

    private TCPClient mTcpClient;//object that is responsible for network exchange with arduino on WiFi
    private SharedPreferences mPrefs;//object that is responsible for saving settings
    private String mwfSSID;
    private String mSrvIp;
    private int mSrvPrt;
    private boolean mSrvAutoconn;

    private ImageView mImageView;//joystick image
    private String mDrive;//where we go: A - forward, B - back
    private int mDvol;//how fast we go 0 ... 9
    private String mPov;//where we turn: C - to the right, D - to the left
    private int mPvol;//how quickly we turn 0 ... 9
    private int mCntrX, mCntrY;//center point of the joystick image
    private int mRmaX, mRmaY;//maximum size of the joystick image

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        Toolbar toolbar = (Toolbar) findViewById(R.id.toolbar);
        setSupportActionBar(toolbar);

        mTxtView = (TextView) findViewById(R.id.textView1);

        mImageView = (ImageView) findViewById(R.id.imageView1);//joystick image
        mPrefs = getSharedPreferences(APPSETTINGS, Context.MODE_PRIVATE);

        //read settings
        mwfSSID = mPrefs.getString(MYWIFISSID, "ESP8266");
        mSrvIp = mPrefs.getString(SRVRIP, "192.168.4.1");
        mSrvPrt = mPrefs.getInt(SRVPRT, 333);
        mSrvAutoconn = mPrefs.getBoolean(AUTOCONNECT, false);

        //initialize the movement, like A0C0 - that is, stand still
        mDrive = "A";
        mDvol = 0;
        mPov = "C";
        mPvol = 0;

        View.OnTouchListener handleTouch = new View.OnTouchListener() {
            @Override
            public boolean onTouch(View view, MotionEvent motionEvent) {
                double alfa=0;
                //initialize data for the calculation of the movement command
                mRmaX = mImageView.getWidth() / 2;
                mRmaY = mImageView.getHeight() / 2;
                if (Build.VERSION.SDK_INT>=Build.VERSION_CODES.KITKAT) {
                    //calculating the center for android 4.4 and above
                    mCntrX = (int) mImageView.getPivotX();
                    mCntrY = (int) mImageView.getPivotY();
                } else {
                    //calculating the center for android 4.3 and below
                    mCntrX = (int) mImageView.getPivotX() + mRmaX;
                    mCntrY = (int) mImageView.getPivotY() + mRmaY;
                }
                int x = (int) motionEvent.getX();
                int y = (int) motionEvent.getY();
                switch (motionEvent.getAction()) {
                    case MotionEvent.ACTION_MOVE:
                        //choose the direction of travel
                        if ((y - mCntrY) > 0) {
                            mDrive = "B";
                        } else {
                            mDrive = "A";
                        }
                        if ((x - mCntrX) < 0) {
                            mPov = "D";
                        } else {
                            mPov = "C";
                        }
                        //calculate the requested speed and the rotation value in the projection on the circle
                        if(((x-mCntrX)==0) || ((y-mCntrY)==0)){
                            if((y-mCntrY)==0){
                                mDvol = 0;
                                if((x-mCntrX)==0){
                                    mPvol = 0;
                                }else {
                                    mPvol = (int) Math.floor(10*Math.abs(x-mCntrX)/mRmaX);
                                }
                            }else {
                                mPvol = 0; // (x-mCntrX)=0
                                mDvol = (int) Math.floor(10*Math.abs(y-mCntrY)/mRmaY);
                            }
                        }else {
                            alfa = Math.toDegrees(Math.atan(Math.abs((float) (y-mCntrY)/ (float) (x-mCntrX))));
                            if ((alfa>=0) && (alfa<=45)){
                                mDvol = (int) Math.floor(10f*Math.abs((float)(y-mCntrY))/((float)mRmaY*Math.cos(Math.toRadians(alfa))));
                                mPvol = (int) Math.floor(10f*Math.abs((float)(x-mCntrX))/((float)mRmaX*Math.cos(Math.toRadians(alfa))));
                            }else {
                                mDvol = (int) Math.floor(10f*Math.abs((float)(y-mCntrY))/((float)mRmaY*Math.sin(Math.toRadians(alfa))));
                                mPvol = (int) Math.floor(10f*Math.abs((float)(x-mCntrX))/((float)mRmaX*Math.sin(Math.toRadians(alfa))));
                            }
                        }
                        //calculate the requested speed
                        //mDvol = (int) Math.round(10 * Math.abs(y - mCntrY) / mRmaY);//-this is square
                        if (mDvol > 9) {
                            mDvol = 9;
                        }
                        //calculate the requested rotation value
                        //mPvol = (int) Math.round(10 * Math.abs(x - mCntrX) / mRmaX);//-this is square
                        if (mPvol > 9) {
                            mPvol = 9;
                        }
                        break;
                    case MotionEvent.ACTION_UP:
                        mDrive = "A";
                        mDvol = 0;
                        mPov = "C";
                        mPvol = 0;
                        break;
                }
                // for debugging
                //String msg = mDrive + mDvol + mPov + mPvol;
                //mTxtView.setText(msg);
                return true;
            }
        };
        mImageView.setOnTouchListener(handleTouch);
    }

    @Override
    protected void onDestroy(){
        //break the connection if it is created
        if (mTcpClient != null) {
            mTcpClient.stopClient();// stop the exchange and break the connection
            mTcpClient = null;
            Toast.makeText(getApplicationContext(), R.string.toast_conn_broken, Toast.LENGTH_SHORT).show();
        }

        super.onDestroy();
    }

    @Override
    protected void onResume() {
        super.onResume();

        //If the settings have changed, you must create a new connection
        String wf = mwfSSID;//save old settings
        String si = mSrvIp;//save old settings
        int sp = mSrvPrt;//save old settings

        //read settings
        mwfSSID = mPrefs.getString(MYWIFISSID, "ESP8266");
        mSrvIp = mPrefs.getString(SRVRIP, "192.168.4.1");
        mSrvPrt = mPrefs.getInt(SRVPRT, 333);
        mSrvAutoconn = mPrefs.getBoolean(AUTOCONNECT, false);


        if(mSrvAutoconn && mTcpClient==null){//autoconnect ty car
            wf = mwfSSID;//to avoid unnecessary reconnections
            si = mSrvIp;//to avoid unnecessary reconnections
            sp = mSrvPrt;//to avoid unnecessary reconnections
            Runnable runnable = new Runnable() {
                        @Override
                        public void run() {
                            mTcpClient = new TCPClient(mSrvIp, mSrvPrt, mwfSSID);//create an instance of the class
                            mTcpClient.runClient(mHndlr, getApplicationContext());//start the exchange process
                        }
            };
            Thread thread = new Thread(runnable); //create a new thread
            thread.start();//start the connection and data exchange
        }
        if(si.compareTo(mSrvIp)!=0 || sp!=mSrvPrt || wf!=mwfSSID){//if the settings have changed, then break the connection
            if (mTcpClient!=null){
                mTcpClient.stopClient();// stop the exchange and break the connection
                mTcpClient = null;
            }
        }
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        // Inflate the menu; this adds items to the action bar if it is present.
        getMenuInflater().inflate(R.menu.menu_main, menu);
        return true;
    }

    @Override
    public boolean onPrepareOptionsMenu(Menu menu) {
        if (mTcpClient != null) {
            if (mTcpClient.testConnection()){
                menu.getItem(1).setEnabled(true);
                menu.getItem(0).setEnabled(false);
            }else {
                menu.getItem(1).setEnabled(false);
                menu.getItem(0).setEnabled(true);
            }
        } else {
            menu.getItem(1).setEnabled(false);
            menu.getItem(0).setEnabled(true);
        }
        return super.onPrepareOptionsMenu(menu);
    }
    Handler mHndlr = new Handler() {
        @Override
        public void handleMessage(Message msg) {
            Bundle bundle = msg.getData();
            String zprs = bundle.getString(HNDLKEY).trim();
            if (zprs.substring(0, 5).equals(READY)) {
                if (mTcpClient != null) {
                    //отвечаем на запрос
                    Runnable runnable = new Runnable() {
                        @Override
                        public void run() {
                            mTcpClient.SendMessage(mDrive + mDvol + mPov + mPvol); //answer the request
                        }
                    };
                    Thread thread = new Thread(runnable); //create a new thread
                    thread.start();//start the connection and data exchange
                } else {
                    Toast.makeText(getApplicationContext(), R.string.toast_err_no_connection, Toast.LENGTH_LONG).show();
                }
                if (zprs.length()>=14) {//message format from toy car: readyXXXYYYZ.Z
                    String tmp = String.format(getResources().getString(R.string.textView_text),
                            zprs.substring(5, 8), zprs.substring(8, 11), zprs.substring(11, 14));
                    mTxtView.setText(tmp);
                }else{
                    mTxtView.setText(zprs);//if the message is too short, then we'll see what came
                }
            } else if (zprs.equals(NOCONNTOMYWIFI)){//there is no connection to appropriate WiFi network
                Toast.makeText(getApplicationContext(), R.string.toast_err_no_conn_mywifi, Toast.LENGTH_LONG).show();
                if (mTcpClient != null){
                    mTcpClient.stopClient();// stop the exchange and break the connection
                }
                mTcpClient = null;
            } else if (zprs.equals(SKT_ERR)) {//error creating socket
                Toast.makeText(getApplicationContext(), R.string.toast_err_skt_err, Toast.LENGTH_LONG).show();
                if (mTcpClient != null){
                    mTcpClient.stopClient();// stop the exchange and break the connection
                }
                mTcpClient = null;
            }else if (zprs.equals(SKT_RD_ERR)) {//error read socket
                Toast.makeText(getApplicationContext(), R.string.toast_err_skt_read_err, Toast.LENGTH_LONG).show();
                if (mTcpClient != null){
                    mTcpClient.stopClient();// stop the exchange and break the connection
                }
                mTcpClient = null;
            }
        }
    };

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        // Handle action bar item clicks here. The action bar will
        // automatically handle clicks on the Home/Up button, so long
        // as you specify a parent activity in AndroidManifest.xml.
        switch (item.getItemId()) {
            case R.id.action_connect:
                Runnable runnable = new Runnable() {
                    @Override
                    public void run() {
                        mTcpClient = new TCPClient(mSrvIp, mSrvPrt, mwfSSID);//create an instance of a class
                        mTcpClient.runClient(mHndlr, getApplicationContext());//start the exchange process
                    }
                };
                Thread thread = new Thread(runnable); //сreate new thread
                thread.start();//start the connection and data exchange
                return true;
            case R.id.action_disconnect:
                if (mTcpClient != null){
                    mTcpClient.stopClient();// stop the exchange and break the connection
                    mTcpClient = null;
                }
                return true;
            case R.id.action_settings: //call the settings window
                Intent intent = new Intent(this, ajSettingsActivity.class);
                startActivity(intent);
                return true;
            default:
                return super.onOptionsItemSelected(item);
        }
    }
}