/*
  Arduino LED Exploration
  8-15-2011  Modified 11-30-20012
  Fixed error: only could do 5 volt Leds
  Added Aref proccessor analogue reference
  Written by Steven R. Cypherd
  
  You must open a Serial Monitor after you Upload this program.
  This is Led Exploration using an Arduino Uno. I read an LDR to 
  to show how LEDs work.
  
  The Capacitors are needed to get stable readings.
  Pulse Width Modulation is noisy.
  RS is based on the formula RS = (Vin - Vled) / Iled where Vin is SYS_VCC from
  the RadioShack Engineer's Mini-Notebook "Optoelectronics Circuits"
  by Forrest M. Mims III(1986).
  The Circuit:
    D3 PWM 10k to Gnd, 100mf cap 16v to Gnd, 104nf cap to Gnd to
    220r to Test LED+ to A1 to 4.7mf cap 35v to Gnd.
    Test LED- to Gnd.
    A0 to LDR 1 to 10k Gnd to 104nf cap to
    LDR 2 to Vcc.
    D4 Button 1-2 to 10k Vcc, 3-4 to Gnd
    D5 to R1k to Run Led +, Run Led - to Gnd
  See my drawing for the circuit.
  I made the probe from the mini LDR that came in the starter pack. Use black
  heat shrink tubing to block light and shield the sides and back of the LDR.
*/

//User Settings
#define Aref_SYS 5              //Proccessor analogue reference
#define ADC_STEPS 1024      //Arduino Uno ADC Steps
#define LED_SYS 5           //VCC of the Led and where it will run
#define R01_PMW 220         //Resistor between input PWM and the test LED
#define LED_MAX_VOLTS 5     //Max LED volts
#define LED_START_VOLTS 1   //LED starting volts
#define TEST_STEPS 5        //In ADC volts
#define PRINT_DETAILS 1     //All Details of the tests
#define PRINT_INSTRUCTIONS 0
#define INPUT_DETAILS 0     //Input, Output port values
#define INPUT_VALUES 0      //Displays the Actual Reading from the Ports
#define LED_DETAILS 1       //Put your details in the print statments

//Digital Pins
#define TEST_PWM 3
#define BTN_MODE 4
#define RUN_LED 5
//Analog Pins
#define LDR_READ A0      //LDR Light Dependent Resistor
#define LED_READ A1      //Test LED

float VoltsLed;         //Floats are needed for easy calculations
float LedVcc;           //and display
float VoltsFirstLed;    //First Light
float VoltsLastLed;
float AmpsFirstLed;
float AmpsLastLed;
float FirstRS;
float AmpsLed;
float VoltsIn;
float RSohms;
int adcOneVolt;     //One volt in Adc steps
int ledStartVolts;  //Led starting volts
int ledMaxVolts;    //Max for the Led
int ledRead;        //Analogue read from Test Led
int ldrRead;        //Analogue read of LDR
int ldrLastRead;    //For calculation of light gap
int lightGap;       //Difference of ldrRead between steps
int lightGapLow;    //When the lightGap drops below 10
int inVolts;
int cnt2;
int cnt3;
int tmp1;
int btnState;
boolean testRunning;
boolean testDone;
boolean testStopped;

void setup() {
  adcOneVolt = ADC_STEPS / Aref_SYS;//create ADC one Volt in an Int
  ledStartVolts = adcOneVolt * LED_START_VOLTS;
  ledMaxVolts = adcOneVolt * LED_MAX_VOLTS;
  LedVcc = (float)LED_SYS;
  pinMode(RUN_LED, OUTPUT);
  pinMode(TEST_PWM, OUTPUT);
  pinMode(BTN_MODE, INPUT);
  Serial.begin(9600);
}// End SetUp

//Main loop clears each test for the next
void loop() {
  digitalWrite(TEST_PWM, LOW); //Shut off PWM
  digitalWrite(RUN_LED, LOW);// Turn off test Led
  delay(20);
  AmpsFirstLed = 0;
  ldrLastRead = 0;
  lightGapLow = 0;
  cnt2 = 0;
  testRunning = false;
  testDone = false;
  testStopped = false;

  if( PRINT_INSTRUCTIONS == 1 ){
    Serial.println("Set up the circuit as shown");
    Serial.println("Place Led to test into the circuit");
    Serial.println("Place black heat shrink tubing over the Led");
    Serial.println("Place your Probe into the tubing and touching ");
    Serial.println("the Led.");
    Serial.println("");
    }

  if( INPUT_DETAILS == 1 ){
    Serial.print("PWM port ");
    Serial.print( TEST_PWM, DEC );
    Serial.print(", LDR port ");
    Serial.print( LDR_READ, DEC );
    Serial.print(", Test LED port ");
    Serial.println( RUN_LED, DEC );
    Serial.print("Led Starting Volts ");
    Serial.print( (float)ledStartVolts / adcOneVolt );
    Serial.print(", Max ");
    Serial.print( (float)ledMaxVolts / adcOneVolt );
    Serial.print(", VCC ");
    Serial.print( LedVcc );
    Serial.print(" R1 "); 
    Serial.println( R01_PMW, DEC );
    Serial.println("");
    }

  Serial.println("Please place the LED in the test possition");
  Serial.println("Press the Button to Start or Stop the Test");  
  
//Wait for user to set up the test
  do {
    tmp1 = digitalRead( BTN_MODE ); //Pauses until Button is pushed
    } while( tmp1 == HIGH );
  btnState = tmp1; //0 is button Down.
  
  Serial.println("Running Test ");
  
  //Report data goes here
  if( LED_DETAILS == 1 ){
    Serial.println("More Data");
    Serial.println("Test Green Led Date 10/29/12");
    }
    
  digitalWrite(RUN_LED, HIGH);
  tmp1 = 0;
  cnt3 = 0;
//Testing Loop---------------------------------------------------
  for( inVolts = ledStartVolts; inVolts <= ledMaxVolts; inVolts += TEST_STEPS ){
    analogWrite(TEST_PWM, inVolts / 4 ); //PWM voltage to Test LED
    delay(20);
    ldrRead = analogRead( LDR_READ );//Read Light output of Led with LDR
    delay(20);
    ledRead = analogRead( LED_READ );//Read led volts
    delay(5);

//Wait for light from the Led before printing details      
    if( ldrRead < 30 ){  //Lowest light reading needed
      Serial.print("-");
      cnt2 += 1;
      if( cnt2 > 35 ){  //Page width
        tmp1 += 1;  //Stop after 3 pages no light
        if( tmp1 > 2 ) break;
        Serial.println("-");
        cnt2 = 0;
        }
      continue;
      }
    
    if( testRunning == false ) {
      Serial.println(".");
      testRunning = true;
      }
      //Trying to clear capacitors
      digitalWrite(TEST_PWM, LOW); //Shut off PWM
//Main calculations
    cnt3 += 1;
    VoltsLed = (float)ledRead / adcOneVolt;
    VoltsIn = (float)inVolts / adcOneVolt;
    AmpsLed = (VoltsIn - VoltsLed) / R01_PMW;
    RSohms = (LedVcc - VoltsLed) / AmpsLed;
    AmpsLed = AmpsLed * 1000;//Adjust for mA display
    if( AmpsFirstLed == 0 ) {
      AmpsFirstLed = AmpsLed;
      VoltsFirstLed = VoltsLed;
      FirstRS = RSohms;
      }
    
    if( ldrLastRead < ldrRead ){ //Setup Last reading for LightGap
      lightGap = ldrRead - ldrLastRead; //Positive LightGap only
      }
    else {
      lightGap = ldrLastRead - ldrRead;
      }
    
    //Check test button
    tmp1 = digitalRead( BTN_MODE );
    if( tmp1 == false && btnState == true ){
      testDone = true;
      testStopped = true;
      }
    else {
      btnState = tmp1;
      }
          
    if( lightGap <= 10 ){  //Lowest of Light Gap
     lightGapLow += 1; //Three or more is a big drop in light
      if( lightGapLow >= 3 ) testDone = true; 
      }  //Number of lowest light gaps to end the test
    else {
      if( lightGapLow != 0 ) lightGapLow = 0;
      }
              
    if( testDone == true ){
      if( testStopped == true ) Serial.println("*** Test Stopped ****");
      Serial.println("Test Results ");
      Serial.print("For VCC of ");
      Serial.print(LedVcc);
      Serial.println(" volts");
      Serial.print("First light volts is ");
      Serial.print( VoltsFirstLed );
      Serial.print(", ");
      Serial.print( AmpsFirstLed );
      Serial.print("mA, ");
      Serial.print("RS ");
      Serial.println( FirstRS, 0 );
      Serial.print("Normal volts is ");
      Serial.print( VoltsLastLed );
      Serial.print(", ");
      Serial.print( AmpsLastLed );
      Serial.print("mA, ");
      Serial.print("RS ");
      Serial.println( RSohms, 0 );
      Serial.println("");
      break; //End the test here
      }
      
    if( PRINT_DETAILS == 1 ){
      if(INPUT_VALUES == 1){
        Serial.print("Input Values In ");
        Serial.print(inVolts);
        Serial.print(", Led ");
        Serial.print(ledRead);
        Serial.print(", LDR ");
        Serial.print(ldrRead);
        Serial.println("");
        }
      else {
        Serial.print(cnt3, DEC);
        Serial.print(" IN ");
        Serial.print(VoltsIn);
        Serial.print(", Led V ");
        Serial.print(VoltsLed);
        Serial.print(", ");
        Serial.print(AmpsLed);
        Serial.print("mA, Rs ");
        Serial.print(RSohms, 0);
        Serial.print(", LGap ");
        Serial.print(lightGap);
        Serial.print(" LDR ");
        Serial.println( ldrRead );
        }
      }
    
    if(lightGapLow == 0 ) { //Catches last readings
      VoltsLastLed = VoltsLed;  //Test ends      
      AmpsLastLed = AmpsLed;    //No more reads
      }
    ldrLastRead = ldrRead;
  }//For...  The testing Loop ^^^^^^^^^^^
  
  if ( testDone == true ) {
    Serial.println( "Test is Done" );
    Serial.println("----------------------------------");
    Serial.println("");
    Serial.println("");
    }
  else {
    Serial.println("");
    Serial.println("*************************************");
    Serial.println( "********Test failed*****************" );
    Serial.println("Check Led and voltage settings ");
    Serial.println("");
    }
}//Loop Set-up the Test and wait. ^^^^^^^^
