/**************************
* Application Information *
**************************/

/*
Device: AtMega8
Clock: External Crystal @ 12MHz
hfuse: 0xC9
lfuse: 0xEF

LED row controller on port D
LED shift registers on port C
12MHz crystal on Xtal1/Xtal2
*/


/*****************
* Hardware Setup *
*****************/

/*		____________________Mega8___________________
		|Reset (PC6)		(ADC5/SCL) 	PC5|
mic2981-Pin1	|PD0 (RXD)		(ADC4/SDA) 	PC4|
mic2981-Pin2	|PD1 (TXD)			(ADC3)	PC3| hef4794-Pin15 (Enable Output)
mic2981-Pin3	|PD2 (Int0)			(ADC2)	PC2| hef4794-Pin3 (Clock)
mic2981-Pin4	|PD3 (Int2)			(ADC1)	PC1| hef4794-Pin2 (Data)
mic2981-Pin5	|PD4 (Xck/T0)			(ADC0)	PC0| hef4794-Pin1 (Latch)
+5V		|VCC					GND| Ground
Ground		|GND				       AREF|
Crystal		|PB6 (Xtal1/Tosc1)		       AVCC| +5V
Crystal		|PB7 (Xtal2/Tosc2)		(SCK)	PB5|
mic2981-Pin6	|PD5 (T1)			(MISO)	PB4|
mic2981-Pin7	|PD6 (AIN0)		(MOSI/OC2)	PB3|
mic2981-Pin8	|PD7 (AIN1)		(SS/OC1B)	PB2|
		|PB0 (ICP1)			(OC1A)	PB1|
		--------------------------------------------
*/



#define F_CPU 12000000UL

#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <avr/pgmspace.h>
#include "font5x8.h"
#include "3x5_numeric.h"

//The pins used to operate the shift register:
#define ShiftPort 	PORTC
#define ShiftDDR	DDRC
#define LatchPin	(1 << 0)
#define DataPin		(1 << 1)
#define ClkPin		(1 << 2)
#define OE		(1 << 3)

#define TotColumns	15	//Up to 16 without further code alteration

//The pins used to switch the row driver:
#define RowPort		PORTD
#define RowDDR		DDRD
#define RowPin0		(1 << 0)
#define RowPin1		(1 << 1)
#define RowPin2		(1 << 2)
#define RowPin3		(1 << 3)
#define RowPin4		(1 << 4)
#define RowPin5		(1 << 5)
#define RowPin6		(1 << 6)
#define RowPin7		(1 << 7)

#define ScrollSpeed	65	//How many milliseconds to pause before shifting columns left

typedef unsigned char u8;
typedef unsigned int u16;

u8 row_track = 0;

volatile u16 row_buffer[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };

static const char PROGMEM Messages1[] = { "8x16 LED Display Matrix" };
static const char PROGMEM Messages2[] = { "The quick brown fox jumps over the lazy dog." };

void Delay_ms(int cnt)	//Function delays a give number of milliseconds.  Depends on F_CPU being defined
{
  while (cnt-->0) _delay_ms(1);
}

static inline void InitPorts(void)	//Function used once to initialize the ports
{
  ShiftDDR |= (LatchPin | ClkPin | DataPin | OE);	//Setup shift register control pins
  RowDDR |= (RowPin0 | RowPin1 | RowPin2 | RowPin3 | RowPin4 | RowPin5 | RowPin6 | RowPin7); //Setup row driver pins
  
  ShiftPort |= OE;
  RowPort |= RowPin0;				//Drive first row and enable shift registers

  ShiftPort |= LatchPin; 				//Set latch pin high
  ShiftPort &= ~(ClkPin | DataPin); 			//Set ClkPin and DataPin low
}

static inline void InitTimers(void)	//Function used once to set up the timer
{
  TCCR1B |= 1<<WGM12 | 1<<CS11 | 1<<CS10;		//Start timer1 in CTC mode with prescaler of 64
  TIMSK |= 1<<OCIE1A;					//Enable compare match interrupt
  OCR1A = 0x00BB;					//Set compare value for 1 mSec
  sei();						//Enable global interrupts
}

void Shift_Int(u16 shiftData)		//Function used to shift in data (Alter this when adding more than 16 columns)
{
  ShiftPort &= ~(LatchPin | ClkPin | DataPin);  	//All pins low: LatchPin low signals a write operation
  
  for (char TrackColumns=0; TrackColumns<TotColumns; TrackColumns++)				//Cycles through each column
  {
    ShiftPort &= ~ClkPin;				//Set ClkPin low

    if (shiftData & (1<<TrackColumns)) ShiftPort |= DataPin;		//Set DataPin high if current bit is 1
    else ShiftPort &= ~DataPin;				//Set DataPin low if current bit is 0

    ShiftPort |= ClkPin;				//Set ClkPin high to increment shift register
    ShiftPort &= ~DataPin;				//Set DataPin low to prepare for next write
  }

  ShiftPort |= LatchPin;  				//Set LatchPin high to signal end of write operation
  ShiftPort &= ~(ClkPin | DataPin);  			//Reset ClkPin and DataPin to low
}

void each_led(int each_led_delay)			//A function that lights each LED one at a time
{
  for (u8 j=0; j<8; j++)
  {
    cli();
    row_buffer[j] = 0x0001;
    sei();
    Delay_ms(each_led_delay);
    for (char i=0; i<TotColumns; i++)
    {
      cli();
      row_buffer[j] <<= 1;
      sei();
      Delay_ms(each_led_delay);
    }
    cli();
    row_buffer[j] = 0x0000;
    sei();
  }
}

void up_down(u16 up_down_delay)
{
  for (u8 j=0; j<15; j++)
  {
    if (j % 2)
    {
      for (u8 b=0; b<8; b++)
      {
        cli();
        if (b == 0) row_buffer[7] = 0;
        else row_buffer[b-1] = 0;
        row_buffer[b] = 1<<j;
        sei();
        Delay_ms(up_down_delay);
      }
    }
    else
    {
      for (u8 b=0; b<8; b++)
      {
        cli();
        if (b == 0) row_buffer[7] = 0;
        else row_buffer[8-b] = 0;
        row_buffer[7-b] = 1<<j;
        sei();
        Delay_ms(up_down_delay);
      }
    }
  }
}

/* Write_Small_Num
Function that writes one digit from the 3x5 font file to the LED array
X and Y should be the location for the top left pixel of the 3x5 box
measuring from the right and bottom of the led array, the first location 
being 0,0 - Using an x value less than 4

X must be 2 or great and Y must be 4 or greater Otherwise a reset will be 
caused when trying to address an invalid array index
*/
void Write_Small_Num(u8 number, u8 x, u8 y)
{
  //TODO: Prevent non-valid characters (anything other than 0-9) from crashing the program
  //numbers com from the numbers3x5[] in progmem (3x5_numeric.h)

  char temp;
  cli();						//Atomic!
  for (char i=0; i<3; i++)				//Read one column of char at a time
  {
    temp = pgm_read_byte((char *)((int)numbers3x5 + (3 * number) + i));	//Get column from progmem
    for (char j=0; j<5; j++)						//Cycle through each bit in column
    {
      //Write bits to appropriate row_buffer location
      if (temp & (1<<j)) row_buffer[y-j] |= 1<<(x-i);
      else row_buffer[y-j] &= ~(1<<(x-i));
    }
  }
  sei();						//Atomic!
}

void Write_Char(u8 letter)		//Function that writes one character to the LED array (no scrolling)
{
  //Writes a char to the led matrix
  //TODO: Prevent non-valid characters from crashing program
  
  //letters come from font5x8[] in progmem (font5x8.h)
  letter -= 32;						//Adjust char value to match our font array indicies
  char temp;
  cli();						//Atomic!
  for (char i=0; i<5; i++)				//Read one column of char at a time
  {
    temp = pgm_read_byte((char *)((int)font5x8 + (5 * letter) + i));	//Get column from progmem
    for (char j=0; j<8; j++)						//Cycle through each bit in column
    {
      //Write bits to appropriate row_buffer location
      if (temp & (1<<j)) row_buffer[7-j] |= 1<<(4-i);
      else row_buffer[7-j] &= ~(1<<(4-i));
    }
  }
  sei();						//Atomic!
}

void shift_row_buffer_left(u8 columns)	//Function that shifts all rows a given amount to the left
{
  cli();							//Atomic!
  for (u8 buffer_index=0; buffer_index<8; buffer_index++)
  {
    row_buffer[buffer_index] <<= columns;
  }
  sei();							//Atomic!
}

void Scroll_Char (char myChar)	//Function that scrolls a character onto the display from right to left
{
  myChar -= 32;			//Adjust char value to match our font array indicies
  char temp;
  shift_row_buffer_left(1);  	//Put a blank column before each new letter.
  Delay_ms(ScrollSpeed);
  for (u8 i=0; i<5; i++)	//Read one column of char at a time
  {
    shift_row_buffer_left(1);
    temp = pgm_read_byte((char *)((int)font5x8 + (5 * myChar) + i));	//Get column from progmem
    cli();								//Atomic!
    for (u8 j=0; j<8; j++)						//Cycle through each bit in column
    {
      //Write bits to appropriate row_buffer location
      if (temp & (1<<j)) row_buffer[7-j] |= 1<<0;
      else row_buffer[7-j] &= ~(1<<0);
    }
    sei();								//Atomic!
    Delay_ms(ScrollSpeed);
  }
}
void Scroll_String(char * myString)	//Function that scrolls a given string onto the display
{
  while (*myString)			//Repeat until a null character is encountered (signals end of a string)
  {
    Scroll_Char(*myString);		//Scroll in the next character
    ++myString;				//Increment the point which tracks where we are in the string

  }
}

void Scroll_String_P(const char * myString)  //Function that reads a string out of memory and displays it
{
  while (pgm_read_byte(myString))
  {
    Scroll_Char(pgm_read_byte(myString));
    ++myString;
  }
}

void Scroll_Clear(void)			//Function that clears the display by shifting everything left
{
  for(u8 i=0; i<TotColumns; i++)
  {
    shift_row_buffer_left(1);
    Delay_ms(ScrollSpeed);
  }
}

void Scroll_From_Top(u8 x_pos, u8 letter)
{
  x_pos -= 5;
  letter -= 32;						//Adjust char value to match our font array indicies
  char temp;
  char temp_buffer[8];

  for (char i=0; i<5; i++)				//Read one column of char at a time
  {
    temp = pgm_read_byte((char *)((int)font5x8 + (5 * letter) + i));	//Get column from progmem
    for (char j=0; j<8; j++)						//Cycle through each bit in column
    {									
      //Write bits to a temporary 8x8 buffer (temp_buffer[])
      if (temp & (1<<j)) temp_buffer[7-j] |= 1<<(4-i);
      else temp_buffer[7-j] &= ~(1<<(4-i));
    }
  }

  //Scroll the display from top to bottom 8 times
  for (u8 i=0; i<8; i++)
  {
    cli();						//Atomic!
    //Shift each row down one
    for (u8 RowShift=0; RowShift<7; RowShift++)
    {
	row_buffer[7-RowShift] &= ~(0b11111<<x_pos);	//Shift 11111 into correct position in row to be written
	row_buffer[7-RowShift] |= (row_buffer[6-RowShift] & (0b11111<<x_pos)); //Mask off and set same columns from previous row.
    }
    row_buffer[0] &= ~(0b11111<<x_pos);			//Shift 11111 into correct position in row to be written
    //u16 temp_row = (temp_buffer[7-i] & 0x00FF);
    row_buffer[0] = (temp_buffer[7-i] & 0b11111)<<x_pos; //Write new row from temp_buffer

    sei();						//Atomic!
    Delay_ms(ScrollSpeed);
  }
}

void Clear_Display(void)
{
  cli();
  for (u8 i=0; i<8; i++) row_buffer[i] = 0x0000;
  sei();
}

void Fill_Display(void)
{
  cli();
  for (u8 i=0; i<8; i++) row_buffer[i] = 0xFFFF;
  sei();
}

void Flash_Display(u8 cycles, u8 speed) //Cycles = number of on/off flashes... Speed = milliseconds between toggles
{
  while (cycles)
  {
    Fill_Display();
    Delay_ms(speed);
    Clear_Display();
    Delay_ms(speed);
    --cycles;
  }
}

void Knight_Rider(u8 cycles, u8 row_number)
{
  cli();
  row_buffer[row_number] = 0x0001;
  sei();
  Delay_ms(20);

  while (cycles)
  {
    for (u8 i=0; i<TotColumns; i++)
    {
      cli();
      row_buffer[row_number] <<= 1;
      sei();
      Delay_ms(20);
    }
    for (u8 i=0; i<TotColumns; i++)
    {
      cli();
      row_buffer[row_number] >>= 1;
      sei();
      Delay_ms(20);
    }
    --cycles;
  }
}

int main(void)
{
  InitPorts();
  InitTimers();

  while(1)
  {
    up_down(35);
    Scroll_From_Top(15,'L');
    Scroll_From_Top(10,'e');
    Scroll_From_Top(5,'t');
    Delay_ms(1000);
    Scroll_Clear();
    Scroll_Char('I');
    Scroll_Char('t');
    shift_row_buffer_left(1);
    Delay_ms(ScrollSpeed);
    shift_row_buffer_left(1);
    Delay_ms(ScrollSpeed);
    Delay_ms(1000);
    Flash_Display(7,45);
    Write_Char('G');
    shift_row_buffer_left(5);
    Write_Char('l');
    shift_row_buffer_left(5);
    Write_Char('o');
    //shift_row_buffer_left(4);
    Delay_ms(2000);
    Knight_Rider(10,0);
    Clear_Display();
    Delay_ms(1000);
    each_led(50);
    Scroll_String_P(Messages2);
    Scroll_Clear();

    Clear_Display();
    cli(); //Atomic!
    row_buffer[3] = 1<<7;
    row_buffer[5] = 1<<7;
    sei(); //Atomic!
    u8 sec_t, sec_o, tsec_t, tsec_o;
    sec_t = sec_o = tsec_t = tsec_o = 0;
    while (sec_t < 6)
    {
      Write_Small_Num(sec_t,14,6);
      Write_Small_Num(sec_o,10,6);
      Write_Small_Num(tsec_t,6,6);
      Write_Small_Num(tsec_o,2,6);
      if (++tsec_o > 9)
      {
        tsec_o = 0;
        if (++tsec_t > 9)
        {
          tsec_t = 0;
          if (++sec_o > 9)
          {
            sec_o = 0;
            ++sec_t;
          }
        }
      }
      Delay_ms(10);
    }
    Scroll_Clear();
    Scroll_String_P(Messages1);
  }
}

ISR(TIMER1_COMPA_vect)				//Interrupt Service Routine handles the display.
{
  if(++row_track == 8) row_track = 0;		//Row tracking
  Shift_Int(row_buffer[7-row_track]);		//Shift in data for next row

  ShiftPort &= ~OE;				//Used to prevent ghosting (might be better to use: asm(NOP); ???)
  if (row_track == 0)				//Shut down high side controller
  {
    RowPort &= ~(1<<7);				//If Row0 is next, shudown Row7
  }
  else
  {
    RowPort &= ~(1<<(row_track-1));		//Shutdown previous row
  }

  ShiftPort |= LatchPin;			//Latch low side shift registers
  RowPort |= (1<<row_track);			//Drive high side controller
  ShiftPort |= OE;				//Used to prevent ghosting (may or may not be worth it.)
}
