/*
OscilloscopeExt.ino

Real time Oscilloscope using interrupts.

This version has edge triggering on PW2

Data is written to a 1000 byte buffer, which when full
is written to the Serial port.
Sent format:
  0 handshake
  4 bytes with integer value of frequency in milli Hertz
  1000 bytes of data
 
David Patterson 10.8.2014
Modified 22/9/2015

Equipment:
  Arduino Mega 2560
  LCD Keypad Shield
    http://www.hobbytronics.co.uk/arduino-lcd-keypad-shield
 
Specifications:
  >> Serial, LCD and flash memory support
  Serial Input can initiate new data capture- command 'again'
  PWM3 Square wave available for testing- toggle with command 'test'
  Trigger level set on or off with command 'trig'
  This trigger is an edge detect (positive or negative)
  The trig level is not used, other than to signal it is on or off!

  Repeated sampling available with command 'run'
    No serial port info is sent during a run- only raw data
    Sample period and edge select can be altered
    run is stopped using lcd select key
  
  Serial set at 115200 baud
  Lcd output of basic information
  Lcd adc port specified with variable lcdport
  Lcd button support:
    Select - initiates a new sample (or curtails a fast run)
    Up     - Positive edge trigger
    Down   - Negative edge trigger
    Left   - Decrease sampled period (increases Prescalar)
    Right  - Increase sampled period (decrease Prescalar)
    
  trig level is limited to on/off using a hardware interrupt
  
  SDCARD detected for future development.
  If required, the following hardware has been tested: 
  MicroSD Breakout Board Regulated with Logic Conversion V2
    http://www.hobbytronics.co.uk/microsd-card-regulated-v2?keyword=micro%20sd
  4GB Micro SD Memory Card
    http://www.hobbytronics.co.uk/4gb-microsd

 */


// Function F puts strings into flash memory, saving ram!
// inclusion not required after build 1.4
//#include <Flash.h>

#include <SD.h>
#include <SPI.h>

// set up variables using the SD utility library functions:
Sd2Card card;
SdVolume volume;
SdFile root;
const int chipSelect = 53; // (Or any suitable pin!)
const int defaultselect =53; // mega

#include <LiquidCrystal.h>
// initialize the library with the numbers of the interface pins
LiquidCrystal lcd(8, 9, 4, 5, 6, 7);

const int lcdlight=10; // lcd backlight control port
int lcdport=0; // adc port used by lcs for buttons
int const adport = 1; // Set the port to be used for input!
const byte testpin=3; // pwm outpin (Only pwm pins 2-3 available with LCD screen)
const unsigned int waittime = 5; // millisecond wait time for debounce

// display detailed info on Serial monitor
boolean showdetails;

boolean hascard=false;
boolean hasdata;
boolean pwtoggle = false;
float frequency, period , interval;
String temp;
unsigned long outtime, outdone;
unsigned long fasttime,starttime;

int i;
char junk=' ';

// variables for interrupt routine
boolean trigplus = true;
boolean triggered;
boolean writeit = false;
unsigned int bufcount;
unsigned long endtime;
int trigger;

boolean fastrun = false;

// default vref
float vref=5;

// Defines for clearing register bits
#ifndef cbi
#define cbi(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))
#endif
// Defines for setting register bits
#ifndef sbi
#define sbi(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))
#endif

#define isnull 0 // null character for serial info

// alter prescalar here 8, 16, 32, 64, 128
// 2 is the fastest sampling rate
// 8 to 128 work
byte prescalar=8;

#define BUF_SIZE 1000
uint8_t bufinfo[5];
uint8_t bufa[BUF_SIZE];

const byte ExtInterrupt = 2;

void setup(){
  /*
  Specimem frequency:
  if(prescalar==8)   frequency = 152.1 KHz;
  if(prescalar==16)  frequency = 76.8 Khz;
  if(prescalar==32)  frequency = 38.4 KHz;
  if(prescalar==64)  frequency = 19.2 KHz;
  if(prescalar==128) frequency = 9.6 KHz;
  */

triggered=true;
//pinMode(ExtInterrupt,INPUT_PULLUP);
pinMode(ExtInterrupt,INPUT);
attachInterrupt (0, gotinterrupt, RISING);

// Disable digital input buffer on ad ports
// power consumption saver
// oscilloscope will work without doing this
sbi(DIDR0,ADC15D);
sbi(DIDR0,ADC14D);
sbi(DIDR0,ADC13D);
sbi(DIDR0,ADC12D);
sbi(DIDR0,ADC11D);
sbi(DIDR0,ADC10D);
sbi(DIDR0,ADC9D);
sbi(DIDR0,ADC8D);
sbi(DIDR0,ADC7D);
sbi(DIDR0,ADC6D);
sbi(DIDR0,ADC5D);
sbi(DIDR0,ADC4D);
sbi(DIDR0,ADC3D);
sbi(DIDR0,ADC2D);
sbi(DIDR0,ADC1D);
sbi(DIDR0,ADC0D);
pinMode(chipSelect, OUTPUT);
  if (chipSelect != defaultselect) { pinMode(defaultselect, OUTPUT); }
Serial.begin(115200); // start serial for output
  while (Serial.available() > 0) junk = Serial.read();
Serial.println(F("Oscilloscope"));
Serial.println(F("This version can be triggered on P2\n"));

lcd.begin(16, 2);
// setup lcdbacklight
pinMode(lcdlight,OUTPUT);
digitalWrite(lcdlight,HIGH);
lcd.setCursor(0,0);lcd.print(F("Hard Trigger P2"));
lcd.setCursor(2,1);lcd.print(F("Oscilloscope"));
delay(1000);

// set testpin to output
pinMode(testpin,OUTPUT);
analogWrite(testpin, 0);

// Ram test out of interest
Serial.print(F("Free RAM: "));
Serial.println(FreeRam());

if (SD.begin(chipSelect)) {
  Serial.println(F("SDcard Present\n"));
  hascard=true;
  card.init(SPI_FULL_SPEED, chipSelect);
  // future development?
  }
trigger=0;
Serial.flush();
showdetails=true;
startad();
}

void startad(){
while (Serial.available() > 0) junk = Serial.read();
hasdata=false;
bufcount=0;
writeit=false;
// pwm square wave
  if(pwtoggle) analogWrite(testpin, 127);
  
// Setup continuous reading of the adc port 'adport' using an interrupt
cli();//disable interrupts

//clear ADCSRA and ADCSRB registers
ADCSRA = 0;
ADCSRB = 0;
ADMUX |= adport;   //set up continuous sampling of analog pin adport 
/*
internal voltage reference options may not be used if an external
reference voltage is being applied to the AREF pin.
  REFS1   REFS0 Voltage reference
  0       0     AREF, Internal Vref turned off
  0       1     AVCC with external capacitor at AREF pin
  1       0     Reserved
  1       1     Internal 1.1V Voltage Reference with external
                capacitor at AREF pin
*/
ADMUX |= (1 << REFS0); //set reference voltage

/*  The ADLAR bit controls the presentation of the ADC conversion
    Write one to ADLAR to left adjust.
    Otherwise, the value is right adjusted.
    Immediate effect on the ADC Data Register. */
ADMUX |= (1 << ADLAR); //left align the ADC value- so we can read highest 8 bits from ADCH register only

/*  These bits determine the division factor between the system clock
    frequency and the input clock to the ADC.
    ADPS2  ADPS1  ADPS0  Division Factor
    0      0      0      2
    0      0      1      2
    0      1      0      4
    0      1      1      8
    1      0      0      16
    1      0      1      32
    1      1      0      64
    1      1      1      128
eg usage, to set 128:
sbi(ADCSRA,ADPS2);
sbi(ADCSRA,ADPS1);
sbi(ADCSRA,ADPS0);
The method detailed below employs a single code line
*/
// 4 prescalar- interrupt faster than program speed
if (prescalar==4) ADCSRA |= (1 << ADPS1);

// 8 prescalar 160Khz (tolerable interrupt speed reduction)
if (prescalar==8) ADCSRA |= (1 << ADPS1) | (1 << ADPS0);
// 16 prescalar - 72 Khz sampling
if (prescalar==16) ADCSRA |= (1 << ADPS2); 
// 32 prescaler - 16mHz/32=500kHz - produces 37 Khz sampling
if (prescalar==32) ADCSRA |= (1 << ADPS2) | (1 << ADPS0);
// 64 prescalar produces 19.2 Khz sampling
if(prescalar==64) ADCSRA |= (1 << ADPS2) | (1 << ADPS1);
// 128 prescalar - 9.4 Khz sampling
if (prescalar==128) ADCSRA |= (1 << ADPS2) | (1 << ADPS1) | (1 << ADPS0);

// enable auto trigger
ADCSRA |= (1 << ADATE);
// Activate ADC Conversion Complete Interrupt
ADCSRA |= (1 << ADIE);

  if(showdetails){
  Serial.println(F("Logging.."));Serial.flush();
  }
sbi(ADCSRA,ADEN); // enable ADC

  if (trigger==0) {
  sbi(ADCSRA,ADSC); //start ADC measurements on interrupt
  starttime=micros();
  }else{
  // allow pw2 interrupt to start adc interrupt
  triggered=false;
  }
sei();//enable interrupts
}
 
// Interrupt routine *******************************************
// this is the key to the program!!
ISR(ADC_vect) {
bufa[bufcount]=ADCH;
bufcount++; // increment buffer counter
  if (bufcount==BUF_SIZE) {
  cbi(ADCSRA,ADEN); // disable ADC
  endtime=micros();
  writeit=true; // flag that a write is needed  
  }
}

void gotinterrupt() {
  if (!triggered){
  triggered=true;
  sbi(ADCSRA,ADSC); //start ADC measurements on interrupt
  starttime=micros();
  }
}
// End Interupt section *******************************************

void loop(){
  if(writeit){
    // switch of pwm
    if (pwtoggle)analogWrite(testpin, 0);
    
    if(endtime<starttime){
    //micros() roll over at 4294967295
    unsigned long rolltime = 4294967295 - starttime;
    period=(rolltime+endtime);
    } else { 
    period= (endtime-starttime);
    }
  // Period in milli seconds, frequency in Khz
  period = period/1000;
  frequency = float(BUF_SIZE)/period;
  // send frequency in milli Hertz
  //byte 0, then 4 bytes big endian
  writebufinfo(frequency*1000000); // setup header with frequency
  outtime=millis();
  Serial.write(bufinfo, 5);// send buffer header
  // initiate block write from bufa
  Serial.write(bufa, BUF_SIZE );
  Serial.flush();
  outdone=millis();
  writeit=false; // the write is done!
  // enable lcd port
  cli();
  ADCSRA = 0;
  ADCSRB = 0;
  ADMUX |= lcdport;
  sbi(ADCSRA,ADEN);
  sei();
  // wake up the port
  i=analogRead(lcdport);
  
    if(showdetails){
    Serial.println(F("\nLogging stopped:"));
    //Serial.print(F("\nSerial Output took "));
    //Serial.print(outdone-outtime);
    //Serial.println(F(" mS"));
    title();
    lcd.clear();
    lcd.print(period,0);
    lcd.print(F(" mS"));
    lcd.setCursor(8,0);
    lcd.print(frequency,1);
    lcd.print(F("KHz"));
    lcd.setCursor(0,1);
    lcd.print(F("Sel:Agn <   >Tm"));
      if(trigplus)lcd.print(F("+")); else lcd.print(F("-"));
    lcdPrescalar();
    }
  hasdata=true; // flag the data presence
  }
  
  if(hasdata==true) {  
  buttoncheck();
    if (fastrun) {
    fasttime=millis();
    temp="";
    do{
    i=Serial.read();
    // handshake or timeout   
    } while ((i == -1) && (millis()-fasttime < 1500));
    startad();
    } else {
    commandcheck();    
    }  
  }
}
// End Void Loop section ******************************************

void buttoncheck(){
int x=analogRead(lcdport);
  if (x > 799) return; // nothing to do
int w; 
  if ((x >599) && (x < 800)) { // enter

    do{
    delay(waittime);
    w=analogRead(lcdport);
    } while (w < 800); 
    if (!fastrun){
    showdetails = true;
    startad();
    } else {
    fastrun = false;
    lcd.clear();
    lcd.print(F("> ended"));
    Serial.println(F("> ended"));
    Serial.flush();
    }
  return;  
  } else if (x < 60) { // right
      do{
      delay(waittime);
      w=analogRead(lcdport);
      } while (w < 800);
      if(prescalar==128) return;
  // right button increase time
  prescalar=prescalar << 1;
  Serial.print(F("Prescalar "));
  Serial.println(prescalar);
  lcdPrescalar();
  } else if  ((x > 399) && (x < 600)) { // left
    do{
    delay(waittime);
    w=analogRead(lcdport);
    } while (w < 800);
    if (prescalar==8) return;
  // left button decrease time
  prescalar=prescalar >> 1;
  Serial.print(F("Prescalar "));
  Serial.println(prescalar);
  lcdPrescalar(); 
  } else if ((x > 59) &&(x < 200)) { // up
    do{
    delay(waittime);
    w=analogRead(lcdport);
    } while (w < 800);
  trigplus=true; // up
  attachInterrupt (0, gotinterrupt, RISING);
  Serial.println(F("Trigger positive"));
  lcd.setCursor(15,1);lcd.print(F("+"));
  } else if ((x > 199 ) && (x < 400)) { // down
    do{
    delay(waittime);
    w=analogRead(lcdport);
    } while (w < 800);
  trigplus=false; // down
  attachInterrupt (0, gotinterrupt, FALLING);
  Serial.println(F("Trigger negative"));
  lcd.setCursor(15,1);lcd.print(F("-"));
  }
Serial.flush();
}

void lcdPrescalar(){
lcd.setCursor(9,1);
  if (prescalar < 100){
  lcd.print(F(" "));
    if (prescalar < 10)lcd.print(F(" "));
  }
lcd.print(prescalar);
}

void commandcheck(){
  if (Serial.available() > 0) {
  temp=Serial.readString();
    if (temp != isnull) {
    String reply="\n> "+temp;
    Serial.println(reply);
    Serial.flush();
    }
    if (temp=="test") {
    pwtoggle=togglepw(pwtoggle);
    } else if (temp=="again") {
    showdetails=true;
    startad();
    } else if(temp=="trig") {
    setTrig();
    } else if(temp=="run") {
    lcd.clear();
    lcd.print(F("Use Select key"));
    lcd.setCursor(0,1);
    lcd.print(F("to exit <   >Tm"));
      if(trigplus)lcd.print(F("+")); else lcd.print(F("-"));
    lcdPrescalar();
    fastrun = true;
    showdetails=false;
    } else if(temp=="vref") {
    setvref();
    } else {
    while (Serial.available() > 0) junk = Serial.read();
    }  
  }
}

void setTrig() {
Serial.print(F("Enter 0 to remove trigger, >0 to trigger , old value "));
if (trigger==0) Serial.println(F("Off")); else Serial.println(F("On"));
Serial.flush();
  do {
  delay(10);
  } while(Serial.available()==0);
unsigned long data=Serial.parseFloat();
trigger = abs(data);
  if (trigger == 0){
  Serial.println(F("Trigger off"));
  }else{
  Serial.println(F("Trigger on- ensure that the input is AC")); 
  }
Serial.flush();
  while (Serial.available() > 0) junk = Serial.read();
}

void setvref() {
Serial.print(F("New vref (0-10V), old value "));
Serial.println(vref);Serial.flush();
  do {
  delay(10);
  } while(Serial.available()==0);
float vtry=Serial.parseFloat();
  if ((vtry>0) && (vtry <=10)) {
  vref = vtry;
  Serial.print(F("New vref "));
  Serial.println(vref);Serial.flush();
  }
while (Serial.available() > 0) junk = Serial.read();
}

void title(){
Serial.print(F("Prescalar "));
Serial.print(prescalar);
Serial.print(F(" , Trigger "));
  if (trigger != 0) Serial.print(F("on"));else Serial.print(F("off"));
Serial.print(F(" , Edge "));
  if (trigplus) {
  Serial.println("+");
  }else{
  Serial.println("-");
  }
Serial.print(F("Samples taken "));
Serial.println(BUF_SIZE);
Serial.print(F("Frequency "));
Serial.print(frequency,4);
Serial.print(F(" KHz , Sample Duration "));
Serial.print(period,3);
Serial.println(F(" mS"));
interval = 1000/frequency;
Serial.print(F("Measuring interval "));
Serial.print(interval,3);
Serial.println(F(" microS"));
Serial.print(F("\nSend 'again' to re-measure, 'test' to toggle PW"));
Serial.print(testpin);
Serial.println(F(" test wave,\n'trig' to alter trig level, 'run' to repeatedly sample."));
Serial.flush();
}

boolean togglepw(boolean tgl){
// toggle test square wave on pwm
while (Serial.available() > 0) junk = Serial.read();
  if(!tgl){
  Serial.print(F("PWM test wave available on PW"));
  Serial.println(testpin);Serial.flush();
  return(true);
  } else {
  Serial.print(F("PWM test wave not available on PW"));
  Serial.println(testpin);Serial.flush();
  return(false);
  }
}

void writebufinfo(unsigned long bufvalue) {
bufinfo[0]=isnull;
// big endian
int temp = int(bufvalue  >> 24);
bufinfo[1]= temp;
bufvalue = bufvalue << 8;
temp = int(bufvalue  >> 24);
bufinfo[2]= temp;
bufvalue = bufvalue << 8;
temp = int(bufvalue  >> 24);
bufinfo[3]= temp;
bufvalue = bufvalue << 8;
temp = int(bufvalue  >> 24);
bufinfo[4]= temp;
// Serial.println(readbufinfo());
}

// Utility functions for future development ********************

unsigned long readbufinfo() { 
unsigned long temp = bufinfo[1];
temp = temp <<8;
temp = temp + bufinfo[2];
temp = temp <<8;
temp = temp + bufinfo[3];
temp = temp <<8;
temp = temp + bufinfo[4];
return(temp);
}

void error(char* s) {
// display error and enter stasus
Serial.print(F("Error "));
Serial.println(s);
lcd.clear();
lcd.print(F("Error"));
lcd.setCursor(0,1);
lcd.print(s);
while(1); // no point in continuing
}

String pad(unsigned long myval,int digit,String p){
// pad a number to digit characters with string p
  temp = String(myval);
int k=temp.length();
  for(int j=0;j < (digit-k);j++){
  temp = p + temp;
  }
  return(temp);
}

// *************************************************************

