// Camera Lens Controller by Guy Perez, 2012
// This sketch allows you to control 2 servo motors with a Wii Classic Controller (using the Library from:
// http://arduino.cc/playground/Main/WiiClassicController10  
// Moving the right stick front-back provides control of the Zoom and Focus servos simultaneously (it synchronizes the focus with the zoom
// to keep the camera in focus based on in focus settings at the widest and closest zoom points).
// Moving the left stick side to side provides control of the focus servo only, allowing you to adjust focus without changing the zoom.
// You can also program a number of settings from the controller using the buttons. All buttons are programmed by moving the servo to
// the desired position and pressing Home, then the button that corresponds to the setting. Here are the button settings and the 
// sequence in which you should set them:

// LEFT SHOULDER - widest zoom setting. Before engaging the servo to the lens, move the lens to it's widest zoom setting and pull back on
// the right stick until the servo stops moving, then engage the servo to the lens gear and press HOME, then LEFT SHOULDER. This sets the 
// widest zoom setting on the lens

// RIGHT SHOULDER - closest zoom setting. Push the right stick forward until the lens hits the closest zoom setting, then back off one 
// degree using the down DPAD and press HOME, then RIGHT SHOULDER to set the maximum zoom of the lens.

// LEFT Z and RIGHT Z - leftmost focus position and rightmost focus position (relative to the left stick movement). Set these the same way 
// you did for Widest and Closest zoom.

// - BUTTON - focus at widest Zoom setting. Press the LEFT SHOULDER to move the lens to the widest zoom setting, then point the camera 
// at the subject that you will maintain focus on while zooming, then dial in the focus using the left stick and left-right D-Pad. Once the 
// focus is perfect, press HOME then "-" to set the focus at the widest setting.

// + BUTTON - focus at the closest zoom setting. Keep camera pointed at the object on which you focus for the "-" Button. Press the 
// RIGHT Shoulder to move the lens to the closest zoom setting, then carefully re-focus on the subject using the left stick and left-right 
// D-Pad. Once the focus is perfect, press HOME then "+" to set the focus at the closest zoom setting.

// Every time you program the - or + buttons, the code recalculates the ratio to move the focus servo to keep you subject in focus as you 
// zoom in and out on that subject. Obviously if the distance between the subject and camera changes, you'll have to change your focus and 
// if the change is great, you may also have to re-program the - and + buttons to calculate a new ratio. However I think that you'll find 
// that this is easy to accomplish .

// a, b, x, and y BUTTONS - "Goto" Zoom/focus settings. These allow you to move the lens to a specific focus and zoom setting by simply
// puching the corresponding button. To program any of these buttons, move the zoom and focus to the desired position using the sticks and 
// D-Pad, then press HOME and then either a, b, x or y to program the button for the corresponding lens positions.

// Follow Focus set up:
// If you want to set the buttons to only move the focus servo for a follow focus type operation, Just leave move only the focus servo 
// using the left stick as you program each button. That way you can program four focus positions and simply click the button to focus the 
// camera to that position as your subject hits the focus mark.

#include <Servo.h>
#include <Wire.h>
#include "WiiClassic.h"

// define pins
//const int speedPin = A0; //orange
const int ratioPin = A1; //yellow

//variables
int oneMotorDelay = 5; //factor used to add delay when only stepping with one servo
int dPadDelay = 40; //used for very slow movement with D pad controls
int fRatio = 10; //focus ratio - the number of steps of zoom motor for each step of focus servo
int speedDelayMin = 7; //overall speed - is used as a delay in milliseconds between steps
int speedDelayMax = 24; //maximum speed delay value (used to map speed range)
int jumpToSpeed = 8; //determines how much to delay between each "step" for button based moves.
int reRead = 2; //how frequently to re-read the controller
int numSteps = 1; //how many degrees per iteration
int updateDelay = 5; //mow long to delay after reading the controller
int angleMax = 360; //maximum angle of the servos
int angleMin = 0; //minimum angle of the servos

int y = reRead;
int zSteps = 0;

//variables for passing angle to servo and set default values
int zoomAngle = angleMax;
int focusAngle = angleMin;

//programmable servo limits
int zoomMin = angleMin; //position of the zoom servo for minimum zoom (widest setting)
int zoomMax = angleMax; //position of the zoom servo for maximum zoom (most zoomed in)
int focusMin = angleMin; //position of the focus servo for the closest focus position of the lens
int focusMax = angleMax; //position of the focus servo for the infinity focus position of the lens

//programmable servo settings 
int yZoom = zoomAngle;
int xZoom = zoomAngle;
int bZoom = zoomAngle;
int aZoom = zoomAngle;
int yFocus = focusAngle;
int xFocus = focusAngle;
int bFocus = focusAngle;
int aFocus = focusAngle;

//programmable 
int focusMinZoom = angleMin;
int focusMaxZoom = angleMax;

boolean programMode = false;

// Connect a stepper motor with 200 steps per revolution (1.8 degree per step)
// to motor port #2 (M3 and M4)
Servo zoomServo;
Servo focusServo;

//controller stuff
WiiClassic controller = WiiClassic();

// the right stick has 1/2 the resolution of the left - these values may vary from one
// controller to another, so you'll need to run a test program to discern the values of 
// each stick position
int yCenterRight = 15;
int yMinRight = 2;
int yMaxRight = 28;
int xCenterRight = 15;
int xMinRight = 3;
int xMaxRight = 28;
int centerOffsetRight = 3;
int endOffsetRight = 0;

int yCenterLeft = 32;
int yMinLeft = 6;    
int yMaxLeft = 55;
int xCenterLeft = 31;
int xMinLeft = 6;
int xMaxLeft = 55;
int centerOffsetLeft = 6;
int endOffsetLeft = 0;



void setup() {
  Serial.begin(9600);           // set up Serial library at 9600 bps
  //attach servos and set them to initial positions for mounting the servos to the lens
  zoomServo.attach(9);
  focusServo.attach(10);
  zoomServo.write(angleMax);
  focusServo.write(angleMin);
  
  //start controller and delay to allow the stick readings to settle
  controller.begin();
  delay(100);
}

void moveZoomFocusTo(int zoomTo, int focusTo, int delayTime) {
  // moves the zoom and focus servos to the specified positions using the delay time provided
  // this procedure is called by all of the programmable buttons and is coded to interleave
  // "steps" for more synchonized movement between focus and zoom
  int moveZoom = abs(zoomTo - zoomServo.read());
  int moveFocus = abs(focusTo - focusServo.read());
  
  //debugging stuff 
  //Serial.print("delayTime;"); Serial.println(delayTime);
/*Serial.print("zoomTo;"); Serial.println(zoomTo);
  Serial.print("focusTo;"); Serial.println(focusTo);
  Serial.print("ratioZoomFocus"); Serial.println(ratioZoomFocus);
  delay(500); */
  int x = 0;
  if (moveZoom > moveFocus) {      //this branch is used when the zoom servo hsa to move more than the focus servo
    int ratio = moveZoom / moveFocus ;
    while ((zoomServo.read() != zoomTo) || (focusServo.read() != focusTo)) {
      if (zoomServo.read() != zoomTo) {
        zoomServo.write(zoomServo.read() + ((zoomTo - zoomServo.read())/abs(zoomTo - zoomServo.read())));
        x++;
      }
      if ((focusServo.read() != focusTo) && (x == ratio)) {
        focusServo.write(focusServo.read() + ((focusTo - focusServo.read())/abs(focusTo - focusServo.read())));
        x = 0;
      }
      delay(delayTime);
    }
  }
  else if (moveFocus > moveZoom) {      //this branch is used when the focus servo has to move more than the zoom servo
    int ratio = moveFocus / moveZoom;
    while ((zoomServo.read() != zoomTo) || (focusServo.read() != focusTo)) {
      if (focusServo.read() != focusTo) {
        focusServo.write(focusServo.read() + ((focusTo - focusServo.read())/abs(focusTo - focusServo.read())));
        x++;
      }
      if ((zoomServo.read() != zoomTo) && (x == ratio)) {
        zoomServo.write(zoomServo.read() + ((zoomTo - zoomServo.read())/abs(zoomTo - zoomServo.read())));
        x = 0;
      }
      delay(delayTime);
    }
  }
  else if (moveFocus = moveZoom) {      //this branch is used when the 2 servos move the same amount - it's rarely going to be used
    while ((zoomServo.read() != zoomTo) || (focusServo.read() != focusTo)) {
      if (focusServo.read() != focusTo) {
        focusServo.write(focusServo.read() + ((focusTo - focusServo.read())/abs(focusTo - focusServo.read())));
      }
      if (zoomServo.read() != zoomTo) {
        zoomServo.write(zoomServo.read() + ((zoomTo - zoomServo.read())/abs(zoomTo - zoomServo.read())));
      }
      delay(delayTime);
    }
  }
}

void loop() {
  //read the speed and ratio values and map them to useable ranges
  controller.update();
  zSteps = 0;
  
  if (controller.homePressed()) {
    delay(500);
    programMode = !programMode;
    //toggles programMode if the home button is pressed -- need to illuminate a led for this down the road
  }
  
  // Notes about servo angles and zoom - focus movement
  // Zoom Servo:  Min zoom at 360 degrees
  //              Max zoom at 0 degrees
  //              (servo moves from 360 to 0 in order to zoom, so NEGATIVE STEPS)
  // Focus Servo: Closest focus at 360 degrees
  //              Inf focus at 0 degress
  //              (servo moves from 360 to 0 in order to focus 
  
  // Notes about servo / lens ring direction
  // I am trying to gifure out how to modify the Servo library to accept a servo direction parameter (I tried a method that
  // someone else tried but it didn't work). Until that is implemented, you may have to change stick-based commands to accomodate
  // different lenses or different servos or mounting positions so that the lens zooms when you puch the right stick forward
  // and goes wider when you push it back. You may also want to change the direction of the lens movement for focus to suit your 
  // preferences. I have mine set up so that if I push the left stick left, the focus moves closer, and if I push the left stick
  // right, the focus moves towards infinity. You don't have to worry about direction for the button commands, since those always
  // recall the servo position recorded when you programmed the button and go to that position. You would only need to change 
  // the servo direction for your stick commands.
  
  // To change the servo direction relative to stick movement, simply change the "-"'s to "+"'s and the ">="'s to "<="'s
  // for the given servo. DON'T CHANGE ANYTHING ELSE. Fr example if you wanted to change the direction that the zoomServo 
  // moves when you press the stick forward you would change this:
  //   while ((zoomServo.read() - numSteps >= zoomMin) && (controller.rightStickY() > yCenterRight + centerOffsetRight)) {
  // to this:
  //   while ((zoomServo.read() + numSteps <= zoomMin) && (controller.rightStickY() > yCenterRight + centerOffsetRight)) {
  // and this:
  //    zoomAngle = zoomServo.read() - numSteps;
  // to this:
  //    zoomAngle = zoomServo.read() + numSteps;
  // You would have to do the opposite to change the direction that the zoomServo moves when you pull the stick back.
  
  //Zoom In with Synchronized focus//moves the zoom lens and adjust focus to maintain focus based on the calculated f-ratio
  while ((zoomServo.read() - numSteps >= zoomMin) && (controller.rightStickY() > yCenterRight + centerOffsetRight)) {
    //first move the zoom servo and increment zSteps counter
    zoomAngle = zoomServo.read() - numSteps;
    zoomServo.write(zoomAngle);
    zSteps++;
    //then only move focus servo for every fRatio steps of zoom servo
    if ((focusServo.read() - numSteps >= focusMin) && (zSteps == fRatio)) {
      focusAngle = focusServo.read() - numSteps;
      focusServo.write(focusAngle);
      zSteps = 0; //need to reset the step counter
    }
    
    //now delay to reflect selected speed
    delay(map(controller.rightStickY(), yCenterRight + centerOffsetRight, yMaxRight - endOffsetRight, speedDelayMax, speedDelayMin)); 
    //read speed, but only every Nth iteration 
    // code to update the nunchuk less frequently
    if (y == reRead) {
      controller.update();
      //delay(updateDelay);
      y = 0;
    }
    else y++;
  }
  
  //Zoom Out with synchronized focus
  while ((zoomServo.read() + numSteps <= zoomMax) && (controller.rightStickY() < yCenterRight - centerOffsetRight)) {
    //first step the zoom motor and increment zSteps counter
    zoomAngle = zoomServo.read() + numSteps;
    zoomServo.write(zoomAngle);
    zSteps++;
    //then only step once for every fRatio steps of zoom motor
    if ((focusServo.read() + numSteps <= focusMax + numSteps) && (zSteps == fRatio)) {
      focusAngle = focusServo.read() + numSteps;
      focusServo.write(focusAngle);
      zSteps = 0; //need to reset the step counter
    }
    
    //now delay to reflect selected speed
    delay(map(controller.rightStickY(), yCenterRight - centerOffsetRight, yMinRight + endOffsetRight, speedDelayMax, speedDelayMin));
    //read speed, but only every Nth iteration 
    // code to update the nunchuk less frequently
    if (y == reRead) {
      controller.update();
      //delay(updateDelay);
      y = 0;
    }
    else y++;
  }
 
  //Focus In only with left stick
  while ((focusServo.read() - numSteps >= focusMin) && (controller.leftStickX() > xCenterLeft + centerOffsetLeft))  {
    focusAngle = focusServo.read() - numSteps;
    focusServo.write(focusAngle);
    delay(map(controller.leftStickX(), xCenterLeft + centerOffsetLeft, xMaxLeft - endOffsetLeft, speedDelayMax, speedDelayMin) + oneMotorDelay);
    //read speed, but only every Nth iteration 
    // code to update the nunchuk less frequently
    if (y == reRead) {
      controller.update();
      //delay(updateDelay);
      y = 0;
    }
    else y++;
  }

  //Focus Out Only with left stick
  while ((focusServo.read() + numSteps <= focusMax) && (controller.leftStickX() < xCenterLeft - centerOffsetLeft)) {
    focusAngle = focusServo.read() + numSteps;
    focusServo.write(focusAngle);
    delay(map(controller.leftStickX(), xCenterLeft - centerOffsetLeft, xMinLeft + endOffsetLeft, speedDelayMax, speedDelayMin) + oneMotorDelay);
    //read speed, but only every Nth iteration 
    // code to update the nunchuk less frequently
    if (y == reRead) {
      controller.update();
      //delay(updateDelay);
      y = 0;
    }
    else y++;
  }  
  
  //Slow Zoom In using pad
  while ((zoomServo.read() - numSteps >= zoomMin) && (controller.upDPressed()))  {
    zoomAngle = zoomServo.read() - numSteps;
    zoomServo.write(zoomAngle);
    delay(dPadDelay);
    //read speed, but only every Nth iteration 
    // code to update the nunchuk less frequently
    if (y == reRead) {
      controller.update();
      //delay(updateDelay);
      y = 0;
    }
    else y++; 
  }

  //Slow Zoom Out using pad
  while ((zoomServo.read() + numSteps <= zoomMax) && (controller.downDPressed())) {
    zoomAngle = zoomServo.read() + numSteps;
    zoomServo.write(zoomAngle);
    delay(dPadDelay);
    //read speed, but only every Nth iteration 
    // code to update the nunchuk less frequently
    if (y == reRead) {
      controller.update();
      //delay(updateDelay);
      y = 0;
    }
    else y++;
  }   
  
  //Slow Focus In using pad
  while ((focusServo.read() - numSteps >= focusMin) && (controller.rightDPressed()))  {
    focusAngle = focusServo.read() - numSteps;
    focusServo.write(focusAngle);
    delay(dPadDelay);
    //read speed, but only every Nth iteration 
    // code to update the nunchuk less frequently
    if (y == reRead) {
      controller.update();
      //delay(updateDelay);
      y = 0;
    }
    else y++; 
  }

  //Slow Focus Out using pad
  while ((focusServo.read() + numSteps <= focusMax) && (controller.leftDPressed())) {
    focusAngle = focusServo.read() + numSteps;
    focusServo.write(focusAngle);
    delay(dPadDelay);
    //read speed, but only every Nth iteration 
    // code to update the nunchuk less frequently
    if (y == reRead) {
      controller.update();
      //delay(updateDelay);
      y = 0;
    }
    else y++;
  }
  
  //one-touch zoom and focus using y, x, b and a buttons (need to add speed control later)
  if (!programMode && controller.yPressed()) {
    moveZoomFocusTo(yZoom, yFocus, jumpToSpeed);
    //zoomServo.write(yZoom);
    //focusServo.write(yFocus);
  }
  if (!programMode && controller.xPressed()) {
    moveZoomFocusTo(xZoom, xFocus, jumpToSpeed);
    //zoomServo.write(xZoom);
    //focusServo.write(xFocus);
  }
  if (!programMode && controller.aPressed()) {
    moveZoomFocusTo(aZoom, aFocus, jumpToSpeed);
    //zoomServo.write(aZoom);
    //focusServo.write(aFocus);
  }
  if (!programMode && controller.bPressed()) {
    moveZoomFocusTo(bZoom, bFocus, jumpToSpeed);
    //zoomServo.write(bZoom);
    //focusServo.write(bFocus);
  }
  if (!programMode && controller.leftShoulderPressed()) {
    moveZoomFocusTo(zoomMax, focusMaxZoom, jumpToSpeed);
    //zoomServo.write(bZoom);
    //focusServo.write(bFocus);
  }
  if (!programMode && controller.rightShoulderPressed()) {
    moveZoomFocusTo(zoomMin, focusMinZoom, jumpToSpeed);
    //zoomServo.write(bZoom);
    //focusServo.write(bFocus);
  }
  
  //programming lens settings is invoked by Home-------------------------------------------------------
    //set min zoom
    if (programMode && controller.leftShoulderPressed()) {
      zoomMax = zoomServo.read();
      programMode = false;
    }
    //set max zoom
    if (programMode && controller.rightShoulderPressed()) {
      zoomMin = zoomServo.read();
      programMode = false;
    }
    //set max focus
    if (programMode && controller.lzPressed()) {
      focusMax = focusServo.read();
      programMode = false;
    }
    //set min focus
    if (programMode && controller.rzPressed()) {
      focusMin = focusServo.read();
      programMode = false;
    }
    //set focus at min zoom and calculate focus ratio
    if (programMode && controller.selectPressed()) {
      focusMaxZoom = focusServo.read();
      fRatio = abs(zoomMax -  zoomMin) / (focusMaxZoom - focusMinZoom);
        //fRatio = difference between min and max zoom / difference between focus at min zoom and focus at max zoom
      programMode = false;
    }
    //set focus at max zoom and calculate focus ratio
    if (programMode && controller.startPressed()) {
      focusMinZoom = focusServo.read();
      fRatio = abs(zoomMax -  zoomMin) / (focusMaxZoom - focusMinZoom);
        //fRatio = difference between min and max zoom / difference between focus at min zoom and focus at max zoom
      programMode = false;
    }
    //set zoom and focus positions for y,x,b and a buttons
    if (programMode && controller.xPressed()) {
      xZoom = zoomServo.read();
      xFocus = focusServo.read();
      programMode = false;
    }
    if (programMode && controller.yPressed()) {
      yZoom = zoomServo.read();
      yFocus = focusServo.read();
      programMode = false;
    }
    if (programMode && controller.bPressed()) {
      bZoom = zoomServo.read();
      bFocus = focusServo.read();
      programMode = false;
    }
    if (programMode && controller.aPressed()) {
      aZoom = zoomServo.read();
      aFocus = focusServo.read();
      programMode = false;
    }
  
  delay(10);

}
