﻿/*=========================================================================
   This file is part of the Cardboard Robot SDK.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;

namespace CBRobot {

    /** Base class for all vector types in the library.
     * 
     *  You cannot do arithmetic on a Vector directly.  Instead, you must
     *  first convert to a specific type of vector, using the methods on this
     *  interface.
     *
     *  Currently, the only supported vector type is DOF (DofVector).  If
     *  support for the Cartesian coordinate system is added in the future,
     *  this class will allow for conversion between the types.
     */
    public abstract class Vector {

        /** Interprets this vector as the position of the arm's tip, and
         *  returns its DOF (motor position) representation.
         *
         *  If there is no exact conversion (i.e., the point is outside of the
         *  arm's reach), an approximation is returned instead (the closest
         *  position that is within reach).
         */
        public abstract DofVector ConvertToDofPoint(Robot robot);

        /** Returns a new vector with the specified component (0 - 2) set to
         *  the given value, doing coordinate system conversion as needed.
         */
        public Vector SetComponent(int componentIndex, double value, CoordinateSystem cs, Robot robot) {
            switch (cs) {
                case CoordinateSystem.Dof:
                    return ConvertToDofPoint(robot).SetComponent(componentIndex, value);
                default:
                    throw new ArgumentException("Unsupported coordinate system", "cs");
            }
        }

        /** Gets the specified vector component (0 - 2), doing coordinate
         *  system conversion as needed.
         */
        public double GetComponent(int componentIndex, CoordinateSystem cs, Robot robot) {
            switch (cs) {
                case CoordinateSystem.Dof:
                    return ConvertToDofPoint(robot).GetComponent(componentIndex);
                default:
                    throw new ArgumentException("Unsupported coordinate system", "cs");
            }
        }
    }

}
