﻿/*=========================================================================
   This file is part of the Cardboard Robot SDK.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Windows.Forms;

namespace CBRobot {

    /** The main cardboard robot class.  Most functionality to control the robot
     *  can be accessed from here.
     *  
     *  Note: You MUST dispose of the robot instance after you are done with
     *  it, or the application will never quit.  A good place to do this is in
     *  the FormClosed event handler of your main form.
     *  
     *  Connection / disconnection will be detected automatically, but only
     *  at an interval of 5 seconds.  To enable faster detection, you must
     *  pass in a valid window handle to the constructor, and forward windows
     *  messages from your form to the HandleWindowsMessage() function.
     */
    public class Robot : IDisposable {

        private double armLength1;                  /* Arm 1 length, from pivot to pivot */
        private double armLength2;                  /* Arm 2 length, from pivot to pivot */
        private Device device;                      /* Connection to the USB device */
        private MotorParameters[] motorParameters;  /* Parameters for each motor */
        private ArmPosition currentPosition;        /* Current position of the arm */
        private ArmPosition targetPosition;         /* Target position of the arm */
        private ArmSpeed speed;                     /* Desired speed of the arm */
        private static ArmSpeed defaultSpeed;       /* Default arm speed */
        private bool paused;                        /* Is the robot paused? */
        private bool resetHomePositionFlag;         /* Reset the home position on next write? */
        private PathExecutor pathExecutor;          /* Path executor */

        static Robot() {
            double degreesPerRadian = 180.0 / Math.PI;
            defaultSpeed = new ArmSpeed(
                10.00 / degreesPerRadian,
                10.00 / degreesPerRadian,
                10.00 / degreesPerRadian,
                10.00 / degreesPerRadian);
        }

        /** Creates a new robot instance. */
        public Robot(IntPtr windowHandle) {
            armLength1 = 82;    // (default arm 1 length, from pivot to pivot, in centimeters)
            armLength2 = 90;    // (default arm 2 length, from pivot to pivot, in centimeters)

            currentPosition = ArmPosition.Zero;
            targetPosition = ArmPosition.Zero;
            speed = defaultSpeed;

            // Set up default motor parameters
            double degreesPerRadian = 180.0 / Math.PI;
            motorParameters = new MotorParameters[] {
                new MotorParameters(17.03 * degreesPerRadian),
                new MotorParameters(17.03 * degreesPerRadian),
                new MotorParameters(11.10 * degreesPerRadian),
                new MotorParameters(17.03 * degreesPerRadian),
            };

            pathExecutor = new PathExecutor(this);
            device = new Device();
            device.ConnectionStatusChanged += device_ConnectionStatusChanged;
            device.StartAutoConnect(windowHandle);
        }

        ~Robot() {
            Dispose(false);
        }

        /** Releases all resources used by the robot, and allows the
         *  application to quit.
         */
        public void Dispose() {
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing) {
            if (disposing) {
                if (pathExecutor != null) { pathExecutor.Dispose(); pathExecutor = null; }
                if (device != null) { device.Dispose(); device = null; }
            }
        }

        /** Call this to process windows messages that affect device status,
         *  such as WM_DEVICECHANGE.
         */
        public void HandleWindowsMessage(ref Message m) {
            device.HandleWindowsMessage(ref m);
        }

        /** Gets whether or not the robot device is currently connected.  Connection to
         *  the device is automatic; there's no need to manually connect.
         */
        public bool Connected {
            get { return device.Connected; }
        }

        /** Gets or sets the current position of the robot's arm.  This value is
         *  frequently overwritten with data read from the device, and setting
         *  it will have no effect.  Set #TargetPosition and #Speed
         *  instead to make the robot's arm move.  You may also have to call
         *  UpdateFromRobot() before reading this property to obtain the latest
         *  position from the device.
         */
        public ArmPosition CurrentPosition {
            get { return currentPosition; }
            set {
                if (value == null) { throw new ArgumentNullException("value"); }
                currentPosition = value;
            }
        }

        /** Gets or sets the target position of the robot's arm */
        public ArmPosition TargetPosition {
            get { return targetPosition; }
            set {
                if (value == null) { throw new ArgumentNullException("value"); }
                if (targetPosition != value) {
                    targetPosition = value;
                    UpdateFromDevice();     // (queues the new position to be written)
                    OnTargetPositionChanged();
                }
            }
        }

        /** Gets or sets the desired speed of the robot's arm */
        public ArmSpeed Speed {
            get { return speed; }
            set {
                if (value == null) { throw new ArgumentNullException("value"); }
                if (speed != value) {
                    speed = value;
                    UpdateFromDevice();     // (queues the new speed to be written)
                    OnSpeedChanged();
                }
            }
        }

        /** Gets the robot's default speed */
        public static ArmSpeed DefaultSpeed {
            get { return defaultSpeed; }
        }

        /** Gets or sets whether the robot is currently in a paused state */
        public bool Paused {
            get { return paused; }
            set {
                paused = value;
                UpdateFromDevice();     // (queues the new pause state to be written)
                OnPausedChanged();
            }
        }

        /** Gets or sets the length of the first arm segment (from pivot to
         *  pivot), measured in centimeters.
         *  
         *  You should generally only need to change this value if you replace
         *  the arm segment with a custom-sized one.
         */
        public double ArmLength1 {
            get { return armLength1; }
            set { armLength1 = value; }
        }

        /** Gets or sets the length of the second arm segment (from pivot to
         *  pivot), measured in centimeters.
         *  
         *  You should generally only need to change this value if you replace
         *  the arm segment with a custom-sized one.
         */
        public double ArmLength2 {
            get { return armLength2; }
            set { armLength2 = value; }
        }

        /** Flag indicating whether the home position should be reset on the
         *  next write.  Call SetHomePosition() to set this.
         */
        internal bool ResetHomePositionFlag {
            get { return resetHomePositionFlag; }
            set { resetHomePositionFlag = value; }
        }

        /** Specifies the path that the robot should execute.  Note that the
         *  target position and speed cannot be updated manually while a path
         *  is executing.  Set this property to null to have the robot stop
         *  executing the current path.
         */
        public Path Path {
            get { return pathExecutor.Path; }
            set {
                pathExecutor.Path = value;
                OnPathChanged();
            }
        }

        /** Updates the robot's CurrentPosition property from the device */
        public void UpdateFromDevice() {
            device.UpdateRobot(this);
        }

        /** Returns the parameters for the specified motor (1 through 4) */
        public MotorParameters GetMotorParameters(int motorNumber) {
            if (motorNumber < 1 || motorNumber > motorParameters.Length) {
                throw new ArgumentOutOfRangeException("motorNumber", motorNumber, "Motor number is out of range");
            }
            return motorParameters[motorNumber - 1];
        }

        /** Converts a step count into an angle (in radians) for the given motor
         *  (1 through 4).  Does NOT round the step count to a whole number.
         */
        public double ConvertStepsToAngle(double steps, int motorNumber) {
            return steps / GetMotorParameters(motorNumber).StepsPerRadian;
        }

        /** Converts an angle (in radians) into a step count for the given motor
         *  (1 through 4).  Does NOT round the resulting step count to a whole
         *  number.
         */
        public double ConvertAngleToSteps(double angle, int motorNumber) {
            return angle * GetMotorParameters(motorNumber).StepsPerRadian;
        }

        /** Sets the current position as the new home position / zero position */
        public void SetHomePosition() {
            resetHomePositionFlag = true;
        }

        /** Returns true if each motor of the robot is within the given number
         *  of steps of its target position (not rounded).
         */
        public bool IsWithinThresholdOfTarget(double steps) {
            if (steps < 0) { throw new ArgumentOutOfRangeException("steps", steps, "Argument out of range"); }
            return IsMotorWithinThresholdOfTarget(1, steps) &&
                IsMotorWithinThresholdOfTarget(2, steps) &&
                IsMotorWithinThresholdOfTarget(3, steps) &&
                IsMotorWithinThresholdOfTarget(4, steps);
        }

        /** Returns true if the given motor (1 - 4) is within the given number
         *  of steps of the target position (not rounded).
         */
        public bool IsMotorWithinThresholdOfTarget(int motor, double steps) {
            if (steps < 0) { throw new ArgumentOutOfRangeException("steps", steps, "Argument out of range"); }
            if (motor < 1 || motor > 4) { throw new ArgumentOutOfRangeException("motor", motor, "Argument out of ragne"); }
            double currentSteps = ConvertAngleToSteps(currentPosition.GetComponent(motor - 1, CoordinateSystem.Dof, this), motor);
            double targetSteps = ConvertAngleToSteps(targetPosition.GetComponent(motor - 1, CoordinateSystem.Dof, this), motor);
            return Math.Abs(targetSteps - currentSteps) < steps;
        }

        /** Called when the connection status of the device changes */
        private void device_ConnectionStatusChanged(object sender, EventArgs e) {
            Paused = true;  // (always pause the robot after connection change)
            OnConnectionStatusChanged();
        }

        /** Fired when the connection status has changed */
        public event EventHandler ConnectionStatusChanged;

        /** Raises the ConnectionStatusChanged event */
        private void OnConnectionStatusChanged() {
            if (ConnectionStatusChanged != null) {
                ConnectionStatusChanged(this, EventArgs.Empty);
            }
        }

        /** Fired when the TargetPosition property changes */
        public event EventHandler TargetPositionChanged;
        
        /** Fires the TargetPositionChanged event */
        private void OnTargetPositionChanged() {
            if (TargetPositionChanged != null) {
                TargetPositionChanged(this, EventArgs.Empty);
            }
        }

        /** Fired when the Paused property changes */
        public event EventHandler PausedChanged;
        
        /** Fires the PausedChanged event */
        private void OnPausedChanged() {
            if (PausedChanged != null) {
                PausedChanged(this, EventArgs.Empty);
            }
        }

        /** Fired when the Speed property changes */
        public event EventHandler SpeedChanged;
        
        /** Fires the SpeedChanged event */
        private void OnSpeedChanged(){
            if (SpeedChanged != null) {
                SpeedChanged(this, EventArgs.Empty);
            }
        }

        /** Fired when the Path property changes */
        public event EventHandler PathChanged;
        
        /** Fires the PathChanged event */
        private void OnPathChanged() {
            if (PathChanged != null) {
                PathChanged(this, EventArgs.Empty);
            }
        }
    }
}
