﻿/*=========================================================================
   This file is part of the Cardboard Robot SDK.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Diagnostics;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using Microsoft.Win32.SafeHandles;

namespace CBRobot {

    /* A high-level connection to the Cardboard Robot USB device. */
    internal class Device : IDisposable {

        private const int AutoconnectInterval = 5000;   /* Milliseconds between autoconnect attempts */

        private DeviceConnection connection;            /* Low-level connection to the device */
        private bool connected;                         /* Local connected state */
        private Timer autoConnectTimer;                 /* Autoconnect timer, in case device notifications don't work */
        private DevNotifySafeHandle devNotifyHandle;    /* HDEVNOTIFY handle, granting us WM_DEVICECHANGE messages */
        private IntPtr windowHandle;                    /* Handle of the window receiving device notifications */

        #region Safe Handles
        /** Safe (auto-releasing) HDEVNOTIFY handle */
        private class DevNotifySafeHandle : SafeHandleZeroOrMinusOneIsInvalid {
            public DevNotifySafeHandle() : base(true) { }
            protected override bool ReleaseHandle() {
                return UnregisterDeviceNotification(handle);
            }
        }
        #endregion Safe Handles

        #region Imports
        /* ---- Imported functions ---- */
        [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern DevNotifySafeHandle RegisterDeviceNotification(   /* HDEVNOTIFY */
            IntPtr hRecipient,                                      /* [in] HANDLE */
            ref DEV_BROADCAST_DEVICEINTERFACE NotificationFilter,   /* [in] LPVOID */
            uint Flags);                                            /* [in] DWORD */

        [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
        private static extern bool UnregisterDeviceNotification(    /* BOOL */
            IntPtr Handle);     /* [in] HDEVNOTIFY */

        /* ---- Imported structures ---- */
        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode, Pack = 4)]
        private struct DEV_BROADCAST_DEVICEINTERFACE {
            public uint dbcc_size;          /* DWORD */
            public uint dbcc_devicetype;    /* DWORD */
            public uint dbcc_reserved;      /* DWORD */
            public Guid dbcc_classguid;     /* GUID */
            public char dbcc_name;          /* TCHAR array (struct can vary in size - this is a placeholder) */
        }   /* size is 32 on both x86 and x64; Pack = 4 achieves this */

        /* ---- Imported constants ---- */
        private const uint DBT_DEVTYP_DEVICEINTERFACE = 0x05;   /* for RegisterDeviceNotification() */
        private const uint DEVICE_NOTIFY_WINDOW_HANDLE = 0x00;  /* for RegisterDeviceNotification() */
        private const uint WM_DEVICECHANGE = 0x0219;            /* WIN32 message */
        private const uint DBT_DEVICEARRIVAL = 0x8000;          /* wParam value for WM_DEVICECHANGE */
        private const uint DBT_DEVICEREMOVECOMPLETE = 0x8004;   /* wParam value for WM_DEVICECHANGE */

        /* GUID for HID class devices */
        private static Guid HIDClassGuid = new Guid(0x4d1e55b2, 0xf16f, 0x11cf, 0x88, 0xcb, 0x00, 0x11, 0x11, 0x00, 0x00, 0x30);

        /* ---- End of Imports ---- */
        #endregion Imports

        public Device() {
            connection = new DeviceConnection();
        }

        ~Device() {
            Dispose(false);
        }
        
        public void Dispose() {
            Dispose(true);
            GC.SuppressFinalize(this);
        }
        
        protected virtual void Dispose(bool disposing) {
            StopAutoConnect();
            Disconnect();
            if (disposing) {
                connection.Dispose();
                connection = null;
            }
        }

        /** Gets whether the device is currently connected */
        public bool Connected {
            get { return connected; }   // DO NOT RETURN connection.Connected (or notifications won't match state)
            private set {
                if (connected != value) {
                    connected = value;
                    OnConnectionStatusChanged();
                }
            }
        }

        /** Fired when the connection status has changed */
        public event EventHandler ConnectionStatusChanged;

        /** Raises the ConnectionStatusChanged event */
        private void OnConnectionStatusChanged() {
            if (ConnectionStatusChanged != null) {
                ConnectionStatusChanged(this, EventArgs.Empty);
            }
        }

        /** Attempts to connect to the device, if not already connected */
        public void TryConnect() {
            if (!connection.Connected) {
                connection.TryConnect();
            }
            Connected = connection.Connected;
        }

        /** Disconnects from the device immediately (doesn't stop auto-reconnection) */
        public void Disconnect() {
            if (connection.Connected) {
                connection.Disconnect();
            }
            Connected = connection.Connected;
        }

        /** Attempts to re-send data to the device to make sure it's still connected.
         *  Disconnects from the device if an error occurs (sets Connected to false).
         */
        public void TestConnection() {
            if (connection.Connected) {
                connection.TestConnection();
            }
            Connected = connection.Connected;
        }

        /** Begins auto-connecting to the device */
        public void StartAutoConnect(IntPtr windowHandle) {
            Debug.Assert(devNotifyHandle == null || windowHandle == this.windowHandle, "Auto-connect was already started using a different window handle");
            if (devNotifyHandle != null) { return; }

            // Register for add / remove notification.
            RegisterForDeviceNotifications(windowHandle);

            // Start the auto-connect timer.
            autoConnectTimer = new Timer();
            autoConnectTimer.Interval = AutoconnectInterval;
            autoConnectTimer.Tick += AutoconnectTimer_Tick;
            autoConnectTimer.Enabled = true;

            // Try connecting now, since we won't receive notification if the device is already connected.
            TryConnect();
        }

        /** Stops auto-connecting to the device */
        public void StopAutoConnect() {
            if (autoConnectTimer != null) {
                autoConnectTimer.Dispose(); autoConnectTimer = null;
            }
            if (devNotifyHandle != null) {
                devNotifyHandle.Close(); devNotifyHandle = null;
            }
            windowHandle = IntPtr.Zero;
        }

        /** Registers the given window to receive device connect / disconnect
         *  messages, and then attempts an initial connection to the device (since
         *  a message won't be sent if the device is already plugged in).
         */
        private void RegisterForDeviceNotifications(IntPtr windowHandle) {
            
            if (windowHandle != IntPtr.Zero) {
                // Register for WM_DEVICECHANGE notifications
                DEV_BROADCAST_DEVICEINTERFACE devInterface = new DEV_BROADCAST_DEVICEINTERFACE();
                devInterface.dbcc_devicetype = DBT_DEVTYP_DEVICEINTERFACE;
                devInterface.dbcc_size = (uint)Marshal.SizeOf(devInterface);
                devInterface.dbcc_classguid = HIDClassGuid;

                devNotifyHandle = RegisterDeviceNotification(windowHandle, ref devInterface, DEVICE_NOTIFY_WINDOW_HANDLE);
                if (devNotifyHandle.IsInvalid) {
                    Debug.Fail("Failed to register for device notifications");
                }
            }
            this.windowHandle = windowHandle;
        }

        /** Called when the auto-connect timer elapses */
        public void AutoconnectTimer_Tick(object sender, EventArgs e) {
            
            // Test for connection or disconnection.
            if (!connection.Connected) {
                TryConnect();
            }
            else {
                TestConnection();
            }
        }

        /** Call this to process windows messages that affect device status,
         *  such as WM_DEVICECHANGE.
         */
        public void HandleWindowsMessage(ref Message m) {

            // Detect device connection / disconnection
            if (m.HWnd == windowHandle && m.Msg == WM_DEVICECHANGE) {
                if ((uint)m.WParam == DBT_DEVICEARRIVAL) {
                    TryConnect();
                }
                else if ((uint)m.WParam == DBT_DEVICEREMOVECOMPLETE) {
                    TestConnection();
                }
            }
        }

        /** @copydoc DeviceConnection.UpdateRobot() */
        public void UpdateRobot(Robot robot) {
            connection.UpdateRobot(robot);
            Connected = connection.Connected;   // (in case it changed)
        }
    }
}
