﻿/*=========================================================================
   This file is part of the Cardboard Robot SDK.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Collections.Generic;
using System.Text;

namespace CBRobot {

    /** Represents the speed of the robot arm */
    public class ArmSpeed {

        private double m1Speed;
        private double m2Speed;
        private double m3Speed;
        private double m4Speed;
        private static ArmSpeed zero;

        static ArmSpeed() {
            zero = new ArmSpeed(0, 0, 0, 0);
        }

        /** Creates a new ArmSpeed using the specified speed for each motor
         *  (in radians per second).
         */
        public ArmSpeed(double m1Speed, double m2Speed, double m3Speed, double m4Speed) {
            this.m1Speed = m1Speed;
            this.m2Speed = m2Speed;
            this.m3Speed = m3Speed;
            this.m4Speed = m4Speed;
        }

        /** Gets the speed of motor #1, in radians per second */
        public double M1Speed {
            get { return m1Speed; }
            // (intentionally read-only)
        }

        /** Gets the speed of motor #2, in radians per second */
        public double M2Speed {
            get { return m2Speed; }
        }

        /** Gets the speed of motor #3, in radians per second */
        public double M3Speed {
            get { return m3Speed; }
        }

        /** Gets the speed of motor #4, in radians per second */
        public double M4Speed {
            get { return m4Speed; }
        }

        /** Returns an arm speed with all components set to zero */
        public static ArmSpeed Zero {
            get { return zero; }
        }

        /** Returns a string description of this ArmSpeed object */
        public override string ToString() {
            return String.Format("({0:N6}, {1:N6}, {2:N6}, {3:N6})", m1Speed, m2Speed, m3Speed, m4Speed);
        }

        /** Returns a new arm speed with the specified component (0 - 3) set to
         *  the given value (in radians per second).
         */
        public ArmSpeed SetComponent(int componentIndex, double value) {
            switch (componentIndex) {
                case 0:
                    return new ArmSpeed(value, m2Speed, m3Speed, m4Speed);
                case 1:
                    return new ArmSpeed(m1Speed, value, m3Speed, m4Speed);
                case 2:
                    return new ArmSpeed(m1Speed, m2Speed, value, m4Speed);
                case 3:
                    return new ArmSpeed(m1Speed, m2Speed, m3Speed, value);
                default:
                    throw new ArgumentOutOfRangeException("componentIndex", componentIndex,
                        "Component index is out of range (0 - 3)");
            }
        }

        /** Returns the value (in radians per second) of the speed component
         *  with the given index (0 - 3).
         */
        public double GetComponent(int componentIndex) {
            switch (componentIndex) {
                case 0:
                    return m1Speed;
                case 1:
                    return m2Speed;
                case 2:
                    return m3Speed;
                case 3:
                    return m4Speed;
                default:
                    throw new ArgumentOutOfRangeException("componentIndex", componentIndex,
                        "Component index is out of range (0 - 3)");
            }
        }
    }
}
