﻿/*=========================================================================
   This file is part of the Cardboard Robot SDK.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;

namespace CBRobot {

    /** Represents a position of the robot arm, which is composed of a tip
     *  position plus the position of motor #4.
     */
    public class ArmPosition {
        
        private Vector tipPosition;
        private double m4;
        private static ArmPosition zero;

        static ArmPosition() {
            zero = new ArmPosition(DofVector.Zero, 0.0);
        }

        /** Creates a new ArmPosition with the specified tip position and
         *  motor #4 position (in radians).
         */
        public ArmPosition(Vector tipPosition, double m4) {
            this.tipPosition = tipPosition;
            this.m4 = m4;
        }

        /** Gets the position of the arm's tip (corresponds to motors 1 through 3) */
        public Vector TipPosition {
            get { return tipPosition; }
            // (intentionally read-only)
        }

        /** Gets the position of the fourth motor (crane, camera swivel), in radians */
        public double M4 {
            get { return m4; }
            // (intentionally read-only)
        }

        /** Returns an ArmPosition which represents the home position (all DOF components are zero) */
        public static ArmPosition Zero {
            get { return zero; }
        }

        /** Returns a string description of this position */
        public override string ToString() {
            return String.Format("(tip:{0}, m4:{1:N6})", tipPosition, m4);
        }

        /** Returns a new arm position with the specified vector
         *  component (0 - 3) to the given value, doing coordinate system
         *  conversion as needed.
         */
        public ArmPosition SetComponent(int componentIndex, double value, CoordinateSystem cs, Robot robot) {
            switch (cs) {
                case CoordinateSystem.Dof:
                    switch (componentIndex) {
                        case 0:
                        case 1:
                        case 2:
                            Vector newTipPosition = tipPosition.ConvertToDofPoint(robot).SetComponent(componentIndex, value);
                            return new ArmPosition(newTipPosition, m4);
                        case 3:
                            return new ArmPosition(tipPosition, value);
                        default:
                            throw new ArgumentOutOfRangeException("componentIndex", componentIndex,
                                "Component index was out of range (0 - 3)");
                    }
                default:
                    throw new ArgumentException("Unsupported coordinate system", "cs");
            }
        }

        /** Gets the specified vector component (0 - 3), doing coordinate
         *  system conversion as needed.
         */
        public double GetComponent(int componentIndex, CoordinateSystem cs, Robot robot) {
            switch (cs) {
                case CoordinateSystem.Dof:
                    switch (componentIndex) {
                        case 0:
                        case 1:
                        case 2:
                            return tipPosition.ConvertToDofPoint(robot).GetComponent(componentIndex);
                        case 3:
                            return m4;
                        default:
                            throw new ArgumentOutOfRangeException("componentIndex", componentIndex,
                                "Component index was out of range (0 - 3)");
                    }
                default:
                    throw new ArgumentException("Unsupported coordinate system", "cs");
            }
        }
    }
}
