﻿/*=========================================================================
   This file is part of the Cardboard Robot Console application.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Diagnostics;
using CBRobot;

namespace CBRobotConsole {

    /** Breaks a speed value into its components, doing unit conversion along
     *  the way.  The components are read/write, and changes to them will
     *  affect the combined speed value.
     */
    public class SpeedTransformer {

        private Robot robot;
        private Unit dofUnit;
        private ArmSpeed speed;
        private double component1;
        private double component2;
        private double component3;
        private double component4;
        private Unit unit1;
        private Unit unit2;
        private Unit unit3;
        private Unit unit4;

        public SpeedTransformer() {
            robot = MainWindow.Instance.Robot;
            dofUnit = Unit.Radians;
            speed = ArmSpeed.Zero;
            UpdateUnits();
            UpdateComponents();
        }

        /** Gets the robot associated with this transformer */
        public Robot Robot {
            get { return robot; }
        }

        /** Gets or sets the unit for DOF components of the transformed speed */
        public Unit DofUnit {
            get { return dofUnit; }
            set {
                if (dofUnit != value) {
                    dofUnit = value;
                    UpdateUnits();
                    UpdateComponents();
                    OnUnitChanged();
                    OnComponentChanged();
                    // (changes components, but does not change the actual speed)
                }
            }
        }

        /** Component 1 of the transformed speed */
        public double Component1 {
            get { return component1; }
            set {
                if (component1 != value) {
                    component1 = value;
                    UpdateSpeedComponent(0, value);
                    OnComponentChanged();
                    OnSpeedChanged();
                }
            }
        }

        /** Component 2 of the transformed speed */
        public double Component2 {
            get { return component2; }
            set {
                if (component2 != value) {
                    component2 = value;
                    UpdateSpeedComponent(1, value);
                    OnComponentChanged();
                    OnSpeedChanged();
                }
            }
        }

        /** Component 3 of the transformed speed */
        public double Component3 {
            get { return component3; }
            set {
                if (component3 != value) {
                    component3 = value;
                    UpdateSpeedComponent(2, value);
                    OnComponentChanged();
                    OnSpeedChanged();
                }
            }
        }

        /** Component 4 of the transformed speed */
        public double Component4 {
            get { return component4; }
            set {
                if (component4 != value) {
                    component4 = value;
                    UpdateSpeedComponent(3, value);
                    OnComponentChanged();
                    OnSpeedChanged();
                }
            }
        }

        /** Gets component 1, or sets all components.  Asserts in debug mode if
         *  the components aren't actually uniform.
         */
        public double UniformComponent {
            get {
                // (note: you may get this assert when switching to "steps" unit, because of differing conversion factors for each motor)
                Debug.Assert(component2 == component1 && component3 == component1 && component4 == component1, @"Uniform component being read, but components are not uniform");
                return Component1;
            }
            set {
                Debug.Assert(component2 == component1 && component3 == component1 && component4 == component1, @"Uniform component being written, but components are not uniform");
                Component1 = value;
                Component2 = value;
                Component3 = value;
                Component4 = value;
            }
        }

        /** Unit for component 1 of the transformed speed */
        public Unit Unit1 {
            get { return unit1; }
        }

        /** Unit for component 2 of the transformed speed */
        public Unit Unit2 {
            get { return unit2; }
        }

        /** Unit for component 3 of the transformed speed */
        public Unit Unit3 {
            get { return unit3; }
        }

        /** Unit for component 4 of the transformed speed */
        public Unit Unit4 {
            get { return unit4; }
        }

        /** Gets or sets the speed to be transformed / broken into components */
        public ArmSpeed Speed {
            get { return speed; }
            set {
                if (speed != value) {
                    speed = value;
                    UpdateComponents();
                    OnComponentChanged();
                    OnSpeedChanged();
                }
            }
        }

        /** Sets the specified component (0 - 3) to the given value */
        public void SetComponent(int componentIndex, double value) {
            switch (componentIndex) {
                case 0:
                    Component1 = value;
                    break;
                case 1:
                    Component2 = value;
                    break;
                case 2:
                    Component3 = value;
                    break;
                case 3:
                    Component4 = value;
                    break;
                default:
                    throw new ArgumentOutOfRangeException("componentIndex", componentIndex,
                        "Component number is out of range (0 - 3)");
            }
        }

        /** Gets the value of the specified component (0 - 3) */
        public double GetComponent(int componentIndex) {
            switch (componentIndex) {
                case 0:
                    return Component1;
                case 1:
                    return Component2;
                case 2:
                    return Component3;
                case 3:
                    return Component4;
                default:
                    throw new ArgumentOutOfRangeException("componentIndex", componentIndex,
                        "Component number is out of range (0 - 3)");
            }
        }

        /** Fired when one or more of the units changes */
        public event EventHandler UnitChanged;

        /** Fires the UnitChanged event */
        private void OnUnitChanged() {
            if (UnitChanged != null) {
                UnitChanged(this, EventArgs.Empty);
            }
        }

        /** Fired when the speed represented by this transformer changes */
        public event EventHandler SpeedChanged;
        
        /** Fires the SpeedChanged event */
        private void OnSpeedChanged()
        {
            if (SpeedChanged != null) {
                SpeedChanged(this, EventArgs.Empty);
            }
        }

        /** Fired when the value of one of the components changes */
        public event EventHandler ComponentChanged;
        
        /** Fires the ComponentChanged event */
        private void OnComponentChanged()
        {
            if (ComponentChanged != null) {
                ComponentChanged(this, EventArgs.Empty);
            }
        }

        /** Updates the Unit1, Unit2, etc. properties based on the current
         *  parameters.
         */
        private void UpdateUnits() {
            unit1 = dofUnit;
            unit2 = dofUnit;
            unit3 = dofUnit;
            unit4 = dofUnit;
        }

        /** Updates the components from the current speed & unit setting */
        private void UpdateComponents() {
            if (speed != null) {
                component1 = speed.M1Speed * GetDofConversionFactor(1);
                component2 = speed.M2Speed * GetDofConversionFactor(2);
                component3 = speed.M3Speed * GetDofConversionFactor(3);
                component4 = speed.M4Speed * GetDofConversionFactor(4);
            }
        }

        /** Helper to set the specified component (0 - 3) of the speed
         *  vector, w/ conversion.
         */
        private void UpdateSpeedComponent(int componentIndex, double value) {
            Debug.Assert(speed != null, "No speed is set");
            if (speed != null) {
                value /= GetDofConversionFactor(componentIndex + 1);
                speed = speed.SetComponent(componentIndex, value);
            }
        }

        /** Returns the conversion factor (radians to target unit) for the specified motor */
        private double GetDofConversionFactor(int motorNumber) {
            switch (dofUnit) {
                case Unit.Radians:
                    return 1.0;
                case Unit.Degrees:
                    return 180.0 / Math.PI;
                case Unit.Steps:
                    return robot.ConvertAngleToSteps(1.0, motorNumber);
                default:
                    throw new InvalidOperationException(String.Format("Unknown unit: {0}", dofUnit));
            }
        }
    }
}
