﻿/*=========================================================================
   This file is part of the Cardboard Robot Console application.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Diagnostics;
using CBRobot;

namespace CBRobotConsole {

    /** Represents a single entry / row within a "saved program" */
    public class ProgramEntry {
        private UndoManager undoManager;
        private PositionTransformer positionTransformer;
        private ScalarSpeedTransformer speedTransformer;
        private double pause;

        public ProgramEntry(UndoManager undoManager) {
            Debug.Assert(undoManager != null);
            this.undoManager = undoManager;
            positionTransformer = new PositionTransformer();
            speedTransformer = new ScalarSpeedTransformer();
            speedTransformer.Speed = Robot.DefaultSpeed.M1Speed;
        }

        /** Returns the undo manager associated with this program entry */
        public UndoManager UndoManager {
            get { return undoManager; }
        }

        /** Returns the position transformer that stores the position value
         *  for this entry.
         */
        public PositionTransformer PositionTransformer {
            get { return positionTransformer; }
        }

        /** Returns the speed transformer that stores the speed value
         *  for this entry.
         */
        public ScalarSpeedTransformer SpeedTransformer {
            get { return speedTransformer; }
        }

        /** Returns the delay value for this entry */
        public double Pause {
            get { return pause; }
            set { pause = value; }
        }

        /** Fired when any value on this entry is changed */
        public event EventHandler ValueChanged;
        
        /** Fires the ValueChanged event */
        private void OnValueChanged() {
            if (ValueChanged != null) {
                ValueChanged(this, EventArgs.Empty);
            }
        }

        /** Sets the specified position component (w/ undo support) */
        public void SetPositionComponent(int componentIndex, double value) {
            // (note: do not simply undo the component change - we need to
            //  perform the undo in terms of unit-independent values)
            PositionTransformer p = new PositionTransformer();
            p.DofUnit = positionTransformer.DofUnit;
            p.Position = PositionTransformer.Position;
            p.SetComponent(componentIndex, value);
            SetPositionImpl(p.Position, null);
        }

        private void SetPositionImpl(object arg1, object arg2) {
            ArmPosition newPos = (ArmPosition)arg1;

            ArmPosition oldPos = positionTransformer.Position;
            positionTransformer.Position = newPos;
            undoManager.AddInverseOperation("Change Value", SetPositionImpl, oldPos, null);

            OnValueChanged();
        }

        /** Sets the speed (w/ undo support) */
        public void SetSpeed(double value) {
            ScalarSpeedTransformer s = new ScalarSpeedTransformer();
            s.DofUnit = speedTransformer.DofUnit;
            s.TransformedSpeed = value;
            SetSpeedImpl(s.Speed, null);
        }

        private void SetSpeedImpl(object arg1, object arg2) {
            double newSpeed = (double)arg1;     // (note: this is the un-transformed speed!)

            double oldSpeed = speedTransformer.Speed;
            speedTransformer.Speed = newSpeed;
            undoManager.AddInverseOperation("Change Value", SetSpeedImpl, oldSpeed, null);

            OnValueChanged();
        }

        /** Sets the pause value (w/ undo support) */
        public void SetPause(double value) {
            SetPauseImpl(value, null);
        }

        private void SetPauseImpl(object arg1, object arg2) {
            double value = (double)arg1;

            double oldValue = pause;
            pause = value;
            undoManager.AddInverseOperation("Change Value", SetPauseImpl, oldValue, null);

            OnValueChanged();
        }
    }
}
