﻿/*=========================================================================
   This file is part of the Cardboard Robot Console application.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.Diagnostics;
using CBRobot;

namespace CBRobotConsole {

    /** Breaks a position value into its components, doing unit conversion
     *  along the way.  The components are read/write, and changes to them
     *  will affect the combined position value.
     */
    public class PositionTransformer {
        
        private Robot robot;
        private CoordinateSystem coordinateSystem;
        private Unit dofUnit;
        private ArmPosition position;
        private double component1;
        private double component2;
        private double component3;
        private double component4;
        private Unit unit1;
        private Unit unit2;
        private Unit unit3;
        private Unit unit4;

        public PositionTransformer() {
            robot = MainWindow.Instance.Robot;
            dofUnit = Unit.Radians;
            coordinateSystem = CoordinateSystem.Dof;
            position = ArmPosition.Zero;
            UpdateUnits();
            UpdateComponents();
        }

        /** Gets the robot associated with this transformer */
        public Robot Robot {
            get { return robot; }
        }

        /** Gets or sets the unit for DOF components of the transformed position */
        public Unit DofUnit {
            get { return dofUnit; }
            set {
                if (dofUnit != value) {
                    dofUnit = value;
                    UpdateUnits();
                    UpdateComponents();
                    OnUnitChanged();
                    OnComponentChanged();
                    // (changes components, but does not change the actual position)
                }
            }
        }

        /** Component 1 of the transformed position */
        public double Component1 {
            get { return component1; }
            set {
                if (component1 != value) {
                    component1 = value;
                    UpdatePositionComponent(0, value);
                    OnComponentChanged();
                    OnPositionChanged();
                }
            }
        }

        /** Component 2 of the transformed position */
        public double Component2 {
            get { return component2; }
            set {
                if (component2 != value) {
                    component2 = value;
                    UpdatePositionComponent(1, value);
                    OnComponentChanged();
                    OnPositionChanged();
                }
            }
        }

        /** Component 3 of the transformed position */
        public double Component3 {
            get { return component3; }
            set {
                if (component3 != value) {
                    component3 = value;
                    UpdatePositionComponent(2, value);
                    OnComponentChanged();
                    OnPositionChanged();
                }
            }
        }

        /** Component 4 of the transformed position */
        public double Component4 {
            get { return component4; }
            set {
                if (component4 != value) {
                    component4 = value;
                    UpdatePositionComponent(3, value);
                    OnComponentChanged();
                    OnPositionChanged();
                }
            }
        }

        /** Unit for component 1 of the transformed position */
        public Unit Unit1 {
            get { return unit1; }
        }

        /** Unit for component 2 of the transformed position */
        public Unit Unit2 {
            get { return unit2; }
        }

        /** Unit for component 3 of the transformed position */
        public Unit Unit3 {
            get { return unit3; }
        }

        /** Unit for component 4 of the transformed position */
        public Unit Unit4 {
            get { return unit4; }
        }

        /** Gets or sets the position to be transformed / broken into components */
        public ArmPosition Position {
            get { return position; }
            set {
                if (position != value) {
                    position = value;
                    UpdateComponents();
                    OnComponentChanged();
                    OnPositionChanged();
                }
            }
        }

        /** Sets the specified component (0 - 3) to the given value */
        public void SetComponent(int componentIndex, double value) {
            switch (componentIndex) {
                case 0:
                    Component1 = value;
                    break;
                case 1:
                    Component2 = value;
                    break;
                case 2:
                    Component3 = value;
                    break;
                case 3:
                    Component4 = value;
                    break;
                default:
                    throw new ArgumentOutOfRangeException("componentIndex", componentIndex,
                        "Component number is out of range (0 - 3)");
            }
        }

        /** Gets the value of the specified component (0 - 3) */
        public double GetComponent(int componentIndex) {
            switch (componentIndex) {
                case 0:
                    return Component1;
                case 1:
                    return Component2;
                case 2:
                    return Component3;
                case 3:
                    return Component4;
                default:
                    throw new ArgumentOutOfRangeException("componentIndex", componentIndex,
                        "Component number is out of range (0 - 3)");
            }
        }

        /** Fired when one or more of the units changes */
        public event EventHandler UnitChanged;

        /** Fires the UnitChanged event */
        private void OnUnitChanged() {
            if (UnitChanged != null) {
                UnitChanged(this, EventArgs.Empty);
            }
        }

        /** Fired when the position represented by this transformer changes */
        public event EventHandler PositionChanged;
        
        /** Fires the PositionChanged event */
        private void OnPositionChanged()
        {
            if (PositionChanged != null) {
                PositionChanged(this, EventArgs.Empty);
            }
        }

        /** Fired when the value of one of the components changes */
        public event EventHandler ComponentChanged;
        
        /** Fires the ComponentChanged event */
        private void OnComponentChanged()
        {
            if (ComponentChanged != null) {
                ComponentChanged(this, EventArgs.Empty);
            }
        }

        /** Updates the Unit1, Unit2, etc. properties based on the current
         *  parameters.
         */
        private void UpdateUnits() {
            switch (coordinateSystem) {
                case CoordinateSystem.Dof:
                    unit1 = dofUnit;
                    unit2 = dofUnit;
                    unit3 = dofUnit;
                    unit4 = dofUnit;
                    break;
                default:
                    Debug.Fail("Unknown coordinate system");
                    return;
            }
        }

        /** Updates the components from the current position & unit setting */
        private void UpdateComponents() {
            if (position != null) {
                switch (coordinateSystem) {
                    case CoordinateSystem.Dof:
                        DofVector v = position.TipPosition.ConvertToDofPoint(robot);
                        component1 = v.M1 * GetDofConversionFactor(1);
                        component2 = v.M2 * GetDofConversionFactor(2);
                        component3 = v.M3 * GetDofConversionFactor(3);
                        component4 = position.M4 * GetDofConversionFactor(4);
                        break;
                    default:
                        Debug.Fail("Unknown coordinate system");
                        return;
                }
            }
        }

        /** Helper to set the specified component (0 - 3) of the position
         *  vector, w/ conversion.
         */
        private void UpdatePositionComponent(int componentIndex, double value) {
            Debug.Assert(position != null, "No position is set");
            if (position != null) {
                value /= GetDofConversionFactor(componentIndex + 1);
                position = position.SetComponent(componentIndex, value, coordinateSystem, robot);
            }
        }

        /** Returns the conversion factor (radians to target unit) for the specified motor */
        private double GetDofConversionFactor(int motorNumber) {
            switch (dofUnit) {
                case Unit.Radians:
                    return 1.0;
                case Unit.Degrees:
                    return 180.0 / Math.PI;
                case Unit.Steps:
                    return robot.ConvertAngleToSteps(1.0, motorNumber);
                default:
                    throw new InvalidOperationException(String.Format("Unknown unit: {0}", dofUnit));
            }
        }
    }
}
