﻿/*=========================================================================
   This file is part of the Cardboard Robot Console application.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;

namespace CBRobotConsole {

    /** Simple button implementation which displays an image instead of text */
    public class ImageButton : Button {

        private bool isDesigning;
        private bool mouseOver = false;
        private bool mousePressed = false;
        private Image normalImage;
        private Image hoverImage;
        private Image pressedImage;

        public ImageButton() {
            isDesigning = LicenseManager.UsageMode == LicenseUsageMode.Designtime;

            if (!isDesigning) {
                Text = "";
                
                // Remove the border
                FlatStyle = FlatStyle.Flat;
                FlatAppearance.BorderSize = 0;

                // Nullify the default button highlighting
                FlatAppearance.MouseOverBackColor = BackColor;
                FlatAppearance.MouseDownBackColor = BackColor;

                MouseEnter += ImageButton_MouseEnter;
                MouseLeave += ImageButton_MouseLeave;
                MouseDown += ImageButton_MouseDown;
                MouseUp += ImageButton_MouseUp;
            }
        }

        [DefaultValue(null)]
        public Image NormalImage {
            get { return normalImage; }
            set { normalImage = value; UpdateImage(); }
        }

        [DefaultValue(null)]
        public Image HoverImage {
            get { return hoverImage; }
            set { hoverImage = value; UpdateImage(); }
        }

        [DefaultValue(null)]
        public Image PressedImage {
            get { return pressedImage; }
            set { pressedImage = value; UpdateImage(); }
        }

        private void UpdateImage() {
            if (!isDesigning) {
                Image = (mousePressed ? pressedImage : (mouseOver ? hoverImage : normalImage)) ?? normalImage;
            }
        }

        private void ImageButton_MouseEnter(object sender, EventArgs e) {
            mouseOver = true;
            UpdateImage();
        }

        private void ImageButton_MouseLeave(object sender, EventArgs e) {
            mouseOver = false;
            UpdateImage();
        }

        private void ImageButton_MouseDown(object sender, MouseEventArgs e) {
            mousePressed = true;
            UpdateImage();
        }

        private void ImageButton_MouseUp(object sender, MouseEventArgs e) {
            mousePressed = false;
            UpdateImage();
        }
    }
}
