﻿/*=========================================================================
   This file is part of the Cardboard Robot Console application.
   
   Copyright (C) 2012 Ken Ihara.
  
   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

using System;
using System.ComponentModel;
using System.Windows.Forms;

namespace CBRobotConsole {

    /** Contains functionality for the large spinner buttons on the main form */
    class BigSpinnerButton : ImageButton {

        private const int SecondTickDelay = 250;    /** How many milliseconds between the 1st and 2nd tick? */
        private const int RepeatTickDelay = 100;    /** How many milliseconds between ticks after that? */

        private bool isDesigning;   /* Are we in the VS designer? (if so, don't set any properties!) */
        private bool direction;     /* false = up, true = down */
        private Timer timer;        /* Tick timer */

        public BigSpinnerButton() {
            isDesigning = LicenseManager.UsageMode == LicenseUsageMode.Designtime;
            MouseDown += BigSpinnerButton_MouseDown;
            MouseUp += BigSpinnerButton_MouseUp;
        }

        protected override void InitLayout() {
            if (!isDesigning) {
                UpdateImages();
            }
        }

        /** Changes the direction of the button - false is up, true is down */
        [DefaultValue(false)]
        public bool Direction {
            get { return direction; }
            set { direction = value; UpdateImages(); }
        }

        /** Fired each time the spinner button "ticks" */
        public event EventHandler Tick;
        
        /** Fires the Tick event */
        private void OnTick()
        {
            if (Tick != null) {
                Tick(this, EventArgs.Empty);
            }
        }

        /** Updates the images based on the current direction */
        private void UpdateImages() {
            if (!isDesigning) {
                if (direction) {
                    NormalImage = Properties.Resources.DownButton_Normal;
                    HoverImage = Properties.Resources.DownButton_Hover;
                    PressedImage = Properties.Resources.DownButton_Pressed;
                }
                else {
                    NormalImage = Properties.Resources.UpButton_Normal;
                    HoverImage = Properties.Resources.UpButton_Hover;
                    PressedImage = Properties.Resources.UpButton_Pressed;
                }
            }
        }

        /** Called when the user presses the mouse on the button */
        private void BigSpinnerButton_MouseDown(object sender, MouseEventArgs e) {
            if (timer != null) { timer.Dispose(); timer = null; }
            timer = new Timer();
            timer.Interval = SecondTickDelay;
            timer.Tick += timer_Tick;
            timer.Enabled = true;
            OnTick();
        }

        /** Called when the user releases the mouse on the button */
        private void BigSpinnerButton_MouseUp(object sender, MouseEventArgs e) {
            if (timer != null) { timer.Dispose(); timer = null; }
        }

        /** Called when the tick timer elapses */
        private void timer_Tick(object sender, EventArgs e) {
            if (timer != null) { timer.Interval = RepeatTickDelay; }
            OnTick();
        }
    }
}
