/*=========================================================================
   This file is part of the Cardboard Robot SDK.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBArmPosition.h"
#import "CBArmSpeed.h"
#import "CBRobot.h"
#import "CBDevice.h"
#import "CBDeviceDelegate.h"
#import "CBDofVector.h"
#import "CBMotorParameters.h"
#import "CBPath.h"
#import "CBPathExecutor.h"


/* Private members */
@interface CBRobot () <CBDeviceDelegate> {
    CBDevice *device;       /* Connection to the USB device */
    NSArray *motorParams;   /* Parameters for each motor */
    CBPathExecutor *pathExecutor;   /* Path executor */
}

@property (assign, nonatomic) BOOL resetHomePositionFlag;   // (not public)

- (void)connectionStatusDidChange;

@end


@implementation CBRobot

@synthesize armLength1;
@synthesize armLength2;
@synthesize currentPosition;
@synthesize targetPosition;
@synthesize speed;
@synthesize defaultSpeed;
@synthesize paused;
@synthesize path;
@synthesize resetHomePositionFlag;

- (void)dealloc {
    [device setDelegate:nil];
    [device stopAutoConnect];   // (allows device to be dealloc'd)
    [pathExecutor setPath:nil]; // (allows executor to be dealloc'd)
    
    [device release]; device = nil;
    [motorParams release]; motorParams = nil;
    [pathExecutor release]; pathExecutor = nil;
    
    [currentPosition release]; currentPosition = nil;
    [targetPosition release]; targetPosition = nil;
    [speed release]; speed = nil;
    [defaultSpeed release]; defaultSpeed = nil;
    
    [super dealloc];
}

/** Initializes a robot instance.  Connection to the actual robot device is
 *  automatic after the CBRobot instance is created.
 */
- (id)init {
    self = [super init];
    if (self) {
        armLength1 = 82;    // (default arm 1 length, from pivot to pivot)
        armLength2 = 90;    // (default arm 2 length, from pivot to pivot)
        currentPosition = [[CBArmPosition zero] retain];
        targetPosition = [[CBArmPosition zero] retain];
        
        // Set up default motor parameters
        double degreesPerRadian = 180.0 / pi;
        motorParams = [[NSArray alloc] initWithObjects:
                       [CBMotorParameters motorParametersWithStepsPerRadian:(17.03 * degreesPerRadian)],
                       [CBMotorParameters motorParametersWithStepsPerRadian:(17.03 * degreesPerRadian)],
                       [CBMotorParameters motorParametersWithStepsPerRadian:(11.10 * degreesPerRadian)],
                       [CBMotorParameters motorParametersWithStepsPerRadian:(17.03 * degreesPerRadian)],
                       nil];

        // Set the default speed
        defaultSpeed = [[CBArmSpeed armSpeedWithM1Speed:(10.00 / degreesPerRadian)
                                             andM2Speed:(10.00 / degreesPerRadian)
                                             andM3Speed:(10.00 / degreesPerRadian)
                                             andM4Speed:(10.00 / degreesPerRadian)] retain];
        speed = [defaultSpeed retain];
        
        pathExecutor = [[CBPathExecutor alloc] initWithRobot:self];
        device = [[CBDevice alloc] init];
        [device setDelegate:self];
        [device startAutoConnect];
    }
    
    return self;
}

// (property documented in header)
- (BOOL)connected {
    return [device connected];
}

// (property documented in header)
- (void)setTargetPosition:(CBArmPosition *)position {
    if (position != targetPosition) {
        [targetPosition release];
        targetPosition = [position retain];
        [self updateFromDevice];
    }
}

// (property documented in header)
- (void)setSpeed:(CBArmSpeed *)aSpeed {
    if (aSpeed != speed) {
        [speed release];
        speed = [aSpeed retain];
        [self updateFromDevice];
    }
}

// (property documented in header)
- (void)setPaused:(BOOL)value {
    if (paused != value) {
        paused = value;
        [self updateFromDevice];
        [[NSNotificationCenter defaultCenter] postNotificationName:CB_NOTIFICATION_PAUSED_STATE_CHANGED object:self];
    }
}

/** Updates the CBRobot's currentPosition property from the device */
- (void)updateFromDevice {
    [device updateRobot:self];
}

/** Returns the set of parameters for the given motor (1 through 4). */
- (CBMotorParameters *)motorParametersForMotor:(int)motorNumber {
    if (motorNumber < 1 || motorNumber > [motorParams count]) {
        [NSException raise:@"CBBadArgument" format:@"Motor index is invalid: %i", motorNumber];
    }
    return (CBMotorParameters *)[motorParams objectAtIndex:(motorNumber - 1)];
}

/** Converts a step count into an angle (in radians) for the given motor (1
 *  through 4).  Does NOT round the step count to a whole number.
 */
- (double)angleFromSteps:(double)steps forMotor:(int)motorNumber {
    if (motorNumber < 1 || motorNumber > [motorParams count]) {
        [NSException raise:@"CBBadArgument" format:@"Motor index is invalid: %i", motorNumber];
    }
    return steps / [[self motorParametersForMotor:motorNumber] stepsPerRadian];
}

/** Converts an angle (in radians) into a step count for the given motor
 *  (1 through 4).  Does NOT round the resulting step count to a whole number.
 */
- (double)stepsFromAngle:(double)angle forMotor:(int)motorNumber {
    if (motorNumber < 1 || motorNumber > [motorParams count]) {
        [NSException raise:@"CBBadArgument" format:@"Motor index is invalid: %i", motorNumber];
    }
    return angle * [[self motorParametersForMotor:motorNumber] stepsPerRadian];
}

/** Called when the device's connection status changes. */
- (void)connectionStatusDidChange {
    [self setPaused:YES];
    [[NSNotificationCenter defaultCenter] postNotificationName:CB_NOTIFICATION_CONNECTION_STATUS_CHANGED object:self];
}

/** Returns YES if each motor of the robot is within the given number of steps
 *  of its target position (not rounded).
 */
- (BOOL)isAtTargetPositionWithThreshold:(double)steps {
    if (steps < 0) {
        [NSException raise:@"CBBadArgument" format:@"Threshold is invalid: %f", steps];
    }
    return [self isMotor:1 atTargetPositionWithThreshold:steps] &&
        [self isMotor:2 atTargetPositionWithThreshold:steps] &&
        [self isMotor:3 atTargetPositionWithThreshold:steps] &&
        [self isMotor:4 atTargetPositionWithThreshold:steps];
}

/** Returns YES if the given motor (1 - 4) is within the given number of steps
 *  of the target position (not rounded).
 */
- (BOOL)isMotor:(int)motor atTargetPositionWithThreshold:(double)steps {
    if (steps < 0) {
        [NSException raise:@"CBBadArgument" format:@"Threshold is invalid: %f", steps];
    }
    if (motor < 1 || motor > 4) {
        [NSException raise:@"CBBadArgument" format:@"Motor index is invalid: %i", motor];
    }
    if (motor == 4) {
        double currentSteps = [self stepsFromAngle:[[self currentPosition] m4] forMotor:4];
        double targetSteps = [self stepsFromAngle:[[self targetPosition] m4] forMotor:4];
        return ABS(targetSteps - currentSteps) < steps;
    }
    else {
        CBDofVector *currentPos = [[[self currentPosition] tipPosition] pointAsDofVectorForRobot:self];
        CBDofVector *targetPos = [[[self targetPosition] tipPosition] pointAsDofVectorForRobot:self];
        double currentSteps = [self stepsFromAngle:[currentPos componentWithIndex:(motor - 1)] forMotor:motor];
        double targetSteps = [self stepsFromAngle:[targetPos componentWithIndex:(motor - 1)] forMotor:motor];
        return ABS(targetSteps - currentSteps) < steps;
    }
}

/** Sets the current position as the new home position / zero position. */
- (void)setHomePosition {
    resetHomePositionFlag = YES;
}

// (property documented in header)
- (void)setPath:(id <CBPath>)aPath {
    if (path != aPath) {
        [path release];
        path = [aPath retain];

        // Forward the message to the path executor, which will handle the actual
        // running of the path.
        [pathExecutor setPath:aPath];
    }
}

@end
