/*=========================================================================
   This file is part of the Cardboard Robot SDK.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBArmPosition.h"
#import "CBArmSpeed.h"
#import "CBDofVector.h"
#import "CBPath.h"
#import "CBPathExecutor.h"
#import "CBRobot.h"

/* Interval at which to compute and send new position / speed values to the robot */
#define CB_PATH_UPDATE_INTERVAL         0.100

/* Threshold (in steps) used to determine if the robot is already at the target position */
#define CB_TARGET_POSITION_THRESHOLD    2.0

/* Delay (in seconds) after reaching the start position before starting the program */
#define CB_PROGRAM_START_DELAY          0.25

static double UpdateTimeValue(struct timeval *lastTime);
static double UpdatePositionValue(CBArmPosition **lastPosition, CBRobot *robot);

/* Private members */
@interface CBPathExecutor() {
    CBRobot *robot;         // Pointer to the robot (NOT retained; we are owned by the robot)
    NSTimer *updateTimer;   // Timer to update the robot's position / speed
    double pathTime;        // Time parameter for current path position
    struct timeval lastTime;        // Time at the last iteration of the update timer
    CBArmPosition *lastPosition;    // Arm position at last update
    BOOL waitingForStartPosition;   // Are we currently waiting to arrive at the start position?
    double startDelay;              // Remaining start delay
}

- (void)updateTimerElapsed:(NSTimer *)timer;

@end


@implementation CBPathExecutor

@synthesize path;

- (void)dealloc {
    [updateTimer release]; updateTimer = nil;
    [lastPosition release]; lastPosition = nil;
    
    [path release]; path = nil;
    
    [super dealloc];
}

- (id)init {
    [NSException raise:@"CBBadCall" format:@"Attempted to call init method without parameters"];
    return nil;
}

/** Initializes and returns a CBPathExecutor object */
- (id)initWithRobot:(CBRobot *)aRobot {
    self = [super init];
    if (self) {
        robot = aRobot;     // (do not retain; we are the child)
        lastPosition = [[CBArmPosition zero] retain];
    }
    return self;
}

// (property documented in header)
- (void)setPath:(id)aPath {
    if (aPath != path) {
        // Stop the timer if there is no path.  This allows this object to be dealloc'd.
        if (aPath == nil) {
            [updateTimer invalidate]; [updateTimer release]; updateTimer = nil;
        }
        
        [path release];
        path = [aPath retain];
        pathTime = 0.0;
        UpdateTimeValue(&lastTime);
        UpdatePositionValue(&lastPosition, robot);
        waitingForStartPosition = YES;
        
        // Start the timer if there is a path to run.
        if (aPath != nil && updateTimer == nil) {
            updateTimer = [[NSTimer scheduledTimerWithTimeInterval:CB_PATH_UPDATE_INTERVAL target:self selector:@selector(updateTimerElapsed:) userInfo:nil repeats:YES] retain];
        }
    }
}

/** Called when the update timer elapses */
- (void)updateTimerElapsed:(NSTimer *)timer {
    if (path != nil) {
        
        // Update the robot's currentPosition property
        [robot updateFromDevice];
        
        // Update the current time
        double realTimeDifference = UpdateTimeValue(&lastTime);
        double timeDifferenceByMovement = UpdatePositionValue(&lastPosition, robot);
        
        if (waitingForStartPosition) {
        
            // Tell the robot to go to the start of the path
            CBArmPosition *startPosition = [path startPosition];
            NSAssert(startPosition != nil, @"Path returned a nil start position");
            if (startPosition != nil) { [robot setTargetPosition:startPosition]; }
            // (user-specified speed value is used when moving to start position)
            
            if ([robot isAtTargetPositionWithThreshold:CB_TARGET_POSITION_THRESHOLD]) {
                waitingForStartPosition = NO;
                startDelay = CB_PROGRAM_START_DELAY;
            }
        }
        else {
        
            // Wait a short amount of time after reaching the start position
            if (startDelay > 0) {
                startDelay -= realTimeDifference;
            }
            else {
                // Increment by real time if the robot's motors are already at their
                // target position.  Otherwise, increment based on how far the arm has
                // moved (to minimize drift between the time parameter and the robot's
                // actual position)
                if ([robot isAtTargetPositionWithThreshold:CB_TARGET_POSITION_THRESHOLD]) {
                    pathTime += realTimeDifference;
                }
                else {
                    pathTime += timeDifferenceByMovement;
                }
                
                // (notify the path that the time parameter has changed)
                if ([path respondsToSelector:@selector(setPathTime:)]) {
                    [path setPathTime:pathTime];
                }
                
                // Ask the path for the target position / speed for the current time value
                CBArmPosition *pos = nil;
                CBArmSpeed *speed = nil;
                [path getTargetPosition:&pos andSpeed:&speed forTime:pathTime];
                NSAssert(pos != nil, @"Path returned a nil position");
                NSAssert(speed != nil, @"Path returned a nil speed");
                
                // Update the robot's target position / speed
                if (pos != nil) { [robot setTargetPosition:pos]; }
                if (speed != nil) { [robot setSpeed:speed]; }
                
                // Stop executing if the path is finished
                if (pathTime >= [path pathLength]) {
                    [robot setPath:nil];
                }
            }
        }
    }
}

/* Updates the given time value, then returns the number of seconds since the
 * previous value.
 */
static double UpdateTimeValue(struct timeval *lastTime) {
    NSCAssert(lastTime != NULL, @"lastTime was NULL");
    
    if (lastTime == NULL) {
        [NSException raise:@"CBBadArgument" format:@"A parameter was NULL"];
    }
    
    struct timeval t;
    gettimeofday(&t, NULL);
    double result = (t.tv_sec - lastTime->tv_sec) + (t.tv_usec - lastTime->tv_usec) / 1000000.0;
    *lastTime = t;
    return result;
}

/** Updates the given position value to the current position, and computes a
 *  time difference based on how far the arm has traveled in the direction of
 *  the target position (using DOF coordinates and DOF speed - may introduce a
 *  lot of error if applied to a path based on cartesian coordinates).
 */
static double UpdatePositionValue(CBArmPosition **lastPosition, CBRobot *robot) {
    NSCAssert(lastPosition != NULL, @"lastPosition was NULL");
    NSCAssert(*lastPosition != NULL, @"*lastPosition was NULL");
    NSCAssert(robot != NULL, @"robot was NULL");
    
    CBArmPosition *currentPos = [robot currentPosition];
    CBArmPosition *targetPos = [robot targetPosition];
    CBDofVector *currentTip = [[currentPos tipPosition] pointAsDofVectorForRobot:robot];
    CBDofVector *targetTip = [[targetPos tipPosition] pointAsDofVectorForRobot:robot];
    CBDofVector *lastTip = [[*lastPosition tipPosition] pointAsDofVectorForRobot:robot];
    
    // Compute a vector representing the robot's expected velocity from its last position
    double v1 = ([targetTip m1] >= [lastTip m1] ? 1.0 : -1.0) * [[robot speed] m1Speed];
    double v2 = ([targetTip m2] >= [lastTip m2] ? 1.0 : -1.0) * [[robot speed] m2Speed];
    double v3 = ([targetTip m3] >= [lastTip m3] ? 1.0 : -1.0) * [[robot speed] m3Speed];
    double v4 = ([targetPos m4] >= [*lastPosition m4] ? 1.0 : -1.0) * [[robot speed] m4Speed];
    
    // (zero out motors that aren't actually going to move)
    if ([robot isMotor:1 atTargetPositionWithThreshold:CB_TARGET_POSITION_THRESHOLD]) { v1 = 0; }
    if ([robot isMotor:2 atTargetPositionWithThreshold:CB_TARGET_POSITION_THRESHOLD]) { v2 = 0; }
    if ([robot isMotor:3 atTargetPositionWithThreshold:CB_TARGET_POSITION_THRESHOLD]) { v3 = 0; }
    if ([robot isMotor:4 atTargetPositionWithThreshold:CB_TARGET_POSITION_THRESHOLD]) { v4 = 0; }
    double vMagSq = v1 * v1 + v2 * v2 + v3 * v3 + v4 * v4;
    
    // Compute the dot product between the travel vector and the expected velocity vector
    double t1 = [currentTip m1] - [lastTip m1];
    double t2 = [currentTip m2] - [lastTip m2];
    double t3 = [currentTip m3] - [lastTip m3];
    double t4 = [currentPos m4] - [*lastPosition m4];
    double dotProduct = t1 * v1 + t2 * v2 + t3 * v3 + t4 * v4;
    
    // Divide the result by the length of the expected velocity vector SQUARED
    // to yield the distance traveled along that vector, RELATIVE to the length
    // of that vector
    double timeTraveled = vMagSq != 0.0 ? dotProduct / vMagSq : 0.0;    // (will be exactly zero if all motors at target position)
    
    [*lastPosition release];
    *lastPosition = [[robot currentPosition] retain];
    return timeTraveled;
}

@end
