/*=========================================================================
   This file is part of the Cardboard Robot SDK.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import <limits.h>
#import "CBArmPosition.h"
#import "CBArmSpeed.h"
#import "CBDeviceConnection.h"
#import "CBRobot.h"
#import "CBMotorParameters.h"
#import "CBDofVector.h"


static void CBFillOutStruct(CBRDataOut *outStruct,
                            CBRobot *robot);

static void CBFillOutBuffer(unsigned char *buffer,
                            unsigned int bufferLen,
                            CBRDataOut *outStruct,
                            BOOL resetHomePosition);

static BOOL CBParseInStruct(CBRDataIn *inStruct,
                            unsigned char *buffer,
                            unsigned int bufferLen);

static BOOL CBDoReadWrite(hid_device *deviceHandle,
                          CBRDataOut *outStruct,
                          CBRDataIn *inStruct,
                          BOOL resetHomePosition);

static void CBUpdateRobot(CBRobot *robot,
                          CBRDataIn *inStruct);


/* Private members */
@interface CBDeviceConnection()

- (void)readWriteThread;

@end


/* Non-public friends from CBRobot */
@interface CBRobot () <CBDeviceDelegate>

@property (assign, nonatomic) BOOL resetHomePositionFlag;

@end


@implementation CBDeviceConnection

- (void)dealloc {
    [self disconnect];
    
    NSAssert(deviceHandle == NULL, nil);
    [thread release]; thread = nil;
    [dataMutex release]; dataMutex = nil;
    [connectionMutex release]; connectionMutex = nil;
    
    [super dealloc];
}

/** Initializes a CBDeviceConnection */
- (id)init {
    self = [super init];
    if (self) {
        
        dataMutex = [[NSConditionLock alloc] initWithCondition:CB_NO_DATA];
        connectionMutex = [[NSConditionLock alloc] initWithCondition:CB_NOT_SENT];
        
        // Create and start the read / write thread
        thread = [[NSThread alloc] initWithTarget:self selector:@selector(readWriteThread) object:nil];
        [thread start];
    }
    return self;
}

/** Terminates the worker thread, allowing this object to be released. */
- (void)cancel {
    [thread cancel];
    [dataMutex lock];
    [dataMutex unlockWithCondition:CB_HAS_DATA];    // (unblock the thread)
}

// (property documented in header)
- (BOOL)connected {
    return deviceHandle != NULL;
}

/** Attempts to connect to the device, if not already connected.
 *  The connected property will be set to YES if the connection was
 *  successful.
 */
- (void)tryConnect {

    if (deviceHandle == NULL) {
        
        // Try to connect.  DO NOT DO THIS WHILE HOLDING THE LOCK, SINCE IT
        // USES A RUN LOOP AND COULD CAUSE US TO DEADLOCK AGAINST OURSELF.
        hid_device *handleToSave = hid_open(CBR_VENDOR_ID, CBR_PRODUCT_ID, NULL);

        if (handleToSave != NULL) {
            
            // Save the handle.
            [connectionMutex lock];
            deviceHandle = handleToSave;
            [connectionMutex unlock];
            
            // Tell the writer thread to write some initial data to the device,
            // and to initially set the home position.
            [dataMutex lock];
            resetHomePositionFlag = YES;
            [dataMutex unlockWithCondition:CB_HAS_DATA];
        }
    }
}

/** Disconnects from the device.  Does nothing if not connected. */
- (void)disconnect {
    hid_device *handleToClose = NULL;
    [connectionMutex lock];
    if (deviceHandle != nil) {
        handleToClose = deviceHandle;
        deviceHandle = nil;
    }
    [connectionMutex unlock];
    
    // Close the device handle.  DO NOT DO THIS WHILE HOLDING THE LOCK, SINCE
    // IT USES A RUN LOOP AND COULD CAUSE US TO DEADLOCK AGAINST OURSELF.
    if (handleToClose != NULL) {
        hid_close(handleToClose);
    }
}

/** Forces a re-send of the previous data, to test if we are still connected.
 *  Sets the "connected" property to NO if the send fails.
 */
- (void)testConnection {
    
    // Reset the connection status to "not sent".
    [connectionMutex lock];
    [connectionMutex unlockWithCondition:CB_NOT_SENT];

    // Set the "has data" flag to force the read / write thread to re-send the data.
    [dataMutex lock];
    [dataMutex unlockWithCondition:CB_HAS_DATA];
    
    // Wait for the write to finish (the connection status will change to "sent").
    [connectionMutex lockWhenCondition:CB_SENT];
    [connectionMutex unlock];
}

/** Updates the given robot with the latest data received from the USB device,
 *  and queues new data to send.
 */
- (void)updateRobot:(CBRobot *)robot {
    
    [dataMutex lock];   // (prevent concurrent access)
    
    // Update the robot with the last-read data.
    CBUpdateRobot(robot, &dataIn);
    
    // Update the output structure, to be written by the background thread.
    CBFillOutStruct(&dataOut, robot);
    
    // Read & reset the robot's flag to reset the home position.
    resetHomePositionFlag |= [robot resetHomePositionFlag];
    [robot setResetHomePositionFlag:NO];
    
    [dataMutex unlockWithCondition:CB_HAS_DATA];    // (signal write thread)
}

/** Main function for the read / write thread. */
- (void)readWriteThread {
    
    while (![thread isCancelled]) {
        
        @autoreleasepool {
            CBRDataOut localDataOut = { 0 };
            CBRDataIn localDataIn = { 0 };
            BOOL localResetHomePositionFlag;
            
            //// DATA MUTEX ///////////////////////////////////////////////////
            
            // Wait for data to arrive, then take a snapshot of it so it
            // doesn't change during the write.  Don't hold the data mutex
            // DURING the write, however!
            [dataMutex lockWhenCondition:CB_HAS_DATA];
            localDataOut = dataOut;
            localResetHomePositionFlag = resetHomePositionFlag;
            resetHomePositionFlag = NO;
            [dataMutex unlockWithCondition:CB_NO_DATA];

            //// CONNECTION MUTEX /////////////////////////////////////////////
            
            [connectionMutex lock];
            
            if (deviceHandle) {
                if (!CBDoReadWrite(deviceHandle, &localDataOut, &localDataIn, localResetHomePositionFlag)) {
                    hid_close(deviceHandle);
                    deviceHandle = NULL;    // (disconnected)
                }
            }
            
            [connectionMutex unlockWithCondition:CB_SENT];

            //// DATA MUTEX ///////////////////////////////////////////////////
            
            // Save the data read from the device.  
            [dataMutex lock];
            dataIn = localDataIn;
            
            // If we're still connected and not at the target location, signal
            // that there is still more data to be read!
            if (deviceHandle != NULL &&
                (localDataIn.m1Current != localDataOut.m1Target ||
                localDataIn.m2Current != localDataOut.m2Target ||
                localDataIn.m3Current != localDataOut.m3Target ||
                localDataIn.m4Current != localDataOut.m4Target)) {
                
                [dataMutex unlockWithCondition:CB_HAS_DATA];
            }
            else {
                [dataMutex unlock];
            }
        }
        
        // Wait a minimum amount of time before doing another read / write.
        [NSThread sleepForTimeInterval:0.005];
    }
}

@end

/** Fills an output structure with the parameters of the given robot */
static void CBFillOutStruct(CBRDataOut *outStruct,
                            CBRobot *robot) {
    
    NSCAssert(outStruct != NULL && robot != nil, @"Argument was NULL");

    memset(outStruct, 0, sizeof(*outStruct));
    
    CBDofVector *targetTipPos = [[[robot targetPosition] tipPosition] pointAsDofVectorForRobot:robot];
    double targetM4Pos = [[robot targetPosition] m4];
    
    if (targetTipPos != nil) {
        outStruct->m1Target = (short)MAX(SHRT_MIN, MIN(SHRT_MAX, round([robot stepsFromAngle:[targetTipPos m1] forMotor:1])));
        outStruct->m2Target = (short)MAX(SHRT_MIN, MIN(SHRT_MAX, round([robot stepsFromAngle:[targetTipPos m2] forMotor:2])));
        outStruct->m3Target = (short)MAX(SHRT_MIN, MIN(SHRT_MAX, round([robot stepsFromAngle:[targetTipPos m3] forMotor:3])));
    }
    
    outStruct->m4Target = (short)MAX(SHRT_MIN, MIN(SHRT_MAX, round([robot stepsFromAngle:targetM4Pos forMotor:4])));
    
    double stepsPerRads1 = [[robot motorParametersForMotor:1] stepsPerRadian];
    double stepsPerRads2 = [[robot motorParametersForMotor:2] stepsPerRadian];
    double stepsPerRads3 = [[robot motorParametersForMotor:3] stepsPerRadian];
    double stepsPerRads4 = [[robot motorParametersForMotor:4] stepsPerRadian];

    // Speed[steps/s] = 60000 / Period
    // => Period = 60000 / Speed[steps/s]
    // => Period = 60000 / (Speed[rads/s] * StepsPerRad[steps/rads])
    double period1 = 60000.0 / ([[robot speed] m1Speed] * stepsPerRads1);
    double period2 = 60000.0 / ([[robot speed] m2Speed] * stepsPerRads2);
    double period3 = 60000.0 / ([[robot speed] m3Speed] * stepsPerRads3);
    double period4 = 60000.0 / ([[robot speed] m4Speed] * stepsPerRads4);
    
    outStruct->m1Speed = MAX(0, (short)round(period1));
    outStruct->m2Speed = MAX(0, (short)round(period2));
    outStruct->m3Speed = MAX(0, (short)round(period3));
    outStruct->m4Speed = MAX(0, (short)round(period4));
    
    outStruct->paused = [robot paused];
}

/** Fills an output buffer with the contents of the given structure */
static void CBFillOutBuffer(unsigned char *buffer,
                            unsigned int bufferLen,
                            CBRDataOut *outStruct,
                            BOOL resetHomePosition) {
    
    NSCAssert(buffer != NULL && outStruct != NULL, @"Argument was NULL");
    if (bufferLen != 65) {
        [NSException raise:@"CBBadArgument"
                    format:@"Invalid structure length"];
    }
    
    // Initialize all unused bytes to 0xFF to lower EMI and power consumption
    memset(buffer, 0xFF, bufferLen);
    
    buffer[0] = 0x00;   // (report ID, always zero)
    buffer[1] = 0x83;   // (0x83 = "send current position")
    
    buffer[2] = (char)(outStruct->m1Target & 0xFF);
    buffer[3] = (char)(outStruct->m1Target >> 8);
    buffer[4] = (char)(outStruct->m2Target & 0xFF);
    buffer[5] = (char)(outStruct->m2Target >> 8);
    buffer[6] = (char)(outStruct->m3Target & 0xFF);
    buffer[7] = (char)(outStruct->m3Target >> 8);
    buffer[8] = (char)(outStruct->m4Target & 0xFF);
    buffer[9] = (char)(outStruct->m4Target >> 8);
    
    buffer[10] = outStruct->paused ? 0x00 : 0xFF;   // byte 10: emergency power shut-off
    buffer[11] = resetHomePosition ? 0xFF : 0x00;   // byte 11: set home position
    
    buffer[12] = (char)(outStruct->m1Speed & 0xFF);
    buffer[13] = (char)(outStruct->m1Speed >> 8);
    buffer[14] = (char)(outStruct->m2Speed & 0xFF);
    buffer[15] = (char)(outStruct->m2Speed >> 8);
    buffer[16] = (char)(outStruct->m3Speed & 0xFF);
    buffer[17] = (char)(outStruct->m3Speed >> 8);
    buffer[18] = (char)(outStruct->m4Speed & 0xFF);
    buffer[19] = (char)(outStruct->m4Speed >> 8);
    
    buffer[20] = outStruct->paused ? 0x00 : 0xFF;   // byte 20: pause function (counter only)
}

/** Parses an input structure from the contents of the given input buffer.
 *  Returns YES if successful, NO if the contents of the buffer were invalid.
 */
static BOOL CBParseInStruct(CBRDataIn *inStruct,
                            unsigned char *buffer,
                            unsigned int bufferLen) {
    
    NSCAssert(inStruct != NULL && buffer != NULL, @"Argument was NULL");
    if (bufferLen != 64) {
        [NSException raise:@"CBBadArgument"
                    format:@"Invalid structure length"];
    }
    
    // Initialize the structure to zero, in case we miss anything.
    memset(inStruct, 0, sizeof(CBRDataIn));
    
    // Do some basic validation
    if (buffer[0] != 0x83) {    // (0x83 = echo of "send current position")
        return NO;
    }
    
    inStruct->m1Current = ((short)buffer[2] << 8) | buffer[1];
    inStruct->m2Current = ((short)buffer[4] << 8) | buffer[3];
    inStruct->m3Current = ((short)buffer[6] << 8) | buffer[5];
    inStruct->m4Current = ((short)buffer[8] << 8) | buffer[7];
    
    return YES;
}

/** Performs the actual read / write.  Returns YES if the write was
 *  successful, or NO if it was not.
 */
static BOOL CBDoReadWrite(hid_device *deviceHandle,
                          CBRDataOut *outStruct,
                          CBRDataIn *inStruct,
                          BOOL resetHomePosition) {
    
    NSCAssert(deviceHandle != NULL &&
              outStruct != NULL &&
              inStruct != NULL, @"Argument was NULL");
    
    // Initialize the output in case we fail.
    memset(inStruct, 0, sizeof(*inStruct));
    
    // (NOTE: input buffers do not contain the report number, and are therefore
    //  only 64 bytes instead of 65)
    unsigned char outBuffer[65] = { 0 };
    unsigned char inBuffer[64] = { 0 };
    
    // Fill the output buffer with data.
    CBFillOutBuffer(outBuffer, sizeof(outBuffer), outStruct, resetHomePosition);
    
    // Write the output buffer.
    int bytesWritten = hid_write(deviceHandle, outBuffer, sizeof(outBuffer));
    if (bytesWritten == sizeof(outBuffer)) {
        
        // Read the response.
        int bytesRead = hid_read(deviceHandle, inBuffer, sizeof(inBuffer));
        if (bytesRead == sizeof(inBuffer)) {
            
            // Fill the input structure from the input buffer.
            if (CBParseInStruct(inStruct, inBuffer, sizeof(inBuffer))) {
                return YES;
            }
            else {
                NSCAssert(NO, @"Input buffer was invalid");
                return NO;
            }
        }
        else {
            return NO;
        }
    }
    else {
        return NO;
    }
}

/** Updates a robot with the values of the given structure */
static void CBUpdateRobot(CBRobot *robot,
                          CBRDataIn *inStruct) {
    
    NSCAssert(robot != nil && inStruct != NULL, @"Argument was NULL");
    
    CBDofVector *tipPosition = [CBDofVector vectorWithM1:[robot angleFromSteps:inStruct->m1Current forMotor:1]
                                                   andM2:[robot angleFromSteps:inStruct->m2Current forMotor:2]
                                                   andM3:[robot angleFromSteps:inStruct->m3Current forMotor:3]];

    double m4 = [robot angleFromSteps:inStruct->m4Current forMotor:4];
    
    CBArmPosition *newPosition = [CBArmPosition armPositionWithTipPosition:tipPosition
                                                                     andM4:m4];

    [robot setCurrentPosition:newPosition];
}
