/*=========================================================================
   This file is part of the Cardboard Robot SDK.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import <IOKit/usb/IOUSBLib.h>  /* for kIOUSBDeviceClassName, etc. */
#import "CBDevice.h"
#import "CBDeviceConnection.h"
#import "CBDeviceDelegate.h"

#define CB_AUTOCONNECT_INTERVAL 5.0     /* Number of seconds between connect / disconnect attempts */

static CFMutableDictionaryRef CBCreateSearchParameters();
static void CBOnDeviceAdded(void *refcon, io_iterator_t iterator);
static void CBOnDeviceRemoved(void *refcon, io_iterator_t iterator);


/* Private members */
@interface CBDevice()

- (void)testConnection;
- (void)setConnected:(BOOL)value;
- (void)autoConnectTimerCallback:(NSTimer *)timer;
- (BOOL)registerForDeviceNotifications;

@end


@implementation CBDevice

@synthesize connected;
@synthesize delegate;

- (void)dealloc {
    
    [self stopAutoConnect];     // (just in case; shouldn't have gotten here without it)
    NSAssert(notificationPort == NULL, nil);
    NSAssert(deviceAddedIterator == 0, nil);
    NSAssert(deviceRemovedIterator == 0, nil);
    NSAssert(autoConnectTimer == nil, nil);
    
    [connection cancel]; [connection release]; connection = nil;
    
    [super dealloc];
}

- (id)init {
    self = [super init];
    if (self) {
        connection = [[CBDeviceConnection alloc] init];
    }
    return self;
}

/** Begins auto-connecting to the device. */
- (void)startAutoConnect {
    if (autoConnectTimer) { return; }   // (already set up)
    
    // Register for add / remove notification.
    [self registerForDeviceNotifications];
    
    // Start the auto-connect timer.
    autoConnectTimer = [[NSTimer scheduledTimerWithTimeInterval:CB_AUTOCONNECT_INTERVAL target:self selector:@selector(autoConnectTimerCallback:) userInfo:nil repeats:YES] retain];
    
    // Try connecting now, since we won't receive notification if the device is already connected.
    [self tryConnect];
}

/** Stops auto-connecting to the device.  This must be called before the
 *  CBDevice object is released, or it won't dealloc.
 */
- (void)stopAutoConnect {
    
    // Release the timer, allowing this object to be dealloc'd
    [autoConnectTimer invalidate]; [autoConnectTimer release]; autoConnectTimer = nil;
    
    // Release the notification port and associated objects (some are released along with the port)
    if (deviceAddedIterator) { IOObjectRelease(deviceAddedIterator); deviceAddedIterator = 0; }
    if (deviceRemovedIterator) { IOObjectRelease(deviceRemovedIterator); deviceRemovedIterator = 0; }
    if (notificationPort) { IONotificationPortDestroy(notificationPort); notificationPort = NULL; }
}

// (property documented in header)
- (BOOL)connected {
    return connected;
}

/** Disconnects from the device immediately (doesn't stop auto-reconnection). */
- (void)disconnect {
    if ([connection connected]) {
        [connection disconnect];
    }
    [self setConnected:[connection connected]];
}

/** Attempts to connect to the device, if not already connected. */
- (void)tryConnect {
    if (![connection connected]) {
        [connection tryConnect];
    }
    [self setConnected:[connection connected]];
}

/** Attempts to re-send data to the device to make sure it's still connected.
 *  Disconnects from the device if an error occurs. */
- (void)testConnection {
    if ([connection connected]) {
        [connection testConnection];
    }
    [self setConnected:[connection connected]];
}

/** Sets the connected status (internal) */
- (void)setConnected:(BOOL)value {
    if (connected != value) {
        connected = value;
        [delegate connectionStatusDidChange];
    }
}

/** @copydoc CBDeviceConnection#updateRobot: */
- (void)updateRobot:(CBRobot *)robot {
    [connection updateRobot:robot];
    [self setConnected:[connection connected]];     // (in case it changed)
}

/** Called when the auto-connect timer elapses. */
- (void)autoConnectTimerCallback:(NSTimer *)timer {
    
    // Test for connection or disconnection.
    if (![connection connected]) {
        [self tryConnect];
    }
    else {
        [self testConnection];
    }
}

/** Registers for device connect / disconnect notifications. */
- (BOOL)registerForDeviceNotifications {
    
    // Don't initialize more than once
    if (notificationPort != NULL) {
        return NO;
    }
    
    // Create a master port
    mach_port_t masterPort;
    if (IOMasterPort(MACH_PORT_NULL, &masterPort) != KERN_SUCCESS) {
        return NO;
    }
    
    // Create a notification port and add it as a run loop source
    notificationPort = IONotificationPortCreate(masterPort);
    if (notificationPort == NULL) {
        return NO;
    }
    CFRunLoopAddSource(CFRunLoopGetCurrent(), IONotificationPortGetRunLoopSource(notificationPort), kCFRunLoopDefaultMode);
    
    // Create a set of search parameters for the "device added" enumerator
    CFMutableDictionaryRef parameters1 = CBCreateSearchParameters();
    if (parameters1 == NULL) {
        return NO;
    }
    
    // Install the "device added" notification request
    if (IOServiceAddMatchingNotification(notificationPort, kIOFirstMatchNotification, parameters1, CBOnDeviceAdded, self, &deviceAddedIterator) != KERN_SUCCESS) {
        CFRelease(parameters1);
        return NO;
    }
    // (NOTE: reference to dictionary is consumed)
    
    // Create a set of search parameters for the "device removed" enumerator
    // (NOTE: we cannot reuse the previous dictionary!)
    CFMutableDictionaryRef parameters2 = CBCreateSearchParameters();
    if (parameters2 == NULL) {
        return NO;
    }
    
    // Install the "device removed" notification request
    if (IOServiceAddMatchingNotification(notificationPort, kIOTerminatedNotification, parameters2, CBOnDeviceRemoved, self, &deviceRemovedIterator) != KERN_SUCCESS) {
        CFRelease(parameters2);
        return NO;
    }
    // (NOTE: reference to dictionary is consumed)
    
    // Arm the notifications by iterating through any existing device entries
    io_object_t object;
    while ((object = IOIteratorNext(deviceAddedIterator)) != 0) { IOObjectRelease(object); }
    while ((object = IOIteratorNext(deviceRemovedIterator)) != 0) { IOObjectRelease(object); }
    
    // Notification installed successfully
    return YES;
}

@end

/** Returns a dictionary containing the parameters used to locate the device in
 *  the IO registry.
 */
CFMutableDictionaryRef CBCreateSearchParameters() {
    
    // Set up parameters to identify our device in the IO registry
    CFMutableDictionaryRef matchParams = IOServiceMatching(kIOUSBDeviceClassName);
    if (matchParams == NULL) { return NULL; }
    
    // (vendor ID)
    CFNumberRef numberRef;
    SInt32 vendorID = CBR_VENDOR_ID;
    numberRef = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &vendorID);
    CFDictionaryAddValue(matchParams, CFSTR(kUSBVendorID), numberRef);
    CFRelease(numberRef);
    
    // (product ID)
    SInt32 productID = CBR_PRODUCT_ID;
    numberRef = CFNumberCreate(kCFAllocatorDefault, kCFNumberSInt32Type, &productID);
    CFDictionaryAddValue(matchParams, CFSTR(kUSBProductID), numberRef);
    CFRelease(numberRef);
    
    return matchParams;
}

/** Called when a robot device is plugged in. */
void CBOnDeviceAdded(void *refcon, io_iterator_t iterator) {
    
    // Try to connect to the device
    CBDevice *device = (CBDevice *)refcon;
    [device tryConnect];
    
    // We won't receive another notification unless we finish iterating
    io_object_t object;
    while ((object = IOIteratorNext(iterator)) != 0) { IOObjectRelease(object); }
}

/** Called when a robot device is removed. */
void CBOnDeviceRemoved(void *refcon, io_iterator_t iterator) {
    
    // Re-send data to the device, and disconnect if it fails.
    CBDevice *device = (CBDevice *)refcon;
    [device testConnection];
    
    // We won't receive another notification unless we finish iterating
    io_object_t object;
    while ((object = IOIteratorNext(iterator)) != 0) { IOObjectRelease(object); }
}
